<?php
/**
 * Meta Keys REST API endpoint for searching post meta keys.
 *
 * @package ALMFilters
 */

/**
 * Register the REST API endpoint.
 */
add_action(
	'rest_api_init',
	function () {
		register_rest_route(
			'alm-filters',
			'/meta-keys',
			[
				'methods'             => 'GET',
				'callback'            => 'alm_filters_get_meta_keys',
				'permission_callback' => function () {
					return current_user_can( 'edit_posts' );
				},
				'args'                => [
					'search'     => [
						'type'              => 'string',
						'sanitize_callback' => 'sanitize_text_field',
						'default'           => '',
					],
					'limit'      => [
						'type'              => 'integer',
						'sanitize_callback' => 'absint',
						'default'           => 25,
					],
					'offset'     => [
						'type'              => 'integer',
						'sanitize_callback' => 'absint',
						'default'           => 0,
					],
					'post_types' => [
						'type'              => 'string',
						'sanitize_callback' => 'sanitize_text_field',
						'default'           => '',
					],
				],
			]
		);
	}
);

/**
 * Get meta keys from the database.
 *
 * @param WP_REST_Request $request The HTTP request object.
 * @return WP_REST_Response
 */
function alm_filters_get_meta_keys( WP_REST_Request $request ) {
	global $wpdb;

	$search     = $request->get_param( 'search' );
	$limit      = min( $request->get_param( 'limit' ), 100 ); // Max 100 results.
	$offset     = $request->get_param( 'offset' );
	$post_types = $request->get_param( 'post_types' );

	// Parse post types from comma-separated string.
	$post_type_array = [];
	if ( ! empty( $post_types ) ) {
		$post_type_array = array_filter( array_map( 'trim', explode( ',', $post_types ) ) );
	}

	/**
	 * Include private meta keys (those starting with underscore) in the autocomplete.
	 * By default, private keys are excluded for a cleaner list.
	 *
	 * @param bool $include_private Whether to include private meta keys. Default false.
	 */
	$include_private = apply_filters( 'alm_filters_meta_include_private_keys', false );

	// Build the query.
	$where_clauses = [];

	// Exclude private keys unless filter says otherwise.
	if ( ! $include_private ) {
		$where_clauses[] = $wpdb->prepare( 'pm.meta_key NOT LIKE %s', $wpdb->esc_like( '_' ) . '%' );
	}

	// Add search filter if provided.
	if ( ! empty( $search ) ) {
		$where_clauses[] = $wpdb->prepare( 'pm.meta_key LIKE %s', '%' . $wpdb->esc_like( $search ) . '%' );
	}

	// Filter by post types if provided.
	if ( ! empty( $post_type_array ) ) {
		$placeholders    = implode( ', ', array_fill( 0, count( $post_type_array ), '%s' ) );
		$where_clauses[] = $wpdb->prepare(
			"p.post_type IN ($placeholders)", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			$post_type_array
		);
	}

	// Combine WHERE clauses.
	$where_sql = ! empty( $where_clauses ) ? 'WHERE ' . implode( ' AND ', $where_clauses ) : '';

	// Use JOIN with posts table when filtering by post types.
	if ( ! empty( $post_type_array ) ) {
		// Get total count first.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$total = $wpdb->get_var(
			"SELECT COUNT(DISTINCT pm.meta_key)
			FROM {$wpdb->postmeta} pm
			INNER JOIN {$wpdb->posts} p ON pm.post_id = p.ID
			{$where_sql}"
		);

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$meta_keys = $wpdb->get_col(
			$wpdb->prepare(
				"SELECT DISTINCT pm.meta_key
				FROM {$wpdb->postmeta} pm
				INNER JOIN {$wpdb->posts} p ON pm.post_id = p.ID
				{$where_sql}
				ORDER BY pm.meta_key ASC
				LIMIT %d OFFSET %d",
				$limit,
				$offset
			)
		);
	} else {
		// Get total count first.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$total = $wpdb->get_var(
			"SELECT COUNT(DISTINCT pm.meta_key)
			FROM {$wpdb->postmeta} pm
			{$where_sql}"
		);

		// Simple query without post type filtering.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$meta_keys = $wpdb->get_col(
			$wpdb->prepare(
				"SELECT DISTINCT pm.meta_key
				FROM {$wpdb->postmeta} pm
				{$where_sql}
				ORDER BY pm.meta_key ASC
				LIMIT %d OFFSET %d",
				$limit,
				$offset
			)
		);
	}

	/**
	 * Filter the list of meta keys returned by the API.
	 *
	 * @param array  $meta_keys  Array of meta key names.
	 * @param string $search     The search term used.
	 * @param array  $post_types Array of post types to filter by.
	 */
	$meta_keys = apply_filters( 'alm_filters_meta_keys', $meta_keys, $search, $post_type_array );

	$total    = (int) $total;
	$has_more = ( $offset + count( $meta_keys ) ) < $total;

	return rest_ensure_response(
		[
			'success'   => true,
			'meta_keys' => array_values( $meta_keys ),
			'count'     => count( $meta_keys ),
			'total'     => $total,
			'has_more'  => $has_more,
		]
	);
}
