<?php
/**
 * Functions related to the textfield field type.
 * Used with DFate Picker, Range Slider, Number and Textfield filters.
 *
 * @package ALMFilters
 */

/**
 * Renders a filter textfield.
 *
 * @param string $id          Filter ID.
 * @param array  $obj         Filter object array.
 * @param string $querystring The querystring.
 * @return string             Raw HTML output.
 * @since 1.0
 */
function alm_filters_display_textfield( $id, $obj, $querystring ) {
	$key     = $obj['key'];
	$text_id = $key . '-' . $obj['field_type'];
	$output  = '';

	// Parse Querystring params.
	if ( $key === 'meta' ) {
		$selected = isset( $querystring[ $obj['meta_key'] ] ) ? $querystring[ $obj['meta_key'] ] : '';
	} elseif ( $key === 'taxonomy' ) {
		$selected = isset( $querystring[ $obj['taxonomy'] ] ) ? $querystring[ $obj['taxonomy'] ] : '';
	} else {
		$selected = isset( $querystring[ $key ] ) ? $querystring[ $key ] : '';
	}

	$placeholder   = isset( $obj['placeholder'] ) ? 'placeholder="' . $obj['placeholder'] . '"' : '';
	$default_value = isset( $obj['default_value'] ) ? $obj['default_value'] : '';
	if ( $selected === '' && $default_value !== '' ) {
		$selected = $default_value;
	}

	// Determine field type.
	$numeric     = $obj['field_type'] === 'number' ? true : false;
	$datepicker  = $obj['field_type'] === 'date_picker' ? true : false;
	$rangeslider = $obj['field_type'] === 'range_slider' ? true : false;

	// Sanitize selected value.
	$selected = str_replace( '<', '&lt;', $selected );
	$selected = str_replace( '%3C', '&lt;', $selected );
	$selected = str_replace( '>', '&gt;', $selected );
	$selected = str_replace( '&gt;', '', $selected );

	// Set submit button label.
	$button_label_hook = has_filter( 'alm_filters_textfield_submit_label' ) ? apply_filters( 'alm_filters_textfield_submit_label', __( 'Submit', 'ajax-load-more-filters' ) ) : false;
	$button_label      = isset( $obj['button_label'] ) ? trim( $obj['button_label'] ) : '';
	if ( ! $button_label && $button_label_hook ) {
		$button_label = $button_label_hook; // Use hook label if no custom label set.
	}
	if ( $button_label === 'hidden' ) {
		$button_label = ''; // Do not display button.
	}

	// Add class if button present.
	$field_class = $button_label ? ' has-button' : '';

	$display_style = '';

	$output .= '<div class="alm-filter--' . $obj['field_type'] . '">';

	if ( $obj['label'] ) {
		$output .= alm_filters_render_label( $id, $obj, $text_id . '-' . $obj['index'] );
	}

	if ( $rangeslider ) {
		// Range Slider opts.
		$range_min = isset( $obj['rangeslider_min'] ) ? $obj['rangeslider_min'] : 0;
		$range_max = isset( $obj['rangeslider_max'] ) ? $obj['rangeslider_max'] : 100;

		$range_start      = isset( $obj['rangeslider_start'] ) ? $obj['rangeslider_start'] : $range_min;
		$range_start_orig = $range_start === '' ? $range_min : $range_start;
		$range_end        = isset( $obj['rangeslider_end'] ) ? $obj['rangeslider_end'] : $range_max;
		$range_end_orig   = $range_end === '' ? $range_max : $range_end;

		$rangeslider_label       = isset( $obj['rangeslider_label'] ) ? $obj['rangeslider_label'] : '{start} - {end}';
		$rangeslider_steps       = isset( $obj['rangeslider_steps'] ) ? $obj['rangeslider_steps'] : 1;
		$rangeslider_orientation = isset( $obj['rangeslider_orientation'] ) ? $obj['rangeslider_orientation'] : 'horizontal';
		$rangeslider_decimals    = isset( $obj['rangeslider_decimals'] ) ? $obj['rangeslider_decimals'] : 'true';
		$rangeslider_reset       = isset( $obj['rangeslider_reset'] ) ? $obj['rangeslider_reset'] : 'true';

		// Parse selected value.
		$values = ! empty( $selected ) ? explode( ',', $selected ) : '';
		if ( ! empty( $values ) ) {
			$range_start = $values[0];
			$range_end   = isset( $values[1] ) ? $values[1] : $range_max;
		}

		$output .= '<div class="alm-range-slider"
						data-min="' . $range_min . '"
						data-max="' . $range_max . '"
						data-start-reset="' . $range_start_orig . '"
						data-start="' . $range_start . '"
						data-end-reset="' . $range_end_orig . '"
						data-end="' . $range_end . '"
						data-label="' . $rangeslider_label . '"
						data-steps="' . $rangeslider_steps . '"
						data-orientation="' . $rangeslider_orientation . '"
						data-decimals="' . $rangeslider_decimals . '"
						>';
		$output .= '<div class="alm-range-slider--target"></div>';
		$output .= '<div class="alm-range-slider--wrap">';
		$output .= '<div class="alm-range-slider--label"></div>';

		if ( $rangeslider_reset !== 'false' ) {
			// Reset Button.
			$output .= '<button class="alm-range-slider--reset alm-range-reset" type="button" style="visibility: hidden;">';
			$output .= apply_filters( 'alm_filters_range_slider_reset_label', __( 'Reset', 'ajax-load-more-filters' ) );
			$output .= '</button>';
		}

		$output       .= '</div>';
		$output       .= '</div>';
		$display_style = ' style="display: none;"';
	}

	$output .= '<div class="alm-filter--text-wrap' . $field_class . '"' . $display_style . '>';

	if ( $datepicker ) {
		// Date Picker.
		$dp_mode        = isset( $obj['datepicker_mode'] ) ? $obj['datepicker_mode'] : 'single';
		$dp_mode_return = isset( $obj['datepicker_mode'] ) ? ' data-display-mode="' . $dp_mode . '"' : ' data-display-mode="single"';
		$dp_format      = isset( $obj['datepicker_format'] ) ? ' data-display-format="' . $obj['datepicker_format'] . '"' : ' data-display-format="Y-m-d"';
		$dp_locale      = isset( $obj['datepicker_locale'] ) ? ' data-date-locale="' . $obj['datepicker_locale'] . '"' : ' data-date-locale="en"';
		$selected       = $dp_mode === 'range' ? str_replace( '+', ' | ', $selected ) : $selected; // Replace `+` with ` | ` for range mode.

		$output .= '<input class="alm-filter--textfield textfield alm-flatpickr" id="' . $text_id . '-' . $obj['index'] . '" name="' . $text_id . '" type="text" value="' . esc_attr( $selected ) . '"' . $placeholder . '' . $dp_format . '' . $dp_mode_return . '' . $dp_locale . ' />';

	} elseif ( $numeric ) {
		// Numeric.
		$numeric_min   = isset( $obj['number_min'] ) ? $obj['number_min'] : false;
		$numeric_max   = isset( $obj['number_max'] ) ? $obj['number_max'] : false;
		$numeric_steps = isset( $obj['number_steps'] ) ? $obj['number_steps'] : false;

		$output .= '<input class="alm-filter--textfield alm-filter--numberfield"';
		$output .= ' type="number"';
		$output .= ' id="' . $text_id . '-' . $obj['index'] . '"';
		$output .= ' name="' . $text_id . '"';
		if ( $numeric_min !== false ) {
			$output .= ' min="' . esc_attr( $numeric_min ) . '"';
		}
		if ( $numeric_max !== false ) {
			$output .= ' max="' . esc_attr( $numeric_max ) . '"';
		}
		if ( $numeric_steps !== false && $numeric_steps > 0 ) {
			$output .= ' step="' . esc_attr( $numeric_steps ) . '"';
		}
		$output .= ' value="' . esc_attr( $selected ) . '"';
		$output .= $placeholder . ' />';

	} else {
		// Textfield || Date Picker.
		$output .= '<input class="alm-filter--textfield textfield" type="text"';
		$output .= ' id="' . $text_id . '-' . $obj['index'] . '"';
		$output .= ' name="' . $text_id . '"';
		$output .= ' value="' . esc_attr( urldecode( $selected ) ) . '"';
		$output .= $placeholder . ' />';
	}

	$output .= $button_label ? '<button type="button">' . $button_label . '</button>' : '';

	$output .= '</div>';
	$output .= '</div>';

	return $output;
}
