<?php
/**
 * A file full of plugin helpers.
 *
 * @package ALMFilters
 */

/**
 * Render Reset/Clear Button.
 *
 * @param object $options The filter config.
 * @param object $obj The filter object.
 * @return string $output
 * @since 1.11.0
 */
function alm_filters_render_controls( $options, $obj ) {
	$output     = '';
	$reset_btn  = '';
	$submit_btn = '';

	// Reset Button.
	if ( $options['reset_button'] && ! empty( $options['reset_button_label'] ) ) {
		$classname = apply_filters( 'alm_filters_reset_button_class', 'alm-filters--reset-button' );

		$reset_btn .= '<div class="alm-filters--reset">';
		$reset_btn .= '<button type="reset" id="alm-filters-reset-button" class="' . $classname . '" style="display: none;"><span>' . $options['reset_button_label'] . '</span></button>';
		$reset_btn .= '</div>';
	}

	// Submit Button.
	$hide_submit = ( $obj['count'] === '1' && ! empty( $obj['button_label'] ) && 'text' === $obj['field_type'] ) ? true : false; // Hide Submit button if count is 1 and field type is textfield.
	if ( 'button' === $options['style'] && ! $hide_submit ) {
		$submit_btn .= '<div class="alm-filters--submit">';
		$submit_btn .= '<button type="button" class="alm-filters--button"><span>' . $options['button_text'] . '</span></button>';
		$submit_btn .= '</div>';
	}

	// Build output.
	if ( ! empty( $reset_btn ) || ! empty( $submit_btn ) ) {
		$output .= '<div class="alm-filters--controls">';
		$output .= $reset_btn;
		$output .= $submit_btn;
		$output .= '</div>';
	}

	return $output;
}

/**
 * Render filter label.
 *
 * @param string $id     The filter ID.
 * @param array  $obj    The filter array/object.
 * @param string $target The target HTML element.
 * @since 1.8.4
 */
function alm_filters_render_label( $id = '', $obj = '', $target = '' ) {
	if ( empty( $id ) || empty( $obj ) || empty( $target ) || ! isset( $obj['label'] ) ) {
		return;
	}

	$filter_key = alm_filters_get_filter_key( $obj );
	$hook_name  = $id . '_' . $filter_key;
	$value      = $obj['label'];

	if ( empty( $value ) && ! has_filter( 'alm_filters_' . $hook_name . '_label' ) ) {
		return; // Bail early if title is empty && filter doesn't exist.
	}

	$output = '<label for="' . $target . '">' . apply_filters( 'alm_filters_' . $hook_name . '_label', $value ) . '</label>';
	return $output;
}

/**
 * Render filter title.
 *
 * @since 1.0
 * @param string  $id     The filter ID.
 * @param array   $obj    The filter array/object.
 * @param boolean $toggle Toggle or not.
 * @param boolean $status The status of the toggle.
 * @return string         Raw HTML output.
 */
function alm_filters_display_title( $id = '', $obj = '', $toggle = false, $status = 'expanded' ) {
	if ( empty( $id ) || empty( $obj ) || ! isset( $obj['title'] ) ) {
		return false; // Bail early if empty.
	}

	$filter_key = alm_filters_get_filter_key( $obj );
	$hook_name  = $id . '_' . $filter_key;
	$value      = $obj['title'];

	if ( empty( $value ) && ! has_filter( 'alm_filters_' . $hook_name . '_title' ) ) {
		return false; // Bail early if title is empty && filter doesn't exist.
	}

	$aria_expanded = ( 'expanded' === $status ) ? 'true' : 'false';

	$toggle_opts = '';
	if ( $toggle ) {
		$toggle_opts = ' class="alm-filter--toggle" tabindex="0" aria-expanded="' . $aria_expanded . '" aria-controls="alm-filter-' . $filter_key . '-inner" role="button"';
	}

	$output  = '<div class="alm-filter--title">';
	$output .= '<' . apply_filters( 'alm_filters_title_element', 'h3' ) . ' id="alm-filter-' . $filter_key . '-title" ' . $toggle_opts . '>';
	$output .= apply_filters( 'alm_filters_' . $hook_name . '_title', $value );
	$output .= '</' . apply_filters( 'alm_filters_title_element', 'h3' ) . '>';
	$output .= '</div>';

	return $output;
}

/**
 * Render filter description.
 *
 * @since 1.0
 * @param string $id  Filter ID.
 * @param array  $obj Filter array.
 * @return string     Raw HTML output.
 */
function alm_filters_display_description( $id = '', $obj = '' ) {
	if ( empty( $id ) || empty( $obj ) || ! isset( $obj['description'] ) ) {
		return false; // Bail early if empty.
	}

	$filter_key = alm_filters_get_filter_key( $obj );
	$hook_name  = $id . '_' . $filter_key;
	$value      = $obj['description'];

	if ( empty( $value ) && ! has_filter( 'alm_filters_' . $hook_name . '_description' ) ) {
		return false; // Bail early if description is empty && filter doesn't exist.
	}

	$output  = '<div class="alm-filter--description">';
	$output .= '<' . apply_filters( 'alm_filters_description_element', 'p' ) . ' id="alm-filter-' . $filter_key . '-description' . '">'; // phpcs:ignore
	$output .= htmlspecialchars_decode( apply_filters( 'alm_filters_' . $hook_name . '_description', $value ) );
	$output .= '</' . apply_filters( 'alm_filters_description_element', 'p' ) . '>';
	$output .= '</div>';

	return $output;
}

/**
 * Get the key for a filter group.
 *
 * @since 1.7.1
 * @param array $obj Filter array.
 * @return string    The key to locate.
 */
function alm_filters_get_filter_key( $obj = [] ) {
	if ( empty( $obj ) || ! isset( $obj['key'] ) ) {
		return false; // Bail early if empty.
	}

	switch ( $obj['key'] ) {
		case 'taxonomy':
			$key = $obj['taxonomy'];
			break;
		case 'meta':
			$key = $obj['meta_key'];
			break;
		default:
			$key = $obj['key'];
			break;
	}

	return $key;
}

/**
 * Get URL Query param for link URLs (Radio/Checkbox).
 *
 * @since 1.8.1
 * @param array  $obj  The filters array.
 * @param string $slug The slug of the URL.
 * @return string      The final URL.
 */
function alm_filters_build_url( $obj, $slug ) {
	if ( ! $obj || $obj['base_url'] === '' || ! $slug ) {
		return false;
	}
	$params = alm_filters_get_query_param( $obj );
	if ( ! $params ) {
		return false;
	}
	$url = $obj['base_url'] . '?' . $params . '=' . $slug;
	return $url;
}

/**
 * Get URL Query param for link URLs (Radio/Checkbox).
 *
 * @since 1.8.1
 * @param array $obj The filters array.
 */
function alm_filters_get_query_param( $obj ) {
	if ( ! $obj ) {
		return false;
	}

	if ( 'taxonomy' === $obj['key'] ) {
		$param = ( alm_filters_is_archive() ) ? '_' . $obj['taxonomy'] : $obj['taxonomy'];
	} elseif ( 'meta' === $obj['key'] ) {
		$param = $obj['meta_key'];
	} else {
		$param = $obj['key'];
	}

	return $param;
}

/**
 * Is the current page a front page or an archive, add _ to prevent redirects.
 *
 * @since 1.8.1
 */
function alm_filters_is_archive() {
	return is_home() || is_front_page() || is_archive() ? true : false;
}

/**
 * Is the current page a front page or an archive, add _ to prevent redirects.
 *
 * @param string $id The filter ID.
 * @since 1.8.1
 */
function alm_filters_add_underscore( $id = '' ) {
	/**
	 * Fix for underscores in taxonomy names when using on archive pages.
	 * Filter to remove '_' from redirects.
	 */
	$redirect_underscore = apply_filters( 'alm_filters_redirect_underscore_' . $id, true );

	if ( alm_filters_is_archive() && ! $redirect_underscore ) {
		return ''; // Exit early to remove the underscore.
	}

	return alm_filters_is_archive() ? '_' : '';
}

/**
 * Remove the underscore from the key.
 *
 * @param string $str The string to search.
 * @since 1.8.1
 */
function alm_filters_remove_underscore( $str ) {
	$first = $str[0];
	return $first === '_' ? substr( $str, 1 ) : $str;
}

/**
 * Open the `inner` wrapper for each filter.
 *
 * @param array   $obj    The filters array object.
 * @param boolean $toggle To toggle or not.
 * @param string  $status The current section status.
 * @return string         Raw HTML output.
 * @since 1.10.1
 */
function alm_filters_open_filter_container( $obj = '', $toggle = false, $status = 'expanded' ) {
	if ( empty( $obj ) ) {
		// Bail early if empty.
		return;
	}

	$aria_hidden = ( 'expanded' === $status ) ? 'false' : 'true';
	$style       = ( 'collapsed' === $status ) ? ' style="display: none;"' : '';

	$key  = alm_filters_get_filter_key( $obj );
	$aria = $toggle ? ' aria-hidden="' . $aria_hidden . '" aria-labelledby="alm-filter-' . $key . '-title" id="alm-filter-' . $key . '-inner"' : '';

	return '<div class="alm-filter--inner"' . $aria . $style . '>';
}

/**
 * Close the `inner` wrapper for each filter.
 *
 * @since 1.10.1
 */
function alm_filters_close_filter_container() {
	return '</div>';
}

/**
 * Render the text for `show_count`.
 *
 * @param boolean $show_count Should we display the count.
 * @param object  $item       The current filter item.
 * @param boolean $html       Render html or just a count.
 * @return string|mixed       Raw HTML output.
 * @since 1.11.0
 */
function alm_filters_build_count( $show_count, $item, $html ) {
	if ( ! isset( $item->count ) || ! isset( $show_count ) || ! $show_count ) {
		return;
	}
	$title = apply_filters( 'alm_filters_show_count_title', $item->count . __( ' results for ', 'ajax-load-more-filters' ) . $item->name, $item );
	$text  = apply_filters( 'alm_filters_show_count_display', $item->count, $item->count );
	return $html ? ' <span class="alm-filter-count" title="' . $title . '">' . $text . '</span>' : $text;
}

/**
 * The possible values for the sort order.
 */
function alm_filters_get_order_array() {
	return [
		'id',
		'author',
		'title',
		'name',
		'type',
		'date',
		'modified',
		'parent',
		'rand',
		'relevance',
		'menu_order',
		'post__in',
		'post__name_in',
		'post_parent__in',
	];
}

/**
 * Get all the facet post types as an array.
 *
 * @param  array $facets The current facets from options table.
 * @return array         An array of post types and facet IDs.
 */
function alm_filters_facet_get_post_types( $facets ) {
	if ( ! $facets ) {
		return '';
	}

	$results = [];
	foreach ( $facets as $facet ) { // Loop each facet.
		$id     = str_replace( ALM_FILTERS_FACET_PREFIX, '', $facet );
		$filter = ALMFilters::alm_filters_get_filter_by_id( $id, true );
		$types  = $filter && isset( $filter['facets_post_types'] ) ? $filter['facets_post_types'] : [];
		if ( $types ) {
			foreach ( $types as $type ) {
				$results[] = [
					'post_type' => $type,
					'id'        => $id,
				];
			}
		}
	}

	return $results;
}

/**
 * Get all distinct meta values for a given meta key.
 *
 * @param string $id         The filter ID.
 * @param string $meta_key   The meta key to get values for.
 * @param array  $post_types Optional. Array of post types to filter by.
 * @param int    $limit      Optional. Maximum number of results. Default 100.
 * @return array             Array of distinct meta values.
 */
function alm_filters_get_meta_values( $id = '', $meta_key = '', $post_types = [], $limit = 100 ) {
	if ( empty( $meta_key ) ) {
		return [];
	}

	global $wpdb;

	$limit = min( absint( $limit ), 500 ); // Cap at 500 results.

	// Build the query based on whether post types are specified.
	if ( ! empty( $post_types ) && is_array( $post_types ) ) {
		$placeholders = implode( ', ', array_fill( 0, count( $post_types ), '%s' ) );
		$query_args   = array_merge( [ $meta_key ], $post_types, [ $limit ] );

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		$meta_values = $wpdb->get_col(
			$wpdb->prepare(
				"SELECT DISTINCT pm.meta_value
				FROM {$wpdb->postmeta} pm
				INNER JOIN {$wpdb->posts} p ON pm.post_id = p.ID
				WHERE pm.meta_key = %s
				AND pm.meta_value != ''
				AND p.post_type IN ($placeholders)
				AND p.post_status = 'publish'
				ORDER BY pm.meta_value ASC
				LIMIT %d",
				$query_args
			)
		);
	} else {
		// Simple query without post type filtering.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$meta_values = $wpdb->get_col(
			$wpdb->prepare(
				"SELECT DISTINCT pm.meta_value
				FROM {$wpdb->postmeta} pm
				INNER JOIN {$wpdb->posts} p ON pm.post_id = p.ID
				WHERE pm.meta_key = %s
				AND pm.meta_value != ''
				AND p.post_status = 'publish'
				ORDER BY pm.meta_value ASC
				LIMIT %d",
				$meta_key,
				$limit
			)
		);
	}

	// Unserialize values and flatten arrays.
	$processed_values = [];
	foreach ( $meta_values as $value ) {
		$unserialized = maybe_unserialize( $value );

		if ( is_array( $unserialized ) ) {
			// Flatten array values.
			foreach ( $unserialized as $item ) {
				if ( is_scalar( $item ) && '' !== $item ) {
					$processed_values[] = (string) $item;
				}
			}
		} elseif ( is_scalar( $unserialized ) && '' !== $unserialized ) {
			$processed_values[] = (string) $unserialized;
		}
	}

	// Remove duplicates.
	$processed_values = array_unique( $processed_values );

	/**
	 * Filter the sort order for meta values.
	 *
	 * Return 'none' to disable sorting, 'asc' for ascending, 'desc' for descending,
	 * or 'numeric' for numeric sorting.
	 *
	 * @param string $order      The sort order. Default 'asc'.
	 */
	$sort_order = apply_filters( "alm/filters/{$id}/{$meta_key}/values_order", 'asc' );

	// Sort based on filter value.
	switch ( $sort_order ) {
		case 'none':
			// No sorting.
			break;
		case 'desc':
			rsort( $processed_values, SORT_NATURAL | SORT_FLAG_CASE );
			break;
		case 'numeric':
			sort( $processed_values, SORT_NUMERIC );
			break;
		case 'numeric_desc':
			rsort( $processed_values, SORT_NUMERIC );
			break;
		case 'asc':
		default:
			sort( $processed_values, SORT_NATURAL | SORT_FLAG_CASE );
			break;
	}

	/**
	 * Filter the list of meta values returned.
	 *
	 * @param array  $meta_values Array of meta values.
	 */
	return apply_filters( "alm/filters/{$id}/{$meta_key}/values", $processed_values );
}
