/**
 * Frontend JavaScript for Gambling Laws Widget
 */

(function($) {
    'use strict';
    
    // Initialize widget when document is ready
    $(document).ready(function() {
        initializeGamblingLawsWidgets();
    });
    
    function initializeGamblingLawsWidgets() {
        $('.gambling-laws-widget-wp').each(function() {
            var $widget = $(this);
            var widgetId = $widget.attr('id');
            
            if (!widgetId) {
                return;
            }
            
            // Initialize this widget instance
            initializeWidget($widget);
        });
    }
    
    function initializeWidget($widget) {
        var $dropdown = $widget.find('.glw-state-dropdown');
        var $loadingSection = $widget.find('.glw-loading-section');
        var $resultsSection = $widget.find('.glw-results-section');
        var $ctaSection = $widget.find('.glw-cta-section');
        
        // Load states data
        loadStatesData($widget);
        
        // Handle state selection
        $dropdown.on('change', function() {
            var selectedStateKey = $(this).val();
            
            if (selectedStateKey) {
                loadStateData($widget, selectedStateKey);
            } else {
                hideResults($widget);
            }
        });
    }
    
    function loadStatesData($widget) {
        var $loadingSection = $widget.find('.glw-loading-section');
        var $dropdown = $widget.find('.glw-state-dropdown');
        
        // Show loading
        $loadingSection.show();
        
        $.ajax({
            url: glw_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'get_gambling_laws',
                nonce: glw_ajax.nonce
            },
            success: function(response) {
                $loadingSection.hide();
                
                if (response.success && response.data.states) {
                    populateStatesDropdown($widget, response.data.states);
                    
                    // Update last updated date
                    $widget.find('.glw-update-date').text(response.data.last_updated);
                    
                    // Store states data for later use
                    $widget.data('states', response.data.states);
                } else {
                    showError($widget, 'Failed to load gambling laws data.');
                }
            },
            error: function() {
                $loadingSection.hide();
                showError($widget, 'Failed to connect to server.');
            }
        });
    }
    
    function populateStatesDropdown($widget, states) {
        var $dropdown = $widget.find('.glw-state-dropdown');
        
        // Clear existing options except the first one
        $dropdown.find('option:not(:first)').remove();
        
        // Sort states alphabetically
        states.sort(function(a, b) {
            return a.state_name.localeCompare(b.state_name);
        });
        
        // Add state options
        states.forEach(function(state) {
            $dropdown.append(
                '<option value="' + escapeHtml(state.state_key) + '">' + 
                escapeHtml(state.state_name) + 
                '</option>'
            );
        });
    }
    
    function loadStateData($widget, stateKey) {
        var states = $widget.data('states');
        var state = states.find(function(s) {
            return s.state_key === stateKey;
        });
        
        if (state) {
            displayStateData($widget, state);
        }
    }
    
    function displayStateData($widget, state) {
        // Update state header
        $widget.find('.glw-state-name').text(state.state_name + ' Gambling Laws');
        $widget.find('.glw-min-age').text(state.min_age);
        
        // Update status badges
        updateStatusBadge($widget, '.glw-sports-betting', state.sports_betting, 'sports');
        updateStatusBadge($widget, '.glw-online-casino', state.online_casino, 'casino');
        updateStatusBadge($widget, '.glw-online-poker', state.online_poker, 'poker');
        updateStatusBadge($widget, '.glw-retail-casino', state.retail_casino, 'retail');
        updateStatusBadge($widget, '.glw-dfs', state.dfs, 'dfs');
        
        // Update notes
        var $notesSection = $widget.find('.glw-notes-section');
        if (state.notes && state.notes.trim()) {
            $widget.find('.glw-notes-text').text(state.notes);
            $notesSection.show();
        } else {
            $notesSection.hide();
        }
        
        // Show results and hide CTA
        $widget.find('.glw-results-section').show();
        $widget.find('.glw-cta-section').hide();
        
        // Smooth scroll to results
        $('html, body').animate({
            scrollTop: $widget.find('.glw-results-section').offset().top - 20
        }, 500);
    }
    
    function updateStatusBadge($widget, selector, status, type) {
        var $badge = $widget.find(selector);
        
        // Remove existing classes
        $badge.removeClass('glw-legal glw-illegal glw-coming-soon glw-tribal-only glw-grey-market');
        
        var className = '';
        var icon = '';
        var showIcons = true; // This could be made configurable
        
        switch (status.toLowerCase()) {
            case 'legal':
                className = 'glw-legal';
                if (showIcons) {
                    switch (type) {
                        case 'sports': icon = '✅ '; break;
                        case 'casino': icon = '🎰 '; break;
                        case 'poker': icon = '🃏 '; break;
                        case 'retail': icon = '🏢 '; break;
                        case 'dfs': icon = '🎯 '; break;
                        default: icon = '✅ '; break;
                    }
                }
                break;
            case 'illegal':
            case 'not legal':
                className = 'glw-illegal';
                if (showIcons) icon = '❌ ';
                break;
            case 'coming soon':
                className = 'glw-coming-soon';
                if (showIcons) icon = '⚠️ ';
                break;
            case 'tribal only':
                className = 'glw-tribal-only';
                if (showIcons) icon = '⚠️ ';
                break;
            case 'grey market':
                className = 'glw-grey-market';
                if (showIcons) icon = '⚠️ ';
                break;
            default:
                className = 'glw-illegal';
                if (showIcons) icon = '❌ ';
        }
        
        $badge.addClass(className).html(icon + escapeHtml(status));
    }
    
    function hideResults($widget) {
        $widget.find('.glw-results-section').hide();
        $widget.find('.glw-cta-section').show();
    }
    
    function showError($widget, message) {
        var $ctaSection = $widget.find('.glw-cta-section');
        var $ctaCard = $ctaSection.find('.glw-cta-card');
        
        $ctaCard.html(
            '<h3 class="glw-cta-title" style="color: #e53e3e;">⚠️ Error Loading Data</h3>' +
            '<p class="glw-cta-text">' + escapeHtml(message) + '</p>' +
            '<button type="button" onclick="window.location.reload()" class="button" style="background: #e53e3e; color: white; border: none; padding: 10px 20px; border-radius: 4px; cursor: pointer;">🔄 Reload Page</button>'
        );
        
        $ctaSection.show();
        $widget.find('.glw-results-section').hide();
    }
    
    function escapeHtml(text) {
        var map = {
            '&': '&amp;',
            '<': '&lt;',
            '>': '&gt;',
            '"': '&quot;',
            "'": '&#039;'
        };
        
        return text.replace(/[&<>"']/g, function(m) { return map[m]; });
    }
    
    // Handle multiple widgets on the same page
    function getWidgetStates($widget) {
        return $widget.data('states') || [];
    }
    
    // Public API for external use
    window.GamblingLawsWidget = {
        refresh: function(widgetId) {
            var $widget = $('#' + widgetId);
            if ($widget.length) {
                loadStatesData($widget);
            }
        },
        selectState: function(widgetId, stateKey) {
            var $widget = $('#' + widgetId);
            if ($widget.length) {
                $widget.find('.glw-state-dropdown').val(stateKey).trigger('change');
            }
        }
    };
    
})(jQuery);