<?php
/**
 * Admin Page for Gambling Laws Widget
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

global $wpdb;
$table_name = $wpdb->prefix . 'gambling_laws';

// Check if setup was completed
if (isset($_GET['setup_complete'])) {
    $count = intval($_GET['setup_complete']);
    echo '<div class="notice notice-success"><p>Database setup completed! Added ' . $count . ' states.</p></div>';
}

// Handle form submission
if (isset($_POST['submit']) && wp_verify_nonce($_POST['glw_nonce'], 'save_gambling_laws')) {
    $state_key = sanitize_text_field($_POST['state_key']);
    
    $result = $wpdb->update(
        $table_name,
        array(
            'state_name' => sanitize_text_field($_POST['state_name']),
            'sports_betting' => sanitize_text_field($_POST['sports_betting']),
            'online_casino' => sanitize_text_field($_POST['online_casino']),
            'online_poker' => sanitize_text_field($_POST['online_poker']),
            'retail_casino' => sanitize_text_field($_POST['retail_casino']),
            'dfs' => sanitize_text_field($_POST['dfs']),
            'min_age' => intval($_POST['min_age']),
            'notes' => sanitize_textarea_field($_POST['notes'])
        ),
        array('state_key' => $state_key)
    );
    
    if ($result !== false) {
        echo '<div class="notice notice-success"><p>State updated successfully!</p></div>';
    } else {
        echo '<div class="notice notice-error"><p>Failed to update state.</p></div>';
    }
}

// Check if table exists and has data
$table_exists = $wpdb->get_var("SHOW TABLES LIKE '$table_name'") == $table_name;
$data_count = $table_exists ? $wpdb->get_var("SELECT COUNT(*) FROM $table_name") : 0;

// Get selected state data
$selected_state = isset($_GET['state']) ? sanitize_text_field($_GET['state']) : '';
$state_data = null;

if ($selected_state && $table_exists) {
    $state_data = $wpdb->get_row(
        $wpdb->prepare("SELECT * FROM $table_name WHERE state_key = %s", $selected_state),
        ARRAY_A
    );
}

// Get all states for dropdown
$states = array();
if ($table_exists && $data_count > 0) {
    $states = $wpdb->get_results("SELECT state_key, state_name FROM $table_name ORDER BY state_name ASC", ARRAY_A);
}
?>

<div class="wrap">
    <h1>🎰 Gambling Laws Widget - Manage States</h1>
    
    <?php if (!$table_exists || $data_count == 0): ?>
        <!-- Database Setup Required -->
        <div class="notice notice-warning">
            <p><strong>Database Setup Required!</strong></p>
            <p>The gambling laws database table is not set up or contains no data.</p>
            <p>
                <a href="<?php echo admin_url('admin.php?page=gambling-laws-widget&glw_setup=1'); ?>" 
                   class="button button-primary">
                   🔧 Set Up Database Now
                </a>
            </p>
        </div>
        
        <div class="glw-setup-info">
            <h3>What this will do:</h3>
            <ul>
                <li>✅ Create the gambling laws database table</li>
                <li>✅ Insert data for all 50 states + Washington D.C.</li>
                <li>✅ Set up current 2025 gambling law statuses</li>
                <li>✅ Enable the state dropdown and editing features</li>
            </ul>
        </div>
    <?php else: ?>
        <!-- Normal Admin Interface -->
        <div class="glw-admin-container">
            <!-- State Selector -->
            <div class="glw-state-selector">
                <h2>Select a State to Edit</h2>
                <form method="get" action="">
                    <input type="hidden" name="page" value="gambling-laws-widget">
                    <select name="state" onchange="this.form.submit()" class="glw-state-dropdown">
                        <option value="">Choose a state...</option>
                        <?php foreach ($states as $state): ?>
                            <option value="<?php echo esc_attr($state['state_key']); ?>" 
                                    <?php selected($selected_state, $state['state_key']); ?>>
                                <?php echo esc_html($state['state_name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </form>
            </div>

        <?php if ($state_data): ?>
        <!-- State Edit Form -->
        <div class="glw-edit-form">
            <h2>Edit <?php echo esc_html($state_data['state_name']); ?> Gambling Laws</h2>
            
            <form method="post" action="">
                <?php wp_nonce_field('save_gambling_laws', 'glw_nonce'); ?>
                <input type="hidden" name="state_key" value="<?php echo esc_attr($state_data['state_key']); ?>">
                
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="state_name">State Name</label>
                        </th>
                        <td>
                            <input type="text" id="state_name" name="state_name" 
                                   value="<?php echo esc_attr($state_data['state_name']); ?>" 
                                   class="regular-text" required>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="sports_betting">✅ Online Sports Betting</label>
                        </th>
                        <td>
                            <select id="sports_betting" name="sports_betting" required>
                                <option value="Legal" <?php selected($state_data['sports_betting'], 'Legal'); ?>>Legal</option>
                                <option value="Illegal" <?php selected($state_data['sports_betting'], 'Illegal'); ?>>Illegal</option>
                                <option value="Coming Soon" <?php selected($state_data['sports_betting'], 'Coming Soon'); ?>>Coming Soon</option>
                            </select>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="online_casino">🎰 Online Casino Gambling</label>
                        </th>
                        <td>
                            <select id="online_casino" name="online_casino" required>
                                <option value="Legal" <?php selected($state_data['online_casino'], 'Legal'); ?>>Legal</option>
                                <option value="Illegal" <?php selected($state_data['online_casino'], 'Illegal'); ?>>Illegal</option>
                                <option value="Coming Soon" <?php selected($state_data['online_casino'], 'Coming Soon'); ?>>Coming Soon</option>
                            </select>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="online_poker">🃏 Online Poker</label>
                        </th>
                        <td>
                            <select id="online_poker" name="online_poker" required>
                                <option value="Legal" <?php selected($state_data['online_poker'], 'Legal'); ?>>Legal</option>
                                <option value="Illegal" <?php selected($state_data['online_poker'], 'Illegal'); ?>>Illegal</option>
                                <option value="Coming Soon" <?php selected($state_data['online_poker'], 'Coming Soon'); ?>>Coming Soon</option>
                                <option value="Grey Market" <?php selected($state_data['online_poker'], 'Grey Market'); ?>>Grey Market</option>
                            </select>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="retail_casino">🏢 Retail Casinos</label>
                        </th>
                        <td>
                            <select id="retail_casino" name="retail_casino" required>
                                <option value="Legal" <?php selected($state_data['retail_casino'], 'Legal'); ?>>Legal</option>
                                <option value="Illegal" <?php selected($state_data['retail_casino'], 'Illegal'); ?>>Illegal</option>
                                <option value="Tribal Only" <?php selected($state_data['retail_casino'], 'Tribal Only'); ?>>Tribal Only</option>
                            </select>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="dfs">🎯 Daily Fantasy Sports (DFS)</label>
                        </th>
                        <td>
                            <select id="dfs" name="dfs" required>
                                <option value="Legal" <?php selected($state_data['dfs'], 'Legal'); ?>>Legal</option>
                                <option value="Illegal" <?php selected($state_data['dfs'], 'Illegal'); ?>>Illegal</option>
                                <option value="Unregulated" <?php selected($state_data['dfs'], 'Unregulated'); ?>>Unregulated</option>
                            </select>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="min_age">🧑‍⚖️ Minimum Legal Age</label>
                        </th>
                        <td>
                            <select id="min_age" name="min_age" required>
                                <option value="18" <?php selected($state_data['min_age'], 18); ?>>18</option>
                                <option value="21" <?php selected($state_data['min_age'], 21); ?>>21</option>
                            </select>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="notes">💰 Tax & Licensing Notes</label>
                        </th>
                        <td>
                            <textarea id="notes" name="notes" rows="4" cols="50" class="large-text"><?php echo esc_textarea($state_data['notes']); ?></textarea>
                            <p class="description">Additional information about tax rates, launch dates, licensing requirements, etc.</p>
                        </td>
                    </tr>
                </table>
                
                <p class="submit">
                    <input type="submit" name="submit" id="submit" class="button-primary" value="Update State Laws">
                </p>
            </form>
            
            <!-- Last Updated Info -->
            <div class="glw-last-updated">
                <p><strong>Last Updated:</strong> <?php echo esc_html($state_data['last_updated']); ?></p>
            </div>
        </div>
        <?php endif; ?>

        <!-- Shortcode Information -->
        <div class="glw-shortcode-info">
            <h2>📋 Usage Instructions</h2>
            <div class="glw-info-box">
                <h3>Shortcode Usage</h3>
                <p>Use this shortcode to display the gambling laws widget on any page or post:</p>
                <code>[gambling_laws_widget]</code>
                
                <h3>Alternative Shortcode</h3>
                <code>[gambling_laws]</code>
                
                <h3>Widget Features</h3>
                <ul>
                    <li>✅ All 50 states + Washington D.C.</li>
                    <li>🎯 Comprehensive gambling law categories</li>
                    <li>📱 Mobile-responsive design</li>
                    <li>🔄 Real-time updates from your edits</li>
                    <li>🎨 Professional gambling industry styling</li>
                </ul>
            </div>
        </div>

        <!-- Export/Import -->
        <div class="glw-export-import">
            <h2>📁 Export / Import</h2>
            <div class="glw-info-box">
                <p>
                    <a href="<?php echo admin_url('admin-ajax.php?action=export_gambling_laws&_wpnonce=' . wp_create_nonce('glw_export')); ?>" 
                       class="button">📥 Export All Data (JSON)</a>
                </p>
                <p>
                    <input type="file" id="import_file" accept=".json">
                    <button type="button" id="import_btn" class="button">📤 Import Data</button>
                </p>
            </div>
        </div>
    <?php endif; ?>
</div>