<?php
/**
 * Settings Page for Gambling Laws Widget
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Handle form submission
if (isset($_POST['submit'])) {
    if (wp_verify_nonce($_POST['glw_settings_nonce'], 'save_settings')) {
        $options = array(
            'widget_title' => sanitize_text_field($_POST['widget_title']),
            'widget_subtitle' => sanitize_textarea_field($_POST['widget_subtitle']),
            'color_scheme' => sanitize_text_field($_POST['color_scheme']),
            'show_icons' => isset($_POST['show_icons']) ? 1 : 0,
            'show_last_updated' => isset($_POST['show_last_updated']) ? 1 : 0,
            'custom_css' => sanitize_textarea_field($_POST['custom_css'])
        );
        
        update_option('gambling_laws_options', $options);
        echo '<div class="notice notice-success"><p>Settings saved successfully!</p></div>';
    }
}

$options = get_option('gambling_laws_options', array(
    'widget_title' => 'U.S. State Gambling Laws Directory',
    'widget_subtitle' => 'Comprehensive guide to gambling and sports betting laws across all 50 states + Washington D.C.',
    'color_scheme' => 'default',
    'show_icons' => 1,
    'show_last_updated' => 1,
    'custom_css' => ''
));
?>

<div class="wrap">
    <h1>⚙️ Gambling Laws Widget - Settings</h1>
    
    <form method="post" action="">
        <?php wp_nonce_field('save_settings', 'glw_settings_nonce'); ?>
        
        <table class="form-table">
            <tr>
                <th scope="row">
                    <label for="widget_title">Widget Title</label>
                </th>
                <td>
                    <input type="text" id="widget_title" name="widget_title" 
                           value="<?php echo esc_attr($options['widget_title']); ?>" 
                           class="regular-text">
                    <p class="description">Main title displayed at the top of the widget</p>
                </td>
            </tr>
            
            <tr>
                <th scope="row">
                    <label for="widget_subtitle">Widget Subtitle</label>
                </th>
                <td>
                    <textarea id="widget_subtitle" name="widget_subtitle" 
                              rows="3" cols="50" class="large-text"><?php echo esc_textarea($options['widget_subtitle']); ?></textarea>
                    <p class="description">Subtitle/description displayed below the main title</p>
                </td>
            </tr>
            
            <tr>
                <th scope="row">
                    <label for="color_scheme">Color Scheme</label>
                </th>
                <td>
                    <select id="color_scheme" name="color_scheme">
                        <option value="default" <?php selected($options['color_scheme'], 'default'); ?>>Default (Blue/Gold)</option>
                        <option value="dark" <?php selected($options['color_scheme'], 'dark'); ?>>Dark Theme</option>
                        <option value="light" <?php selected($options['color_scheme'], 'light'); ?>>Light Theme</option>
                        <option value="custom" <?php selected($options['color_scheme'], 'custom'); ?>>Custom CSS</option>
                    </select>
                </td>
            </tr>
            
            <tr>
                <th scope="row">Display Options</th>
                <td>
                    <fieldset>
                        <label for="show_icons">
                            <input type="checkbox" id="show_icons" name="show_icons" value="1" 
                                   <?php checked($options['show_icons'], 1); ?>>
                            Show status icons (✅🎰🃏🏢🎯❌⚠️)
                        </label><br>
                        
                        <label for="show_last_updated">
                            <input type="checkbox" id="show_last_updated" name="show_last_updated" value="1" 
                                   <?php checked($options['show_last_updated'], 1); ?>>
                            Show "Last Updated" information
                        </label>
                    </fieldset>
                </td>
            </tr>
            
            <tr>
                <th scope="row">
                    <label for="custom_css">Custom CSS</label>
                </th>
                <td>
                    <textarea id="custom_css" name="custom_css" 
                              rows="10" cols="50" class="large-text code"><?php echo esc_textarea($options['custom_css']); ?></textarea>
                    <p class="description">Add custom CSS to override default styling (only applies when Color Scheme is set to "Custom CSS")</p>
                </td>
            </tr>
        </table>
        
        <p class="submit">
            <input type="submit" name="submit" id="submit" class="button-primary" value="Save Settings">
        </p>
    </form>
    
    <!-- Plugin Information -->
    <div class="glw-plugin-info">
        <h2>📋 Plugin Information</h2>
        <div class="glw-info-box">
            <table class="widefat">
                <tr>
                    <td><strong>Plugin Version:</strong></td>
                    <td><?php echo GLW_VERSION; ?></td>
                </tr>
                <tr>
                    <td><strong>WordPress Version:</strong></td>
                    <td><?php echo get_bloginfo('version'); ?></td>
                </tr>
                <tr>
                    <td><strong>PHP Version:</strong></td>
                    <td><?php echo PHP_VERSION; ?></td>
                </tr>
                <tr>
                    <td><strong>Database Table:</strong></td>
                    <td><?php global $wpdb; echo $wpdb->prefix . 'gambling_laws'; ?></td>
                </tr>
                <tr>
                    <td><strong>Total States:</strong></td>
                    <td>
                        <?php 
                        global $wpdb;
                        $count = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}gambling_laws");
                        echo $count;
                        ?>
                    </td>
                </tr>
            </table>
        </div>
    </div>
    
    <!-- Support Information -->
    <div class="glw-support-info">
        <h2>🆘 Support & Documentation</h2>
        <div class="glw-info-box">
            <h3>Shortcode Usage</h3>
            <p>Use <code>[gambling_laws_widget]</code> or <code>[gambling_laws]</code> to display the widget.</p>
            
            <h3>Common Issues</h3>
            <ul>
                <li><strong>Widget not displaying:</strong> Check if shortcode is placed correctly</li>
                <li><strong>Styling issues:</strong> Try clearing cache and checking for theme conflicts</li>
                <li><strong>Data not updating:</strong> Verify WordPress database permissions</li>
            </ul>
            
            <h3>Reset Plugin Data</h3>
            <p>
                <button type="button" id="reset_plugin_data" class="button button-secondary" 
                        onclick="if(confirm('Are you sure? This will reset all gambling laws data to defaults.')) { window.location.href='?page=gambling-laws-settings&reset=1&_wpnonce=<?php echo wp_create_nonce('reset_data'); ?>'; }">
                    🔄 Reset All Data to Defaults
                </button>
            </p>
        </div>
    </div>
</div>

<?php
// Handle reset
if (isset($_GET['reset']) && wp_verify_nonce($_GET['_wpnonce'], 'reset_data')) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'gambling_laws';
    $wpdb->query("TRUNCATE TABLE $table_name");
    
    // Re-insert default data
    $plugin = new GamblingLawsWidget();
    $plugin->insert_default_data();
    
    echo '<div class="notice notice-success"><p>Plugin data has been reset to defaults!</p></div>';
}
?>