/**
 * Admin JavaScript for Gambling Laws Widget
 */

(function($) {
    'use strict';
    
    $(document).ready(function() {
        initializeAdmin();
    });
    
    function initializeAdmin() {
        handleImportExport();
        handleFormValidation();
        handleStatusPreview();
        handleBulkActions();
    }
    
    function handleImportExport() {
        // Export functionality
        $(document).on('click', '[href*="export_gambling_laws"]', function(e) {
            e.preventDefault();
            exportGamblingLaws();
        });
        
        // Import functionality
        $('#import_btn').on('click', function() {
            var fileInput = $('#import_file')[0];
            if (fileInput.files.length === 0) {
                alert('Please select a JSON file to import.');
                return;
            }
            
            importGamblingLaws(fileInput.files[0]);
        });
    }
    
    function exportGamblingLaws() {
        // Show loading state
        var $exportBtn = $('[href*="export_gambling_laws"]');
        var originalText = $exportBtn.text();
        $exportBtn.text('📥 Exporting...');
        
        $.ajax({
            url: glw_admin_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'export_gambling_laws',
                nonce: glw_admin_ajax.nonce
            },
            success: function(response) {
                if (response.success) {
                    // Create and trigger download
                    var dataStr = "data:text/json;charset=utf-8," + encodeURIComponent(JSON.stringify(response.data, null, 2));
                    var downloadAnchorNode = document.createElement('a');
                    downloadAnchorNode.setAttribute("href", dataStr);
                    downloadAnchorNode.setAttribute("download", "gambling-laws-" + getCurrentDate() + ".json");
                    document.body.appendChild(downloadAnchorNode);
                    downloadAnchorNode.click();
                    downloadAnchorNode.remove();
                    
                    showAdminNotice('Export completed successfully!', 'success');
                } else {
                    showAdminNotice('Export failed: ' + response.data, 'error');
                }
                
                $exportBtn.text(originalText);
            },
            error: function() {
                showAdminNotice('Export failed: Server error', 'error');
                $exportBtn.text(originalText);
            }
        });
    }
    
    function importGamblingLaws(file) {
        var reader = new FileReader();
        
        reader.onload = function(e) {
            try {
                var jsonData = JSON.parse(e.target.result);
                
                // Validate JSON structure
                if (!validateImportData(jsonData)) {
                    showAdminNotice('Invalid file format. Please check your JSON structure.', 'error');
                    return;
                }
                
                // Show loading state
                $('#import_btn').text('📤 Importing...').prop('disabled', true);
                
                $.ajax({
                    url: glw_admin_ajax.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'import_gambling_laws',
                        nonce: glw_admin_ajax.nonce,
                        import_data: JSON.stringify(jsonData)
                    },
                    success: function(response) {
                        if (response.success) {
                            showAdminNotice('Import completed successfully!', 'success');
                            setTimeout(function() {
                                window.location.reload();
                            }, 2000);
                        } else {
                            showAdminNotice('Import failed: ' + response.data, 'error');
                        }
                        
                        $('#import_btn').text('📤 Import Data').prop('disabled', false);
                    },
                    error: function() {
                        showAdminNotice('Import failed: Server error', 'error');
                        $('#import_btn').text('📤 Import Data').prop('disabled', false);
                    }
                });
                
            } catch (error) {
                showAdminNotice('Invalid JSON file format.', 'error');
            }
        };
        
        reader.readAsText(file);
    }
    
    function validateImportData(data) {
        // Basic validation for import data structure
        if (!data || typeof data !== 'object') {
            return false;
        }
        
        // Check if it has states data
        if (data.states && typeof data.states === 'object') {
            // Validate a few state entries
            var stateKeys = Object.keys(data.states);
            if (stateKeys.length === 0) {
                return false;
            }
            
            var sampleState = data.states[stateKeys[0]];
            var requiredFields = ['state_name', 'sports_betting', 'online_casino', 'online_poker', 'retail_casino', 'dfs', 'min_age'];
            
            return requiredFields.every(function(field) {
                return sampleState.hasOwnProperty(field);
            });
        }
        
        return false;
    }
    
    function handleFormValidation() {
        // Add validation to the state edit form
        $('form[method="post"]').on('submit', function(e) {
            var form = this;
            var isValid = true;
            var errors = [];
            
            // Validate state name
            var stateName = $(form).find('[name="state_name"]').val();
            if (!stateName || stateName.trim().length === 0) {
                errors.push('State name is required.');
                isValid = false;
            }
            
            // Validate minimum age
            var minAge = $(form).find('[name="min_age"]').val();
            if (!minAge || (minAge != 18 && minAge != 21)) {
                errors.push('Minimum age must be 18 or 21.');
                isValid = false;
            }
            
            // Validate status fields
            var statusFields = ['sports_betting', 'online_casino', 'online_poker', 'retail_casino', 'dfs'];
            statusFields.forEach(function(field) {
                var value = $(form).find('[name="' + field + '"]').val();
                if (!value || value.trim().length === 0) {
                    errors.push(field.replace('_', ' ') + ' status is required.');
                    isValid = false;
                }
            });
            
            if (!isValid) {
                e.preventDefault();
                showAdminNotice('Please fix the following errors:\n• ' + errors.join('\n• '), 'error');
            }
        });
    }
    
    function handleStatusPreview() {
        // Add live preview for status changes
        $('select[name$="_betting"], select[name$="_casino"], select[name$="_poker"], select[name$="casino"], select[name="dfs"]').on('change', function() {
            var $select = $(this);
            var value = $select.val();
            var $preview = $select.siblings('.glw-status-preview');
            
            // Remove existing preview
            $preview.remove();
            
            // Add new preview
            var className = getStatusClass(value);
            var $newPreview = $('<span class="glw-status-preview ' + className + '">' + value + '</span>');
            $select.after($newPreview);
        });
        
        // Initialize existing previews
        $('select[name$="_betting"], select[name$="_casino"], select[name$="_poker"], select[name$="casino"], select[name="dfs"]').trigger('change');
    }
    
    function getStatusClass(status) {
        switch (status.toLowerCase()) {
            case 'legal':
                return 'legal';
            case 'illegal':
            case 'not legal':
                return 'illegal';
            case 'coming soon':
            case 'tribal only':
                return 'coming-soon';
            default:
                return 'illegal';
        }
    }
    
    function handleBulkActions() {
        // Add bulk update functionality (future enhancement)
        $('#bulk-update-btn').on('click', function() {
            // Implementation for bulk updates
            showAdminNotice('Bulk update functionality coming soon!', 'info');
        });
    }
    
    function showAdminNotice(message, type) {
        type = type || 'info';
        
        var noticeClass = 'notice notice-' + type;
        if (type === 'success') {
            noticeClass += ' is-dismissible';
        }
        
        var $notice = $('<div class="' + noticeClass + '"><p>' + escapeHtml(message) + '</p></div>');
        
        // Remove existing notices
        $('.glw-admin-container .notice').remove();
        
        // Add new notice
        $('.glw-admin-container').prepend($notice);
        
        // Auto-dismiss success notices
        if (type === 'success') {
            setTimeout(function() {
                $notice.fadeOut();
            }, 5000);
        }
        
        // Scroll to notice
        $('html, body').animate({
            scrollTop: $notice.offset().top - 50
        }, 300);
    }
    
    function getCurrentDate() {
        var now = new Date();
        var year = now.getFullYear();
        var month = String(now.getMonth() + 1).padStart(2, '0');
        var day = String(now.getDate()).padStart(2, '0');
        return year + '-' + month + '-' + day;
    }
    
    function escapeHtml(text) {
        var map = {
            '&': '&amp;',
            '<': '&lt;',
            '>': '&gt;',
            '"': '&quot;',
            "'": '&#039;'
        };
        
        return text.replace(/[&<>"']/g, function(m) { return map[m]; });
    }
    
    // AJAX Save functionality for quick updates
    function saveStateAjax(stateKey, stateData) {
        $.ajax({
            url: glw_admin_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'save_gambling_laws',
                nonce: glw_admin_ajax.nonce,
                state_data: stateData
            },
            success: function(response) {
                if (response.success) {
                    showAdminNotice('State updated successfully!', 'success');
                } else {
                    showAdminNotice('Failed to update state: ' + response.data, 'error');
                }
            },
            error: function() {
                showAdminNotice('Failed to update state: Server error', 'error');
            }
        });
    }
    
    // Initialize tooltips (if needed)
    function initializeTooltips() {
        $('[data-tooltip]').on('mouseenter', function() {
            var tooltip = $(this).data('tooltip');
            // Add tooltip implementation if needed
        });
    }
    
})(jQuery);