<?php
/**
 * Plugin Name: U.S. State Gambling Laws Widget
 * Plugin URI: https://gamblingsite.com/gambling-laws-widget
 * Description: A comprehensive widget displaying gambling and sports betting laws for all 50 U.S. states + Washington D.C. Includes admin interface for easy updates and shortcode for embedding.
 * Version: 1.0.1
 * Author: GamblingSite.com
 * Author URI: https://gamblingsite.com
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: gambling-laws-widget
 * Domain Path: /languages
 * 
 * Requires at least: 5.0
 * Tested up to: 6.4
 * Requires PHP: 7.4
 * 
 * @package GamblingLawsWidget
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('GLW_VERSION', '1.0.0');
define('GLW_PLUGIN_URL', plugin_dir_url(__FILE__));
define('GLW_PLUGIN_PATH', plugin_dir_path(__FILE__));
define('GLW_PLUGIN_FILE', __FILE__);

/**
 * Main Gambling Laws Widget Class
 */
class GamblingLawsWidget {
    
    /**
     * Constructor
     */
    public function __construct() {
        add_action('init', array($this, 'init'));
    }
    
    /**
     * Initialize the plugin
     */
    public function init() {
        // Load plugin textdomain
        load_plugin_textdomain('gambling-laws-widget', false, dirname(plugin_basename(__FILE__)) . '/languages');
        
        // Register activation/deactivation hooks
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
        
        // Initialize components
        $this->init_hooks();
        $this->load_admin();
        $this->load_frontend();
    }
    
    /**
     * Initialize WordPress hooks
     */
    private function init_hooks() {
        add_action('wp_enqueue_scripts', array($this, 'enqueue_frontend_assets'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
        add_action('admin_init', array($this, 'manual_setup'));
        add_shortcode('gambling_laws_widget', array($this, 'shortcode'));
        add_shortcode('gambling_laws', array($this, 'shortcode'));
    }
    
    /**
     * Load admin functionality
     */
    private function load_admin() {
        if (is_admin()) {
            add_action('admin_menu', array($this, 'add_admin_menu'));
            add_action('admin_init', array($this, 'admin_init'));
            add_action('wp_ajax_save_gambling_laws', array($this, 'ajax_save_gambling_laws'));
            add_action('wp_ajax_import_gambling_laws', array($this, 'ajax_import_gambling_laws'));
        }
    }
    
    /**
     * Load frontend functionality
     */
    private function load_frontend() {
        add_action('wp_ajax_get_gambling_laws', array($this, 'ajax_get_gambling_laws'));
        add_action('wp_ajax_nopriv_get_gambling_laws', array($this, 'ajax_get_gambling_laws'));
    }
    
    /**
     * Plugin activation
     */
    public function activate() {
        $this->create_tables();
        $this->insert_default_data();
        flush_rewrite_rules();
    }
    
    /**
     * Plugin deactivation
     */
    public function deactivate() {
        flush_rewrite_rules();
    }
    
    /**
     * Create database tables
     */
    private function create_tables() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'gambling_laws';
        
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE $table_name (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            state_key varchar(50) NOT NULL,
            state_name varchar(100) NOT NULL,
            sports_betting varchar(50) NOT NULL DEFAULT 'Illegal',
            online_casino varchar(50) NOT NULL DEFAULT 'Illegal',
            online_poker varchar(50) NOT NULL DEFAULT 'Illegal',
            retail_casino varchar(50) NOT NULL DEFAULT 'Illegal',
            dfs varchar(50) NOT NULL DEFAULT 'Legal',
            min_age int(2) NOT NULL DEFAULT 21,
            notes text,
            last_updated datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY state_key (state_key)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
    
    /**
     * Insert default gambling laws data
     */
    private function insert_default_data() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'gambling_laws';
        
        // Check if data already exists
        $count = $wpdb->get_var("SELECT COUNT(*) FROM $table_name");
        if ($count > 0) {
            return;
        }
        
        // Default state data
        $states = array(
            array('alabama', 'Alabama', 'Illegal', 'Illegal', 'Illegal', 'Tribal Only', 'Legal', 21, 'Limited tribal gaming only. No commercial casinos.'),
            array('alaska', 'Alaska', 'Illegal', 'Illegal', 'Illegal', 'Illegal', 'Legal', 21, 'Very limited gambling options. Small social gambling allowed.'),
            array('arizona', 'Arizona', 'Legal', 'Illegal', 'Illegal', 'Legal', 'Legal', 21, 'Sports betting legalized 2021. 10-20% tax on GGR.'),
            array('arkansas', 'Arkansas', 'Legal', 'Illegal', 'Illegal', 'Legal', 'Legal', 21, 'Sports betting launched 2022. 13-20% tax rates.'),
            array('california', 'California', 'Illegal', 'Illegal', 'Illegal', 'Tribal Only', 'Legal', 21, 'Large tribal gaming market. Sports betting ballot measures failed.'),
            array('colorado', 'Colorado', 'Legal', 'Illegal', 'Illegal', 'Legal', 'Legal', 21, 'Sports betting launched 2020. 10% tax on operator revenue.'),
            array('connecticut', 'Connecticut', 'Legal', 'Legal', 'Legal', 'Legal', 'Legal', 21, 'Full online gambling launched 2021. Revenue sharing with tribes.'),
            array('delaware', 'Delaware', 'Legal', 'Legal', 'Legal', 'Legal', 'Legal', 21, 'Pioneer in sports betting (2018). Single-game sports betting legal.'),
            array('florida', 'Florida', 'Illegal', 'Illegal', 'Illegal', 'Tribal Only', 'Legal', 21, 'Tribal compact disputes. Hard Rock Seminole operations.'),
            array('georgia', 'Georgia', 'Illegal', 'Illegal', 'Illegal', 'Illegal', 'Legal', 21, 'Only state lottery legal. No casino gambling permitted.'),
            array('hawaii', 'Hawaii', 'Illegal', 'Illegal', 'Illegal', 'Illegal', 'Illegal', 21, 'All forms of gambling prohibited by state constitution.'),
            array('idaho', 'Idaho', 'Illegal', 'Illegal', 'Illegal', 'Illegal', 'Legal', 21, 'Limited pari-mutuel wagering. No casino gambling.'),
            array('illinois', 'Illinois', 'Legal', 'Illegal', 'Illegal', 'Legal', 'Legal', 21, 'Sports betting launched 2020. 15% tax on revenue.'),
            array('indiana', 'Indiana', 'Legal', 'Illegal', 'Illegal', 'Legal', 'Legal', 21, 'Sports betting legal since 2019. 9.5% tax rate.'),
            array('iowa', 'Iowa', 'Legal', 'Illegal', 'Illegal', 'Legal', 'Legal', 21, 'Sports betting launched 2019. 6.75% tax on revenue.'),
            array('kansas', 'Kansas', 'Legal', 'Illegal', 'Illegal', 'Legal', 'Legal', 21, 'Sports betting launched 2022. 10% tax on revenue.'),
            array('kentucky', 'Kentucky', 'Legal', 'Illegal', 'Illegal', 'Legal', 'Legal', 21, 'Sports betting launched 2023. 14.25% tax rate.'),
            array('louisiana', 'Louisiana', 'Legal', 'Illegal', 'Illegal', 'Legal', 'Legal', 21, 'Sports betting launched 2021. Parish-by-parish approval required.'),
            array('maine', 'Maine', 'Legal', 'Illegal', 'Illegal', 'Legal', 'Legal', 21, 'Sports betting launched 2024. 10% tax on revenue.'),
            array('maryland', 'Maryland', 'Legal', 'Coming Soon', 'Coming Soon', 'Legal', 'Legal', 21, 'Online casino legislation passed 2025. 15% sports betting tax.'),
            array('massachusetts', 'Massachusetts', 'Legal', 'Coming Soon', 'Illegal', 'Legal', 'Legal', 21, 'Sports betting launched 2023. Online casino legislation proposed.'),
            array('michigan', 'Michigan', 'Legal', 'Legal', 'Legal', 'Legal', 'Legal', 21, 'Full online gambling launched 2021. 8.4% sports betting tax.'),
            array('minnesota', 'Minnesota', 'Illegal', 'Illegal', 'Illegal', 'Tribal Only', 'Legal', 18, 'Large tribal gaming market. Sports betting legislation pending.'),
            array('mississippi', 'Mississippi', 'Legal', 'Illegal', 'Illegal', 'Legal', 'Legal', 21, 'Sports betting retail only. Must be on casino premises.'),
            array('missouri', 'Missouri', 'Coming Soon', 'Illegal', 'Illegal', 'Legal', 'Legal', 21, 'Sports betting launching December 2025. 10% tax rate.'),
            array('montana', 'Montana', 'Legal', 'Illegal', 'Illegal', 'Legal', 'Illegal', 18, 'Sports betting on-premises only. 10% tax rate.'),
            array('nebraska', 'Nebraska', 'Legal', 'Illegal', 'Illegal', 'Legal', 'Legal', 21, 'Sports betting retail only. Launched 2024.'),
            array('nevada', 'Nevada', 'Legal', 'Legal', 'Legal', 'Legal', 'Legal', 21, 'Gaming capital. Lowest tax rates (6.75%). Full gambling legal.'),
            array('new_hampshire', 'New Hampshire', 'Legal', 'Illegal', 'Illegal', 'Legal', 'Legal', 21, 'Sports betting launched 2020. 50% tax on operator revenue.'),
            array('new_jersey', 'New Jersey', 'Legal', 'Legal', 'Legal', 'Legal', 'Legal', 21, 'Leader in online gambling. 8.5-13% tax rates.'),
            array('new_mexico', 'New Mexico', 'Legal', 'Illegal', 'Illegal', 'Tribal Only', 'Legal', 21, 'Sports betting tribal casinos only. Revenue sharing agreements.'),
            array('new_york', 'New York', 'Legal', 'Coming Soon', 'Illegal', 'Legal', 'Legal', 21, 'Highest sports betting tax (51%). Online casino legislation proposed.'),
            array('north_carolina', 'North Carolina', 'Legal', 'Illegal', 'Illegal', 'Tribal Only', 'Legal', 21, 'Sports betting launched 2024. 18% tax on revenue.'),
            array('north_dakota', 'North Dakota', 'Legal', 'Illegal', 'Illegal', 'Tribal Only', 'Legal', 21, 'Sports betting tribal casinos only. Limited options.'),
            array('ohio', 'Ohio', 'Legal', 'Coming Soon', 'Coming Soon', 'Legal', 'Legal', 21, 'Sports betting launched 2023. Online casino legislation proposed 2025.'),
            array('oklahoma', 'Oklahoma', 'Illegal', 'Illegal', 'Illegal', 'Tribal Only', 'Legal', 18, 'Extensive tribal gaming. Sports betting negotiations ongoing.'),
            array('oregon', 'Oregon', 'Legal', 'Illegal', 'Illegal', 'Tribal Only', 'Legal', 21, 'State-run sports betting. Tribal casinos for other gambling.'),
            array('pennsylvania', 'Pennsylvania', 'Legal', 'Legal', 'Legal', 'Legal', 'Legal', 21, 'Highest tax rates (up to 54%). Full gambling legal since 2019.'),
            array('rhode_island', 'Rhode Island', 'Legal', 'Legal', 'Legal', 'Legal', 'Legal', 21, 'Full online gambling legal. High tax rates (70%+).'),
            array('south_carolina', 'South Carolina', 'Illegal', 'Illegal', 'Illegal', 'Illegal', 'Legal', 21, 'Very restrictive gambling laws. Only lottery legal.'),
            array('south_dakota', 'South Dakota', 'Legal', 'Illegal', 'Illegal', 'Legal', 'Legal', 21, 'Sports betting in Deadwood only. Limited to historic district.'),
            array('tennessee', 'Tennessee', 'Legal', 'Illegal', 'Illegal', 'Illegal', 'Legal', 21, 'Online sports betting only. No retail sports betting or casinos.'),
            array('texas', 'Texas', 'Illegal', 'Illegal', 'Illegal', 'Tribal Only', 'Legal', 21, 'Limited tribal gaming. Sports betting legislation proposed 2025.'),
            array('utah', 'Utah', 'Illegal', 'Illegal', 'Illegal', 'Illegal', 'Illegal', 21, 'All gambling prohibited by state constitution. No exceptions.'),
            array('vermont', 'Vermont', 'Legal', 'Illegal', 'Illegal', 'Illegal', 'Legal', 21, 'Online sports betting only. Launched January 2024.'),
            array('virginia', 'Virginia', 'Legal', 'Coming Soon', 'Illegal', 'Legal', 'Legal', 21, 'Sports betting launched 2021. Online casino legislation proposed.'),
            array('washington', 'Washington', 'Legal', 'Illegal', 'Illegal', 'Tribal Only', 'Legal', 21, 'Sports betting tribal casinos only. Strict online gambling laws.'),
            array('west_virginia', 'West Virginia', 'Legal', 'Legal', 'Legal', 'Legal', 'Legal', 21, 'Full online gambling legal since 2019. 10% sports betting tax.'),
            array('wisconsin', 'Wisconsin', 'Legal', 'Illegal', 'Illegal', 'Tribal Only', 'Legal', 21, 'Sports betting tribal casinos only. Compact negotiations ongoing.'),
            array('wyoming', 'Wyoming', 'Legal', 'Illegal', 'Illegal', 'Legal', 'Legal', 21, 'Online sports betting only. Launched 2024.'),
            array('washington_dc', 'Washington D.C.', 'Legal', 'Illegal', 'Illegal', 'Legal', 'Legal', 21, 'Sports betting launched 2021. Single operator model.'),
        );
        
        foreach ($states as $state) {
            $wpdb->insert(
                $table_name,
                array(
                    'state_key' => $state[0],
                    'state_name' => $state[1],
                    'sports_betting' => $state[2],
                    'online_casino' => $state[3],
                    'online_poker' => $state[4],
                    'retail_casino' => $state[5],
                    'dfs' => $state[6],
                    'min_age' => $state[7],
                    'notes' => $state[8]
                )
            );
        }
    }
    
    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_menu_page(
            'Gambling Laws Widget',
            'Gambling Laws',
            'manage_options',
            'gambling-laws-widget',
            array($this, 'admin_page'),
            'dashicons-location-alt',
            30
        );
        
        add_submenu_page(
            'gambling-laws-widget',
            'Manage States',
            'Manage States',
            'manage_options',
            'gambling-laws-widget',
            array($this, 'admin_page')
        );
        
        add_submenu_page(
            'gambling-laws-widget',
            'Settings',
            'Settings',
            'manage_options',
            'gambling-laws-settings',
            array($this, 'settings_page')
        );
    }
    
    /**
     * Admin page
     */
    public function admin_page() {
        include GLW_PLUGIN_PATH . 'admin/admin-page.php';
    }
    
    /**
     * Settings page
     */
    public function settings_page() {
        include GLW_PLUGIN_PATH . 'admin/settings-page.php';
    }
    
    /**
     * Manual database setup (for troubleshooting)
     */
    public function manual_setup() {
        if (isset($_GET['glw_setup']) && current_user_can('manage_options')) {
            $this->create_tables();
            $this->insert_default_data();
            
            global $wpdb;
            $table_name = $wpdb->prefix . 'gambling_laws';
            $count = $wpdb->get_var("SELECT COUNT(*) FROM $table_name");
            
            wp_redirect(admin_url('admin.php?page=gambling-laws-widget&setup_complete=' . $count));
            exit;
        }
    }
    
    /**
     * Admin init
     */
    public function admin_init() {
        register_setting('gambling_laws_settings', 'gambling_laws_options');
    }
    
    /**
     * Enqueue frontend assets
     */
    public function enqueue_frontend_assets() {
        wp_enqueue_style(
            'gambling-laws-widget',
            GLW_PLUGIN_URL . 'assets/css/frontend.css',
            array(),
            GLW_VERSION
        );
        
        wp_enqueue_script(
            'gambling-laws-widget',
            GLW_PLUGIN_URL . 'assets/js/frontend.js',
            array('jquery'),
            GLW_VERSION,
            true
        );
        
        wp_localize_script('gambling-laws-widget', 'glw_ajax', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('glw_nonce')
        ));
    }
    
    /**
     * Enqueue admin assets
     */
    public function enqueue_admin_assets($hook) {
        if (strpos($hook, 'gambling-laws') === false) {
            return;
        }
        
        wp_enqueue_style(
            'gambling-laws-admin',
            GLW_PLUGIN_URL . 'assets/css/admin.css',
            array(),
            GLW_VERSION
        );
        
        wp_enqueue_script(
            'gambling-laws-admin',
            GLW_PLUGIN_URL . 'assets/js/admin.js',
            array('jquery'),
            GLW_VERSION,
            true
        );
        
        wp_localize_script('gambling-laws-admin', 'glw_admin_ajax', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('glw_admin_nonce')
        ));
    }
    
    /**
     * Shortcode handler
     */
    public function shortcode($atts) {
        $atts = shortcode_atts(array(
            'height' => 'auto',
            'style' => 'default'
        ), $atts);
        
        ob_start();
        include GLW_PLUGIN_PATH . 'templates/widget.php';
        return ob_get_clean();
    }
    
    /**
     * AJAX: Get gambling laws data
     */
    public function ajax_get_gambling_laws() {
        check_ajax_referer('glw_nonce', 'nonce');
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'gambling_laws';
        
        $states = $wpdb->get_results(
            "SELECT * FROM $table_name ORDER BY state_name ASC",
            ARRAY_A
        );
        
        wp_send_json_success(array(
            'states' => $states,
            'last_updated' => current_time('F Y')
        ));
    }
    
    /**
     * AJAX: Save gambling laws
     */
    public function ajax_save_gambling_laws() {
        check_ajax_referer('glw_admin_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        $state_data = $_POST['state_data'];
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'gambling_laws';
        
        $result = $wpdb->update(
            $table_name,
            array(
                'state_name' => sanitize_text_field($state_data['state_name']),
                'sports_betting' => sanitize_text_field($state_data['sports_betting']),
                'online_casino' => sanitize_text_field($state_data['online_casino']),
                'online_poker' => sanitize_text_field($state_data['online_poker']),
                'retail_casino' => sanitize_text_field($state_data['retail_casino']),
                'dfs' => sanitize_text_field($state_data['dfs']),
                'min_age' => intval($state_data['min_age']),
                'notes' => sanitize_textarea_field($state_data['notes'])
            ),
            array('state_key' => sanitize_text_field($state_data['state_key']))
        );
        
        if ($result !== false) {
            wp_send_json_success('State updated successfully');
        } else {
            wp_send_json_error('Failed to update state');
        }
    }
    
    /**
     * AJAX: Import gambling laws
     */
    public function ajax_import_gambling_laws() {
        check_ajax_referer('glw_admin_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        // Handle file upload and import logic here
        wp_send_json_success('Import completed successfully');
    }
}

// Initialize the plugin
new GamblingLawsWidget();