<?php
/**
 * Widget Template for Gambling Laws Widget
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

$options = get_option('gambling_laws_options', array(
    'widget_title' => 'U.S. State Gambling Laws Directory',
    'widget_subtitle' => 'Comprehensive guide to gambling and sports betting laws across all 50 states + Washington D.C.',
    'show_icons' => 1,
    'show_last_updated' => 1
));

$widget_id = 'glw-widget-' . wp_rand(1000, 9999);
?>

<div id="<?php echo $widget_id; ?>" class="gambling-laws-widget-wp">
    <!-- Header -->
    <div class="glw-header-section">
        <div class="glw-container">
            <div class="glw-main-title">
                🎰 <?php echo esc_html($options['widget_title']); ?>
            </div>
            <p class="glw-subtitle">
                <?php echo esc_html($options['widget_subtitle']); ?>
            </p>
        </div>
    </div>

    <!-- State Selector -->
    <div class="glw-selector-section">
        <div class="glw-container">
            <div class="glw-selector-card">
                <label for="glw-state-select-<?php echo $widget_id; ?>" class="glw-selector-label">
                    📍 Select a U.S. State or Territory
                </label>
                <select id="glw-state-select-<?php echo $widget_id; ?>" class="glw-state-dropdown">
                    <option value="">Choose a state...</option>
                </select>
            </div>
        </div>
    </div>

    <!-- Loading State -->
    <div class="glw-loading-section" style="display: none;">
        <div class="glw-container">
            <div class="glw-loading-spinner">
                <div class="glw-spinner"></div>
                <p>Loading gambling laws data...</p>
            </div>
        </div>
    </div>

    <!-- Results Panel -->
    <div class="glw-results-section" style="display: none;">
        <div class="glw-container">
            <div class="glw-results-card">
                <div class="glw-state-header">
                    <div class="glw-state-name"></div>
                    <div class="glw-age-badge">
                        <span class="glw-age-icon">🧑‍⚖️</span>
                        Minimum Age: <strong class="glw-min-age"></strong>
                    </div>
                </div>

                <div class="glw-laws-grid">
                    <div class="glw-law-item">
                        <div class="glw-law-header">
                            <div class="glw-law-title">Online Sports Betting</div>
                        </div>
                        <span class="glw-status-badge glw-sports-betting"></span>
                    </div>

                    <div class="glw-law-item">
                        <div class="glw-law-header">
                            <div class="glw-law-title">Online Casino Gambling</div>
                        </div>
                        <span class="glw-status-badge glw-online-casino"></span>
                    </div>

                    <div class="glw-law-item">
                        <div class="glw-law-header">
                            <div class="glw-law-title">Online Poker</div>
                        </div>
                        <span class="glw-status-badge glw-online-poker"></span>
                    </div>

                    <div class="glw-law-item">
                        <div class="glw-law-header">
                            <div class="glw-law-title">Retail Casinos</div>
                        </div>
                        <span class="glw-status-badge glw-retail-casino"></span>
                    </div>

                    <div class="glw-law-item">
                        <div class="glw-law-header">
                            <div class="glw-law-title">Daily Fantasy Sports (DFS)</div>
                        </div>
                        <span class="glw-status-badge glw-dfs"></span>
                    </div>
                </div>

                <div class="glw-notes-section" style="display: none;">
                    <div class="glw-notes-title">
                        💰 Tax & Licensing Information
                    </div>
                    <p class="glw-notes-text"></p>
                </div>

                <?php if ($options['show_last_updated']): ?>
                <div class="glw-footer-info">
                    <p class="glw-last-updated">
                        <span class="glw-update-icon">🕐</span>
                        Last Updated: <span class="glw-update-date"></span>
                    </p>
                    <p class="glw-disclaimer">
                        <span class="glw-warning-icon">⚠️</span>
                        Laws subject to change. Consult state gaming authorities for latest regulations.
                    </p>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Call to Action -->
    <div class="glw-cta-section">
        <div class="glw-container">
            <div class="glw-cta-card">
                <div class="glw-cta-title">Stay Informed About Gambling Laws</div>
                <p class="glw-cta-text">
                    Select any state above to view comprehensive gambling regulations, tax implications, and legal status updates.
                </p>
                <div class="glw-feature-list">
                    <div class="glw-feature-item">✅ All 50 States + Washington D.C.</div>
                    <div class="glw-feature-item">🎯 Sports Betting & Casino Laws</div>
                    <div class="glw-feature-item">📱 Mobile-Friendly Interface</div>
                    <div class="glw-feature-item">🔄 Regularly Updated Data</div>
                </div>
            </div>
        </div>
    </div>
</div>

<script type="text/javascript">
jQuery(document).ready(function($) {
    var widgetId = '<?php echo $widget_id; ?>';
    var widget = $('#' + widgetId);
    var showIcons = <?php echo $options['show_icons'] ? 'true' : 'false'; ?>;
    
    // Load states on page load
    loadStates();
    
    function loadStates() {
        $.ajax({
            url: glw_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'get_gambling_laws',
                nonce: glw_ajax.nonce
            },
            success: function(response) {
                if (response.success) {
                    populateStatesDropdown(response.data.states);
                    widget.find('.glw-update-date').text(response.data.last_updated);
                }
            },
            error: function() {
                console.error('Failed to load gambling laws data');
            }
        });
    }
    
    function populateStatesDropdown(states) {
        var dropdown = widget.find('.glw-state-dropdown');
        dropdown.empty().append('<option value="">Choose a state...</option>');
        
        states.forEach(function(state) {
            dropdown.append(
                '<option value="' + state.state_key + '">' + state.state_name + '</option>'
            );
        });
        
        // Handle state selection
        dropdown.on('change', function() {
            var selectedStateKey = $(this).val();
            if (selectedStateKey) {
                var selectedState = states.find(function(state) {
                    return state.state_key === selectedStateKey;
                });
                if (selectedState) {
                    displayStateData(selectedState);
                }
            } else {
                hideResults();
            }
        });
    }
    
    function displayStateData(state) {
        // Update state header
        widget.find('.glw-state-name').text(state.state_name + ' Gambling Laws');
        widget.find('.glw-min-age').text(state.min_age);
        
        // Update status badges
        updateStatusBadge('.glw-sports-betting', state.sports_betting, 'sports');
        updateStatusBadge('.glw-online-casino', state.online_casino, 'casino');
        updateStatusBadge('.glw-online-poker', state.online_poker, 'poker');
        updateStatusBadge('.glw-retail-casino', state.retail_casino, 'retail');
        updateStatusBadge('.glw-dfs', state.dfs, 'dfs');
        
        // Update notes
        if (state.notes && state.notes.trim()) {
            widget.find('.glw-notes-text').text(state.notes);
            widget.find('.glw-notes-section').show();
        } else {
            widget.find('.glw-notes-section').hide();
        }
        
        // Show results and hide CTA
        widget.find('.glw-results-section').show();
        widget.find('.glw-cta-section').hide();
    }
    
    function updateStatusBadge(selector, status, type) {
        var badge = widget.find(selector);
        badge.removeClass('glw-legal glw-illegal glw-coming-soon glw-tribal-only glw-grey-market');
        
        var className = '';
        var icon = '';
        
        switch (status.toLowerCase()) {
            case 'legal':
                className = 'glw-legal';
                if (showIcons) {
                    switch (type) {
                        case 'sports': icon = '✅ '; break;
                        case 'casino': icon = '🎰 '; break;
                        case 'poker': icon = '🃏 '; break;
                        case 'retail': icon = '🏢 '; break;
                        case 'dfs': icon = '🎯 '; break;
                        default: icon = '✅ '; break;
                    }
                }
                break;
            case 'illegal':
            case 'not legal':
                className = 'glw-illegal';
                if (showIcons) icon = '❌ ';
                break;
            case 'coming soon':
                className = 'glw-coming-soon';
                if (showIcons) icon = '⚠️ ';
                break;
            case 'tribal only':
                className = 'glw-tribal-only';
                if (showIcons) icon = '⚠️ ';
                break;
            case 'grey market':
                className = 'glw-grey-market';
                if (showIcons) icon = '⚠️ ';
                break;
            default:
                className = 'glw-illegal';
                if (showIcons) icon = '❌ ';
        }
        
        badge.addClass(className).html(icon + status);
    }
    
    function hideResults() {
        widget.find('.glw-results-section').hide();
        widget.find('.glw-cta-section').show();
    }
});
</script>