// Gutenberg Block Editor JavaScript for Geo Ranking Tables
/* global wp, grtBlockData */
(function() {
    'use strict';
    
    var __ = wp.i18n.__;
    var registerBlockType = wp.blocks.registerBlockType;
    var createElement = wp.element.createElement;
    var Fragment = wp.element.Fragment;
    var InspectorControls = wp.blockEditor.InspectorControls;
    var PanelBody = wp.components.PanelBody;
    var CheckboxControl = wp.components.CheckboxControl;
    var RangeControl = wp.components.RangeControl;
    var TextControl = wp.components.TextControl;
    var ToggleControl = wp.components.ToggleControl;
    
    registerBlockType('geo-ranking-tables/ranking-table', {
        title: __('Geo Ranking Table', 'geo-ranking-tables'),
        description: __('Display a geo-filtered ranking table of gambling sites', 'geo-ranking-tables'),
        icon: 'location-alt',
        category: 'widgets',
        keywords: [
            __('ranking', 'geo-ranking-tables'),
            __('gambling', 'geo-ranking-tables'),
            __('geo', 'geo-ranking-tables'),
            __('table', 'geo-ranking-tables')
        ],
        
        attributes: {
            useManaged: {
                type: 'boolean',
                default: false
            },
            managedTableId: {
                type: 'number',
                default: 0
            },
            overrideSettings: {
                type: 'object',
                default: {}
            },
            selectedSites: {
                type: 'array',
                default: []
            },
            maxDisplay: {
                type: 'number',
                default: 5
            },
            fallbackBlockId: {
                type: 'number',
                default: 0
            },
            tableName: {
                type: 'string',
                default: __('Top Gambling Sites', 'geo-ranking-tables')
            },
            showRatings: {
                type: 'boolean',
                default: true
            },
            showBonuses: {
                type: 'boolean',
                default: true
            },
            showLicenses: {
                type: 'boolean',
                default: false
            },
            showDebugInfo: {
                type: 'boolean',
                default: false
            }
        },
        
        edit: function(props) {
            var attributes = props.attributes;
            var setAttributes = props.setAttributes;
            
            function onSiteChange(siteId, checked) {
                var selectedSites = attributes.selectedSites ? attributes.selectedSites.slice() : [];
                
                if (checked) {
                    if (selectedSites.indexOf(siteId) === -1) {
                        selectedSites.push(siteId);
                    }
                } else {
                    var index = selectedSites.indexOf(siteId);
                    if (index > -1) {
                        selectedSites.splice(index, 1);
                    }
                }
                
                setAttributes({ selectedSites: selectedSites });
            }
            
            var siteBlocks = (typeof grtBlockData !== 'undefined' && grtBlockData.siteBlocks) ? grtBlockData.siteBlocks : [];
            var managedTables = (typeof grtBlockData !== 'undefined' && grtBlockData.managedTables) ? grtBlockData.managedTables : [];
            var selectedCount = attributes.selectedSites ? attributes.selectedSites.length : 0;
            var useManaged = attributes.useManaged || false;
            var managedTableId = attributes.managedTableId || 0;
            var hasOverrides = attributes.overrideSettings && Object.keys(attributes.overrideSettings).length > 0;
            
            return createElement(Fragment, {},
                createElement(InspectorControls, {},
                    createElement(PanelBody, {
                        title: __('Table Mode', 'geo-ranking-tables'),
                        initialOpen: true
                    },
                        createElement(ToggleControl, {
                            label: __('Use Managed Table', 'geo-ranking-tables'),
                            help: useManaged ? __('Using centrally managed table configuration', 'geo-ranking-tables') : __('Using custom configuration for this block', 'geo-ranking-tables'),
                            checked: useManaged,
                            onChange: function(value) {
                                setAttributes({ useManaged: value });
                            }
                        }),
                        
                        useManaged && managedTables.length > 0 ? createElement('div', {
                            style: { marginTop: '15px' }
                        },
                            createElement('label', {
                                style: { display: 'block', marginBottom: '8px', fontWeight: '600' }
                            }, __('Select Managed Table', 'geo-ranking-tables')),
                            createElement('select', {
                                value: managedTableId,
                                onChange: function(e) {
                                    setAttributes({ managedTableId: parseInt(e.target.value) });
                                },
                                style: { width: '100%', padding: '6px', borderRadius: '4px', border: '1px solid #ddd' }
                            },
                                createElement('option', { value: 0 }, __('-- Select a table --', 'geo-ranking-tables')),
                                managedTables.map(function(table) {
                                    return createElement('option', {
                                        key: table.id,
                                        value: table.id
                                    }, table.title + ' (' + table.sites_count + ' sites)');
                                })
                            )
                        ) : null,
                        
                        useManaged && managedTables.length === 0 ? createElement('div', {
                            style: { marginTop: '15px', padding: '12px', background: '#fff3cd', borderRadius: '4px', fontSize: '13px' }
                        }, __('No managed tables found. Create one in Geo Rankings > Managed Tables.', 'geo-ranking-tables')) : null,
                        
                        useManaged && managedTableId > 0 ? createElement('div', {
                            style: { marginTop: '15px', padding: '12px', background: '#e7f3ff', borderRadius: '4px', fontSize: '13px' }
                        },
                            createElement('strong', {}, '✓ ' + __('Connected to managed table', 'geo-ranking-tables')),
                            createElement('p', { style: { margin: '8px 0 0 0' } }, __('Changes to the managed table will automatically appear here.', 'geo-ranking-tables'))
                        ) : null
                    ),
                    
                    !useManaged ? createElement(PanelBody, {
                        title: __('Table Settings', 'geo-ranking-tables'),
                        initialOpen: true
                    },
                        createElement(TextControl, {
                            label: __('Table Name', 'geo-ranking-tables'),
                            value: attributes.tableName || '',
                            onChange: function(value) {
                                setAttributes({ tableName: value });
                            }
                        }),
                        
                        createElement(RangeControl, {
                            label: __('Max Sites to Display', 'geo-ranking-tables'),
                            value: attributes.maxDisplay || 5,
                            onChange: function(value) {
                                setAttributes({ maxDisplay: value });
                            },
                            min: 1,
                            max: 20
                        })
                    ) : null,
                    
                    !useManaged ? createElement(PanelBody, {
                        title: __('Display Options', 'geo-ranking-tables'),
                        initialOpen: false
                    },
                        createElement(ToggleControl, {
                            label: __('Show Star Ratings', 'geo-ranking-tables'),
                            checked: attributes.showRatings !== false,
                            onChange: function(value) {
                                setAttributes({ showRatings: value });
                            }
                        }),
                        
                        createElement(ToggleControl, {
                            label: __('Show Bonus Information', 'geo-ranking-tables'),
                            checked: attributes.showBonuses !== false,
                            onChange: function(value) {
                                setAttributes({ showBonuses: value });
                            }
                        }),
                        
                        createElement(ToggleControl, {
                            label: __('Show License Information', 'geo-ranking-tables'),
                            checked: attributes.showLicenses === true,
                            onChange: function(value) {
                                setAttributes({ showLicenses: value });
                            }
                        })
                    ) : null,
                    
                    !useManaged ? createElement(PanelBody, {
                        title: __('Developer Options', 'geo-ranking-tables'),
                        initialOpen: false
                    },
                        createElement(ToggleControl, {
                            label: __('Show Debug Information', 'geo-ranking-tables'),
                            help: __('Display detailed debug information on the frontend (for development only)', 'geo-ranking-tables'),
                            checked: attributes.showDebugInfo === true,
                            onChange: function(value) {
                                setAttributes({ showDebugInfo: value });
                            }
                        })
                    ) : null,
                    
                    !useManaged ? createElement(PanelBody, {
                        title: __('Site Selection', 'geo-ranking-tables') + ' (' + selectedCount + ' ' + __('selected', 'geo-ranking-tables') + ')',
                        initialOpen: true
                    },
                        siteBlocks.length === 0 ? 
                            createElement('p', { 
                                style: { color: '#666', fontStyle: 'italic' },
                                className: 'grt-no-sites'
                            }, __('No site blocks found. Please create some site blocks first.', 'geo-ranking-tables')
                            ) :
                            siteBlocks.map(function(site) {
                                return createElement(CheckboxControl, {
                                    key: site.id,
                                    label: site.site_name || site.title,
                                    help: site.description ? site.description.substring(0, 100) + '...' : '',
                                    checked: attributes.selectedSites ? attributes.selectedSites.indexOf(site.id) !== -1 : false,
                                    onChange: function(checked) {
                                        onSiteChange(site.id, checked);
                                    }
                                });
                            })
                    ) : null
                ),
                
                // Block preview in editor
                createElement('div', {
                    className: 'grt-block-preview'
                },
                    useManaged && managedTableId > 0 ? createElement('div', {
                        style: { marginBottom: '15px', padding: '12px', background: '#e7f3ff', borderLeft: '4px solid #0073aa', borderRadius: '4px' }
                    },
                        createElement('div', {
                            style: { display: 'flex', alignItems: 'center', gap: '8px' }
                        },
                            createElement('span', { className: 'dashicons dashicons-editor-table', style: { color: '#0073aa' } }),
                            createElement('strong', { style: { color: '#0073aa' } }, __('Managed Table Mode', 'geo-ranking-tables'))
                        ),
                        createElement('p', { style: { margin: '8px 0 0 0', fontSize: '13px' } }, __('This block is linked to a centrally managed table. Changes will propagate automatically.', 'geo-ranking-tables'))
                    ) : null,
                    
                    createElement('h3', {
                        style: { margin: '0 0 10px 0', color: '#333' }
                    }, attributes.tableName || __('Top Gambling Sites', 'geo-ranking-tables')),
                    
                    createElement('div', {
                        style: {
                            display: 'flex',
                            alignItems: 'center',
                            justifyContent: 'center',
                            gap: '8px',
                            marginBottom: '15px'
                        }
                    },
                        createElement('span', {
                            className: 'dashicons dashicons-location-alt',
                            style: { color: '#666' }
                        }),
                        createElement('span', { 
                            style: { color: '#666' }
                        }, __('Geo-filtered ranking table', 'geo-ranking-tables'))
                    ),
                    
                    useManaged && managedTableId === 0 ? 
                        createElement('p', {
                            style: { color: '#d63638', fontWeight: 'bold' }
                        }, __('Please select a managed table from the settings panel →', 'geo-ranking-tables')) :
                        
                    !useManaged && selectedCount === 0 ? 
                        createElement('p', {
                            style: { color: '#d63638', fontWeight: 'bold' }
                        }, __('Please select at least one site block from the settings panel →', 'geo-ranking-tables')) :
                        
                        createElement('div', {},
                            createElement('p', {
                                style: { color: '#135e96', fontWeight: 'bold', margin: '0 0 8px 0' }
                            }, selectedCount + ' ' + __('site(s) selected', 'geo-ranking-tables')),
                            
                            createElement('p', {
                                style: { color: '#666', fontSize: '14px', margin: '0' }
                            }, __('Showing up to', 'geo-ranking-tables') + ' ' + (attributes.maxDisplay || 5) + ' ' + __('sites based on user location', 'geo-ranking-tables')),
                            
                            createElement('div', {
                                style: {
                                    marginTop: '15px',
                                    padding: '10px',
                                    backgroundColor: '#e7f3ff',
                                    borderRadius: '4px',
                                    fontSize: '12px',
                                    color: '#0073aa'
                                }
                            }, __('This table will automatically filter and display gambling sites available in the visitor\'s location.', 'geo-ranking-tables')),
                            
                            // Debug info indicator
                            attributes.showDebugInfo ? createElement('div', {
                                style: {
                                    marginTop: '10px',
                                    padding: '8px',
                                    backgroundColor: '#fff3cd',
                                    border: '1px solid #ffeaa7',
                                    borderRadius: '4px',
                                    fontSize: '11px',
                                    color: '#856404'
                                }
                            }, '🔍 Debug information will be shown on frontend') : null
                        )
                )
            );
        },
        
        save: function() {
            // Dynamic block - rendered by PHP
            return null;
        }
    });
})();