<?php
/**
 * Exporter Class
 * Handles exporting site blocks and managed tables to JSON
 */

if (!defined('ABSPATH')) {
    exit;
}

class GRT_Exporter {

    private $export_type = 'full';
    private $selected_sites = array();
    private $selected_tables = array();

    /**
     * Set export options
     */
    public function set_options($options) {
        if (isset($options['export_type'])) {
            $this->export_type = sanitize_text_field($options['export_type']);
        }
        if (isset($options['selected_sites'])) {
            $this->selected_sites = array_map('intval', $options['selected_sites']);
        }
        if (isset($options['selected_tables'])) {
            $this->selected_tables = array_map('intval', $options['selected_tables']);
        }
    }

    /**
     * Generate complete export data
     */
    public function generate_export_data() {
        $data = array(
            'plugin_version' => GRT_VERSION,
            'export_date' => current_time('c'),
            'site_url' => get_site_url(),
            'export_type' => $this->export_type,
            'site_blocks' => array(),
            'managed_tables' => array(),
        );

        // Export site blocks
        if ($this->export_type === 'full' || $this->export_type === 'sites_only' || $this->export_type === 'selected') {
            $data['site_blocks'] = $this->export_site_blocks();
        }

        // Export managed tables
        if ($this->export_type === 'full' || $this->export_type === 'tables_only' || $this->export_type === 'selected') {
            $data['managed_tables'] = $this->export_managed_tables();
        }

        // Validate we have something to export
        if (empty($data['site_blocks']) && empty($data['managed_tables'])) {
            return new WP_Error('no_data', __('No data to export. Please select items or ensure data exists.', 'geo-ranking-tables'));
        }

        return $data;
    }

    /**
     * Export site blocks
     */
    private function export_site_blocks() {
        $args = array(
            'post_type' => 'grt_site_block',
            'post_status' => 'publish',
            'numberposts' => -1,
            'orderby' => 'ID',
            'order' => 'ASC',
        );

        // If exporting selected items, filter by IDs
        if ($this->export_type === 'selected' && !empty($this->selected_sites)) {
            $args['post__in'] = $this->selected_sites;
        } elseif ($this->export_type === 'selected' && empty($this->selected_sites)) {
            return array(); // No sites selected
        }

        $posts = get_posts($args);
        $site_blocks = array();

        foreach ($posts as $post) {
            $site_blocks[] = array(
                'export_id' => $post->ID,
                'post_data' => array(
                    'post_title' => $post->post_title,
                    'post_name' => $post->post_name,
                    'post_content' => $post->post_content,
                    'post_status' => $post->post_status,
                    'post_excerpt' => $post->post_excerpt,
                ),
                'meta' => array(
                    '_grt_site_name' => get_post_meta($post->ID, '_grt_site_name', true),
                    '_grt_site_description' => get_post_meta($post->ID, '_grt_site_description', true),
                    '_grt_star_rating' => get_post_meta($post->ID, '_grt_star_rating', true),
                    '_grt_affiliate_link' => get_post_meta($post->ID, '_grt_affiliate_link', true),
                    '_grt_bonus_info' => get_post_meta($post->ID, '_grt_bonus_info', true),
                    '_grt_license_info' => get_post_meta($post->ID, '_grt_license_info', true),
                    '_grt_states_accepted' => get_post_meta($post->ID, '_grt_states_accepted', true),
                ),
                'featured_image' => $this->get_featured_image_data($post->ID),
            );
        }

        return $site_blocks;
    }

    /**
     * Export managed tables
     */
    private function export_managed_tables() {
        $args = array(
            'post_type' => 'grt_managed_table',
            'post_status' => 'publish',
            'numberposts' => -1,
            'orderby' => 'ID',
            'order' => 'ASC',
        );

        // If exporting selected items, filter by IDs
        if ($this->export_type === 'selected' && !empty($this->selected_tables)) {
            $args['post__in'] = $this->selected_tables;
        } elseif ($this->export_type === 'selected' && empty($this->selected_tables)) {
            return array(); // No tables selected
        }

        $posts = get_posts($args);
        $managed_tables = array();

        foreach ($posts as $post) {
            $managed_tables[] = array(
                'export_id' => $post->ID,
                'post_data' => array(
                    'post_title' => $post->post_title,
                    'post_name' => $post->post_name,
                    'post_status' => $post->post_status,
                ),
                'meta' => array(
                    '_grt_managed_sites' => get_post_meta($post->ID, '_grt_managed_sites', true),
                    '_grt_managed_max_display' => get_post_meta($post->ID, '_grt_managed_max_display', true),
                    '_grt_managed_table_name' => get_post_meta($post->ID, '_grt_managed_table_name', true),
                    '_grt_managed_show_ratings' => get_post_meta($post->ID, '_grt_managed_show_ratings', true),
                    '_grt_managed_show_bonuses' => get_post_meta($post->ID, '_grt_managed_show_bonuses', true),
                    '_grt_managed_show_licenses' => get_post_meta($post->ID, '_grt_managed_show_licenses', true),
                    '_grt_managed_show_debug' => get_post_meta($post->ID, '_grt_managed_show_debug', true),
                    '_grt_managed_slug' => get_post_meta($post->ID, '_grt_managed_slug', true),
                ),
            );
        }

        return $managed_tables;
    }

    /**
     * Get featured image data for a post
     */
    private function get_featured_image_data($post_id) {
        $thumbnail_id = get_post_thumbnail_id($post_id);

        if (!$thumbnail_id) {
            return null;
        }

        $image_url = wp_get_attachment_url($thumbnail_id);
        $image_meta = wp_get_attachment_metadata($thumbnail_id);
        $attachment = get_post($thumbnail_id);

        if (!$image_url) {
            return null;
        }

        return array(
            'url' => $image_url,
            'filename' => basename($image_url),
            'alt' => get_post_meta($thumbnail_id, '_wp_attachment_image_alt', true),
            'title' => $attachment ? $attachment->post_title : '',
            'caption' => $attachment ? $attachment->post_excerpt : '',
        );
    }

    /**
     * Generate export filename
     */
    public function generate_filename() {
        $prefix = 'grt-export';

        switch ($this->export_type) {
            case 'sites_only':
                $prefix .= '-sites';
                break;
            case 'tables_only':
                $prefix .= '-tables';
                break;
            case 'selected':
                $prefix .= '-selected';
                break;
            default:
                $prefix .= '-full';
        }

        return $prefix . '-' . date('Y-m-d-His') . '.json';
    }
}
