<?php
/**
 * Managed Tables Class
 * Handles centralized ranking table configurations
 */

if (!defined('ABSPATH')) {
    exit;
}

class GRT_ManagedTables {
    
    public function __construct() {
        // Post type is now registered in main plugin file
        add_action('add_meta_boxes', array($this, 'add_meta_boxes'));
        add_action('save_post_grt_managed_table', array($this, 'save_meta_boxes'), 10, 1);
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        add_filter('post_updated_messages', array($this, 'updated_messages'));
        add_filter('manage_grt_managed_table_posts_columns', array($this, 'add_custom_columns'));
        add_action('manage_grt_managed_table_posts_custom_column', array($this, 'custom_column_content'), 10, 2);

        // Disable ACF on managed tables post type
        add_filter('acf/get_field_groups', array($this, 'remove_acf_field_groups'), 10, 2);
    }
    
    /**
     * Disable ACF admin interface for managed tables post type
     */
    public function disable_acf_on_managed_tables($show) {
        global $post_type;
        
        if ($post_type === 'grt_managed_table') {
            return false;
        }
        
        // Also check current screen
        if (function_exists('get_current_screen')) {
            $screen = get_current_screen();
            if ($screen && $screen->post_type === 'grt_managed_table') {
                return false;
            }
        }
        
        return $show;
    }
    
    /**
     * Remove ACF field groups from managed tables
     */
    public function remove_acf_field_groups($field_groups, $args) {
        // Check if we're on a managed table edit screen (GET request)
        if (isset($_GET['post']) && get_post_type($_GET['post']) === 'grt_managed_table') {
            return array();
        }

        if (isset($_GET['post_type']) && $_GET['post_type'] === 'grt_managed_table') {
            return array();
        }

        // Check during POST/save operations
        if (isset($_POST['post_ID']) && get_post_type($_POST['post_ID']) === 'grt_managed_table') {
            return array();
        }

        if (isset($_POST['post_type']) && $_POST['post_type'] === 'grt_managed_table') {
            return array();
        }

        global $post_type;
        if ($post_type === 'grt_managed_table') {
            return array();
        }

        return $field_groups;
    }
    
    /**
     * Add meta boxes for managed table configuration
     */
    public function add_meta_boxes() {
        add_meta_box(
            'grt_managed_table_config',
            __('Table Configuration', 'geo-ranking-tables'),
            array($this, 'table_config_callback'),
            'grt_managed_table',
            'normal',
            'high'
        );
        
        add_meta_box(
            'grt_managed_table_info',
            __('Table Information', 'geo-ranking-tables'),
            array($this, 'table_info_callback'),
            'grt_managed_table',
            'side',
            'default'
        );
    }
    
    /**
     * Table configuration meta box callback
     */
    public function table_config_callback($post) {
        wp_nonce_field('grt_managed_table_meta', 'grt_managed_table_nonce');
        
        // Get current values
        $managed_sites = get_post_meta($post->ID, '_grt_managed_sites', true);
        $max_display = get_post_meta($post->ID, '_grt_managed_max_display', true);
        $table_name = get_post_meta($post->ID, '_grt_managed_table_name', true);
        $show_ratings = get_post_meta($post->ID, '_grt_managed_show_ratings', true);
        $show_bonuses = get_post_meta($post->ID, '_grt_managed_show_bonuses', true);
        $show_licenses = get_post_meta($post->ID, '_grt_managed_show_licenses', true);
        $show_debug = get_post_meta($post->ID, '_grt_managed_show_debug', true);
        $fallback_table_id = get_post_meta($post->ID, '_grt_fallback_table_id', true);

        // Set defaults
        if (!is_array($managed_sites)) {
            $managed_sites = array();
        }
        if (empty($max_display)) {
            $max_display = 5;
        }
        if (empty($table_name)) {
            $table_name = $post->post_title;
        }
        if ($show_ratings === '') {
            $show_ratings = '1';
        }
        if ($show_bonuses === '') {
            $show_bonuses = '1';
        }
        if (empty($fallback_table_id)) {
            $fallback_table_id = 0;
        }

        // Get all site blocks (limit to 100 to prevent memory issues)
        $site_blocks = get_posts(array(
            'post_type' => 'grt_site_block',
            'post_status' => 'publish',
            'numberposts' => 100,
            'orderby' => 'title',
            'order' => 'ASC',
            'no_found_rows' => true, // Skip pagination counting for performance
            'update_post_meta_cache' => true, // We need meta data
            'update_post_term_cache' => false, // We don't need terms
        ));

        ?>
        <div class="grt-managed-table-config">
            <?php if (empty($site_blocks)): ?>
                <div class="notice notice-warning inline">
                    <p>
                        <strong><?php _e('No site blocks found.', 'geo-ranking-tables'); ?></strong>
                        <?php _e('Please create some gambling site blocks first before creating managed tables.', 'geo-ranking-tables'); ?>
                        <a href="<?php echo admin_url('post-new.php?post_type=grt_site_block'); ?>" class="button">
                            <?php _e('Add Site Block', 'geo-ranking-tables'); ?>
                        </a>
                    </p>
                </div>
            <?php endif; ?>
            
            <table class="form-table">
                <tr>
                    <th scope="row">
                        <label for="grt_table_name"><?php _e('Display Name', 'geo-ranking-tables'); ?></label>
                    </th>
                    <td>
                        <input type="text" id="grt_table_name" name="grt_table_name" value="<?php echo esc_attr($table_name); ?>" class="regular-text" />
                        <p class="description"><?php _e('The name shown on the frontend. Leave empty to use the title.', 'geo-ranking-tables'); ?></p>
                    </td>
                </tr>
                
                <tr>
                    <th scope="row">
                        <label for="grt_max_display"><?php _e('Max Sites to Display', 'geo-ranking-tables'); ?></label>
                    </th>
                    <td>
                        <input type="number" id="grt_max_display" name="grt_max_display" value="<?php echo esc_attr($max_display); ?>" min="1" max="20" />
                        <p class="description"><?php _e('Maximum number of sites to show (1-20)', 'geo-ranking-tables'); ?></p>
                    </td>
                </tr>
                
                <tr>
                    <th scope="row">
                        <?php _e('Display Options', 'geo-ranking-tables'); ?>
                    </th>
                    <td>
                        <fieldset>
                            <label>
                                <input type="checkbox" name="grt_show_ratings" value="1" <?php checked($show_ratings, '1'); ?> />
                                <?php _e('Show Star Ratings', 'geo-ranking-tables'); ?>
                            </label><br>
                            
                            <label>
                                <input type="checkbox" name="grt_show_bonuses" value="1" <?php checked($show_bonuses, '1'); ?> />
                                <?php _e('Show Bonus Information', 'geo-ranking-tables'); ?>
                            </label><br>
                            
                            <label>
                                <input type="checkbox" name="grt_show_licenses" value="1" <?php checked($show_licenses, '1'); ?> />
                                <?php _e('Show License Information', 'geo-ranking-tables'); ?>
                            </label><br>
                            
                            <label>
                                <input type="checkbox" name="grt_show_debug" value="1" <?php checked($show_debug, '1'); ?> />
                                <?php _e('Show Debug Information', 'geo-ranking-tables'); ?>
                            </label>
                        </fieldset>
                    </td>
                </tr>

                <tr>
                    <th scope="row">
                        <label for="grt_fallback_table"><?php _e('Fallback Table', 'geo-ranking-tables'); ?></label>
                    </th>
                    <td>
                        <?php
                        // Get all managed tables except the current one
                        $other_tables = get_posts(array(
                            'post_type' => 'grt_managed_table',
                            'post_status' => 'publish',
                            'numberposts' => 100,
                            'post__not_in' => array($post->ID),
                            'orderby' => 'title',
                            'order' => 'ASC',
                        ));
                        ?>
                        <select id="grt_fallback_table" name="grt_fallback_table">
                            <option value="0"><?php _e('-- No Fallback --', 'geo-ranking-tables'); ?></option>
                            <?php foreach ($other_tables as $table): ?>
                                <option value="<?php echo esc_attr($table->ID); ?>" <?php selected($fallback_table_id, $table->ID); ?>>
                                    <?php echo esc_html($table->post_title); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                        <p class="description">
                            <?php _e('Sites from the fallback table will be shown (without geo-filtering) when no sites match the user\'s location.', 'geo-ranking-tables'); ?>
                        </p>
                    </td>
                </tr>
            </table>
            
            <h3><?php _e('Select Sites', 'geo-ranking-tables'); ?></h3>
            <p class="description"><?php _e('Choose which gambling sites to include in this managed table. They will be automatically filtered by user location.', 'geo-ranking-tables'); ?></p>
            
            <div class="grt-site-selection">
                <?php if (empty($site_blocks)): ?>
                    <p class="grt-no-sites-message">
                        <?php _e('No site blocks found. Please create some site blocks first.', 'geo-ranking-tables'); ?>
                        <a href="<?php echo admin_url('post-new.php?post_type=grt_site_block'); ?>" class="button">
                            <?php _e('Add Site Block', 'geo-ranking-tables'); ?>
                        </a>
                    </p>
                <?php else: ?>
                    <div class="grt-select-all-controls" style="margin-bottom: 15px;">
                        <button type="button" id="grt-select-all-sites" class="button"><?php _e('Select All', 'geo-ranking-tables'); ?></button>
                        <button type="button" id="grt-deselect-all-sites" class="button"><?php _e('Deselect All', 'geo-ranking-tables'); ?></button>
                    </div>
                    
                    <div class="grt-sites-list">
                        <?php foreach ($site_blocks as $site): ?>
                            <?php
                            $site_name = get_post_meta($site->ID, '_grt_site_name', true);
                            $rating = get_post_meta($site->ID, '_grt_star_rating', true);
                            $states = get_post_meta($site->ID, '_grt_states_accepted', true);
                            $is_selected = in_array($site->ID, $managed_sites);
                            ?>
                            <div class="grt-site-item <?php echo $is_selected ? 'selected' : ''; ?>">
                                <label>
                                    <input type="checkbox" name="grt_managed_sites[]" value="<?php echo esc_attr($site->ID); ?>" <?php checked($is_selected); ?> />
                                    <div class="grt-site-item-info">
                                        <strong class="grt-site-item-name"><?php echo esc_html($site_name ?: $site->post_title); ?></strong>
                                        <div class="grt-site-item-meta">
                                            <?php if ($rating): ?>
                                                <span class="grt-rating">⭐ <?php echo esc_html($rating); ?>/5</span>
                                            <?php endif; ?>
                                            <?php if (is_array($states) && !empty($states)): ?>
                                                <span class="grt-states"><?php echo count($states); ?> states</span>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </label>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        
        <style>
        .grt-managed-table-config {
            margin: 0 -12px;
        }
        
        .grt-site-selection {
            background: #f9f9f9;
            border: 1px solid #ddd;
            border-radius: 4px;
            padding: 15px;
            margin-top: 10px;
        }
        
        .grt-sites-list {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
            gap: 10px;
            max-height: 500px;
            overflow-y: auto;
            padding: 10px;
        }
        
        .grt-site-item {
            background: #fff;
            border: 2px solid #ddd;
            border-radius: 4px;
            padding: 12px;
            transition: all 0.2s ease;
        }
        
        .grt-site-item:hover {
            border-color: #0073aa;
            background: #f0f7fc;
        }
        
        .grt-site-item.selected {
            border-color: #0073aa;
            background: #e7f3ff;
        }
        
        .grt-site-item label {
            display: flex;
            align-items: flex-start;
            cursor: pointer;
            margin: 0;
        }
        
        .grt-site-item input[type="checkbox"] {
            margin: 2px 10px 0 0;
            flex-shrink: 0;
        }
        
        .grt-site-item-info {
            flex: 1;
        }
        
        .grt-site-item-name {
            display: block;
            font-size: 14px;
            color: #23282d;
            margin-bottom: 4px;
        }
        
        .grt-site-item-meta {
            font-size: 12px;
            color: #666;
        }
        
        .grt-site-item-meta span {
            margin-right: 10px;
        }
        
        .grt-no-sites-message {
            text-align: center;
            padding: 30px;
            background: #fff;
            border: 2px dashed #ddd;
            border-radius: 4px;
        }
        </style>
        <?php
    }
    
    /**
     * Table information meta box callback
     */
    public function table_info_callback($post) {
        $slug = get_post_meta($post->ID, '_grt_managed_slug', true);
        
        if (empty($slug) && !empty($post->post_name)) {
            $slug = $post->post_name;
        }
        
        ?>
        <div class="grt-table-info">
            <p>
                <strong><?php _e('Shortcode:', 'geo-ranking-tables'); ?></strong><br>
                <?php if ($post->post_status === 'publish'): ?>
                    <code>[grt_table id="<?php echo esc_attr($post->ID); ?>"]</code><br>
                    <?php if (!empty($slug)): ?>
                        <code>[grt_table slug="<?php echo esc_attr($slug); ?>"]</code>
                    <?php endif; ?>
                <?php else: ?>
                    <em><?php _e('Publish to get shortcode', 'geo-ranking-tables'); ?></em>
                <?php endif; ?>
            </p>
            
            <p>
                <strong><?php _e('PHP Function:', 'geo-ranking-tables'); ?></strong><br>
                <?php if ($post->post_status === 'publish'): ?>
                    <code>grt_display_table(<?php echo esc_attr($post->ID); ?>)</code>
                <?php else: ?>
                    <em><?php _e('Publish to get PHP code', 'geo-ranking-tables'); ?></em>
                <?php endif; ?>
            </p>
            
            <?php if ($post->post_status === 'publish'): ?>
                <p>
                    <strong><?php _e('Usage:', 'geo-ranking-tables'); ?></strong><br>
                    <em><?php _e('Usage tracking available after first publish', 'geo-ranking-tables'); ?></em>
                </p>
            <?php endif; ?>
        </div>
        <?php
    }
    
    /**
     * Save meta box data
     */
    public function save_meta_boxes($post_id) {
        // Check nonce
        if (!isset($_POST['grt_managed_table_nonce'])) {
            return;
        }

        if (!wp_verify_nonce($_POST['grt_managed_table_nonce'], 'grt_managed_table_meta')) {
            return;
        }

        // Check permissions
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }

        // Don't save on autosave
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }

        // Check post type
        if (get_post_type($post_id) !== 'grt_managed_table') {
            return;
        }

        // Save managed sites
        $managed_sites = isset($_POST['grt_managed_sites']) && is_array($_POST['grt_managed_sites'])
            ? array_map('intval', $_POST['grt_managed_sites'])
            : array();
        update_post_meta($post_id, '_grt_managed_sites', $managed_sites);

        // Save max display
        $max_display = isset($_POST['grt_max_display']) ? intval($_POST['grt_max_display']) : 5;
        $max_display = max(1, min(20, $max_display));
        update_post_meta($post_id, '_grt_managed_max_display', $max_display);

        // Save table name
        $table_name = isset($_POST['grt_table_name']) ? sanitize_text_field($_POST['grt_table_name']) : '';
        update_post_meta($post_id, '_grt_managed_table_name', $table_name);

        // Save display options
        update_post_meta($post_id, '_grt_managed_show_ratings', isset($_POST['grt_show_ratings']) ? '1' : '0');
        update_post_meta($post_id, '_grt_managed_show_bonuses', isset($_POST['grt_show_bonuses']) ? '1' : '0');
        update_post_meta($post_id, '_grt_managed_show_licenses', isset($_POST['grt_show_licenses']) ? '1' : '0');
        update_post_meta($post_id, '_grt_managed_show_debug', isset($_POST['grt_show_debug']) ? '1' : '0');

        // Save fallback table ID
        $fallback_table_id = isset($_POST['grt_fallback_table']) ? intval($_POST['grt_fallback_table']) : 0;
        if ($fallback_table_id === $post_id) {
            $fallback_table_id = 0; // Prevent self-reference
        }
        update_post_meta($post_id, '_grt_fallback_table_id', $fallback_table_id);

        // Generate slug from post title if post_name doesn't exist yet
        $post = get_post($post_id);
        if ($post) {
            $slug = !empty($post->post_name) ? $post->post_name : sanitize_title($post->post_title);
            if (!empty($slug)) {
                update_post_meta($post_id, '_grt_managed_slug', $slug);
            }
        }
    }
    
    /**
     * Enqueue admin scripts
     */
    public function enqueue_admin_scripts($hook) {
        if ('post.php' == $hook || 'post-new.php' == $hook) {
            global $post_type;
            if ('grt_managed_table' == $post_type) {
                wp_enqueue_script('jquery');
                wp_add_inline_script('jquery', '
                    jQuery(document).ready(function($) {
                        $("#grt-select-all-sites").on("click", function(e) {
                            e.preventDefault();
                            $(".grt-sites-list input[type=checkbox]").prop("checked", true).closest(".grt-site-item").addClass("selected");
                        });
                        
                        $("#grt-deselect-all-sites").on("click", function(e) {
                            e.preventDefault();
                            $(".grt-sites-list input[type=checkbox]").prop("checked", false).closest(".grt-site-item").removeClass("selected");
                        });
                        
                        $(".grt-sites-list input[type=checkbox]").on("change", function() {
                            $(this).closest(".grt-site-item").toggleClass("selected", this.checked);
                        });
                    });
                ');
            }
        }
    }
    
    /**
     * Custom updated messages
     */
    public function updated_messages($messages) {
        $post = get_post();
        
        $messages['grt_managed_table'] = array(
            0  => '',
            1  => __('Managed table updated.', 'geo-ranking-tables'),
            2  => __('Custom field updated.', 'geo-ranking-tables'),
            3  => __('Custom field deleted.', 'geo-ranking-tables'),
            4  => __('Managed table updated.', 'geo-ranking-tables'),
            5  => isset($_GET['revision']) ? sprintf(__('Managed table restored to revision from %s', 'geo-ranking-tables'), wp_post_revision_title((int) $_GET['revision'], false)) : false,
            6  => __('Managed table published.', 'geo-ranking-tables'),
            7  => __('Managed table saved.', 'geo-ranking-tables'),
            8  => __('Managed table submitted.', 'geo-ranking-tables'),
            9  => sprintf(__('Managed table scheduled for: <strong>%1$s</strong>.', 'geo-ranking-tables'), date_i18n(__('M j, Y @ G:i'), strtotime($post->post_date))),
            10 => __('Managed table draft updated.', 'geo-ranking-tables')
        );
        
        return $messages;
    }
    
    /**
     * Add custom columns
     */
    public function add_custom_columns($columns) {
        $new_columns = array();
        $new_columns['cb'] = $columns['cb'];
        $new_columns['title'] = $columns['title'];
        $new_columns['sites_count'] = __('Sites', 'geo-ranking-tables');
        $new_columns['shortcode'] = __('Shortcode', 'geo-ranking-tables');
        $new_columns['usage'] = __('Usage', 'geo-ranking-tables');
        $new_columns['date'] = $columns['date'];
        
        return $new_columns;
    }
    
    /**
     * Custom column content
     */
    public function custom_column_content($column, $post_id) {
        switch ($column) {
            case 'sites_count':
                $sites = get_post_meta($post_id, '_grt_managed_sites', true);
                if (is_array($sites)) {
                    echo esc_html(count($sites));
                } else {
                    echo '0';
                }
                break;
                
            case 'shortcode':
                echo '<code>[grt_table id="' . esc_attr($post_id) . '"]</code>';
                break;
                
            case 'usage':
                echo '<em>' . __('Available after publish', 'geo-ranking-tables') . '</em>';
                break;
        }
    }
    
    /**
     * Get usage count for a managed table
     */
    private function get_usage_count($table_id) {
        global $wpdb;
        
        // Use transient to cache the result for 5 minutes
        $cache_key = 'grt_usage_count_' . $table_id;
        $cached = get_transient($cache_key);
        
        if ($cached !== false) {
            return $cached;
        }
        
        $total_count = 0;
        
        try {
            // Count blocks in post content
            $block_count = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_content LIKE %s AND post_status = 'publish'",
                '%"managedTableId":' . intval($table_id) . '%'
            ));
            
            $total_count += intval($block_count);
            
            // Count shortcodes in post content
            $slug = get_post_meta($table_id, '_grt_managed_slug', true);
            if (!empty($slug)) {
                $shortcode_count = $wpdb->get_var($wpdb->prepare(
                    "SELECT COUNT(*) FROM {$wpdb->posts} WHERE (post_content LIKE %s OR post_content LIKE %s) AND post_status = 'publish'",
                    '%[grt_table id="' . intval($table_id) . '"%',
                    '%[grt_table slug="' . $wpdb->esc_like($slug) . '"%'
                ));
                
                $total_count += intval($shortcode_count);
            }
        } catch (Exception $e) {
            // Log error but don't break the page
            error_log('GRT: Error counting usage: ' . $e->getMessage());
        }
        
        // Cache for 5 minutes
        set_transient($cache_key, $total_count, 300);
        
        return $total_count;
    }
    
    /**
     * Get managed table configuration
     */
    public static function get_config($table_id) {
        $post = get_post($table_id);
        
        if (!$post || $post->post_type !== 'grt_managed_table' || $post->post_status !== 'publish') {
            return false;
        }
        
        return array(
            'id' => $table_id,
            'title' => $post->post_title,
            'selectedSites' => get_post_meta($table_id, '_grt_managed_sites', true) ?: array(),
            'maxDisplay' => get_post_meta($table_id, '_grt_managed_max_display', true) ?: 5,
            'tableName' => get_post_meta($table_id, '_grt_managed_table_name', true) ?: $post->post_title,
            'showRatings' => get_post_meta($table_id, '_grt_managed_show_ratings', true) === '1',
            'showBonuses' => get_post_meta($table_id, '_grt_managed_show_bonuses', true) === '1',
            'showLicenses' => get_post_meta($table_id, '_grt_managed_show_licenses', true) === '1',
            'showDebugInfo' => get_post_meta($table_id, '_grt_managed_show_debug', true) === '1',
            'slug' => get_post_meta($table_id, '_grt_managed_slug', true) ?: $post->post_name,
            'fallbackTableId' => intval(get_post_meta($table_id, '_grt_fallback_table_id', true)) ?: 0,
        );
    }
    
    /**
     * Get managed table by slug
     */
    public static function get_by_slug($slug) {
        $tables = get_posts(array(
            'post_type' => 'grt_managed_table',
            'post_status' => 'publish',
            'meta_key' => '_grt_managed_slug',
            'meta_value' => $slug,
            'posts_per_page' => 1,
        ));
        
        if (empty($tables)) {
            return false;
        }
        
        return self::get_config($tables[0]->ID);
    }
}

