<?php
/**
 * Site Blocks Management Class
 * Handles the custom post type and meta fields for gambling sites
 */

if (!defined('ABSPATH')) {
    exit;
}

class GRT_SiteBlocks {
    
    public function __construct() {
        add_action('add_meta_boxes', array($this, 'add_meta_boxes'));
        add_action('save_post_grt_site_block', array($this, 'save_meta_boxes'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        add_filter('post_updated_messages', array($this, 'updated_messages'));
        add_filter('manage_grt_site_block_posts_columns', array($this, 'add_custom_columns'));
        add_action('manage_grt_site_block_posts_custom_column', array($this, 'custom_column_content'), 10, 2);
    }
    
    /**
     * Add meta boxes for site data
     */
    public function add_meta_boxes() {
        add_meta_box(
            'grt_site_details',
            __('Site Details', 'geo-ranking-tables'),
            array($this, 'site_details_callback'),
            'grt_site_block',
            'normal',
            'high'
        );
        
        add_meta_box(
            'grt_geo_settings',
            __('Geographic Settings', 'geo-ranking-tables'),
            array($this, 'geo_settings_callback'),
            'grt_site_block',
            'normal',
            'high'
        );
    }
    
    /**
     * Site details meta box callback
     */
    public function site_details_callback($post) {
        wp_nonce_field('grt_save_meta_boxes', 'grt_meta_nonce');
        
        $site_name = get_post_meta($post->ID, '_grt_site_name', true);
        $site_description = get_post_meta($post->ID, '_grt_site_description', true);
        $star_rating = get_post_meta($post->ID, '_grt_star_rating', true);
        $affiliate_link = get_post_meta($post->ID, '_grt_affiliate_link', true);
        $bonus_info = get_post_meta($post->ID, '_grt_bonus_info', true);
        $license_info = get_post_meta($post->ID, '_grt_license_info', true);
        
        ?>
        <table class="form-table">
            <tr>
                <th scope="row">
                    <label for="grt_site_name"><?php _e('Site Name', 'geo-ranking-tables'); ?></label>
                </th>
                <td>
                    <input type="text" id="grt_site_name" name="grt_site_name" value="<?php echo esc_attr($site_name); ?>" class="regular-text" />
                    <p class="description"><?php _e('The name of the gambling site', 'geo-ranking-tables'); ?></p>
                </td>
            </tr>
            <tr>
                <th scope="row">
                    <label for="grt_site_description"><?php _e('Site Description', 'geo-ranking-tables'); ?></label>
                </th>
                <td>
                    <textarea id="grt_site_description" name="grt_site_description" rows="4" class="large-text"><?php echo esc_textarea($site_description); ?></textarea>
                    <p class="description"><?php _e('Brief description of the gambling site', 'geo-ranking-tables'); ?></p>
                </td>
            </tr>
            <tr>
                <th scope="row">
                    <label for="grt_star_rating"><?php _e('Star Rating', 'geo-ranking-tables'); ?></label>
                </th>
                <td>
                    <select id="grt_star_rating" name="grt_star_rating">
                        <?php for ($i = 0; $i <= 50; $i += 5): ?>
                            <option value="<?php echo $i/10; ?>" <?php selected($star_rating, $i/10); ?>><?php echo $i/10; ?> Stars</option>
                        <?php endfor; ?>
                    </select>
                    <p class="description"><?php _e('Rating out of 5 stars', 'geo-ranking-tables'); ?></p>
                </td>
            </tr>
            <tr>
                <th scope="row">
                    <label for="grt_affiliate_link"><?php _e('Affiliate Link', 'geo-ranking-tables'); ?></label>
                </th>
                <td>
                    <input type="url" id="grt_affiliate_link" name="grt_affiliate_link" value="<?php echo esc_url($affiliate_link); ?>" class="regular-text" />
                    <p class="description"><?php _e('Affiliate/referral link for this site', 'geo-ranking-tables'); ?></p>
                </td>
            </tr>
            <tr>
                <th scope="row">
                    <label for="grt_bonus_info"><?php _e('Bonus Information', 'geo-ranking-tables'); ?></label>
                </th>
                <td>
                    <textarea id="grt_bonus_info" name="grt_bonus_info" rows="3" class="large-text"><?php echo esc_textarea($bonus_info); ?></textarea>
                    <p class="description"><?php _e('Welcome bonus or promotional information', 'geo-ranking-tables'); ?></p>
                </td>
            </tr>
            <tr>
                <th scope="row">
                    <label for="grt_license_info"><?php _e('License Information', 'geo-ranking-tables'); ?></label>
                </th>
                <td>
                    <input type="text" id="grt_license_info" name="grt_license_info" value="<?php echo esc_attr($license_info); ?>" class="regular-text" />
                    <p class="description"><?php _e('Licensing authority and license number', 'geo-ranking-tables'); ?></p>
                </td>
            </tr>
        </table>
        <?php
    }
    
    /**
     * Geographic settings meta box callback
     */
    public function geo_settings_callback($post) {
        $states_accepted = get_post_meta($post->ID, '_grt_states_accepted', true);
        if (!is_array($states_accepted)) {
            $states_accepted = array();
        }
        
        $us_states = array(
            'AL' => 'Alabama', 'AK' => 'Alaska', 'AZ' => 'Arizona', 'AR' => 'Arkansas', 'CA' => 'California',
            'CO' => 'Colorado', 'CT' => 'Connecticut', 'DE' => 'Delaware', 'FL' => 'Florida', 'GA' => 'Georgia',
            'HI' => 'Hawaii', 'ID' => 'Idaho', 'IL' => 'Illinois', 'IN' => 'Indiana', 'IA' => 'Iowa',
            'KS' => 'Kansas', 'KY' => 'Kentucky', 'LA' => 'Louisiana', 'ME' => 'Maine', 'MD' => 'Maryland',
            'MA' => 'Massachusetts', 'MI' => 'Michigan', 'MN' => 'Minnesota', 'MS' => 'Mississippi', 'MO' => 'Missouri',
            'MT' => 'Montana', 'NE' => 'Nebraska', 'NV' => 'Nevada', 'NH' => 'New Hampshire', 'NJ' => 'New Jersey',
            'NM' => 'New Mexico', 'NY' => 'New York', 'NC' => 'North Carolina', 'ND' => 'North Dakota', 'OH' => 'Ohio',
            'OK' => 'Oklahoma', 'OR' => 'Oregon', 'PA' => 'Pennsylvania', 'RI' => 'Rhode Island', 'SC' => 'South Carolina',
            'SD' => 'South Dakota', 'TN' => 'Tennessee', 'TX' => 'Texas', 'UT' => 'Utah', 'VT' => 'Vermont',
            'VA' => 'Virginia', 'WA' => 'Washington', 'WV' => 'West Virginia', 'WI' => 'Wisconsin', 'WY' => 'Wyoming',
            'DC' => 'District of Columbia'
        );
        
        ?>
        <div class="grt-states-container">
            <h4><?php _e('States Where This Site Accepts Players', 'geo-ranking-tables'); ?></h4>
            <p class="description"><?php _e('Select all states where this gambling site legally accepts players', 'geo-ranking-tables'); ?></p>
            
            <div class="grt-select-all-controls" style="margin: 10px 0;">
                <button type="button" id="grt-select-all-states" class="button"><?php _e('Select All', 'geo-ranking-tables'); ?></button>
                <button type="button" id="grt-deselect-all-states" class="button"><?php _e('Deselect All', 'geo-ranking-tables'); ?></button>
            </div>
            
            <div class="grt-states-grid" style="display: grid; grid-template-columns: repeat(auto-fill, minmax(200px, 1fr)); gap: 10px; max-height: 300px; overflow-y: auto; border: 1px solid #ddd; padding: 15px;">
                <?php foreach ($us_states as $code => $name): ?>
                    <label style="display: flex; align-items: center;">
                        <input type="checkbox" name="grt_states_accepted[]" value="<?php echo esc_attr($code); ?>" <?php checked(in_array($code, $states_accepted)); ?> />
                        <span style="margin-left: 5px;"><?php echo esc_html($name . ' (' . $code . ')'); ?></span>
                    </label>
                <?php endforeach; ?>
            </div>
        </div>
        
        <script>
        jQuery(document).ready(function($) {
            $('#grt-select-all-states').click(function() {
                $('.grt-states-grid input[type="checkbox"]').prop('checked', true);
            });
            
            $('#grt-deselect-all-states').click(function() {
                $('.grt-states-grid input[type="checkbox"]').prop('checked', false);
            });
        });
        </script>
        <?php
    }
    
    /**
     * Save meta box data
     */
    public function save_meta_boxes($post_id) {
        // Check if nonce is set
        if (!isset($_POST['grt_meta_nonce'])) {
            return;
        }
        
        // Verify nonce
        if (!wp_verify_nonce($_POST['grt_meta_nonce'], 'grt_save_meta_boxes')) {
            return;
        }
        
        // Check if user has permission to edit
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }
        
        // Don't save on autosave
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }
        
        // Save meta fields
        $meta_fields = array(
            'grt_site_name',
            'grt_site_description',
            'grt_star_rating',
            'grt_affiliate_link',
            'grt_bonus_info',
            'grt_license_info'
        );
        
        foreach ($meta_fields as $field) {
            if (isset($_POST[$field])) {
                update_post_meta($post_id, '_' . $field, sanitize_text_field($_POST[$field]));
            }
        }
        
        // Save states accepted
        if (isset($_POST['grt_states_accepted']) && is_array($_POST['grt_states_accepted'])) {
            $states = array_map('sanitize_text_field', $_POST['grt_states_accepted']);
            update_post_meta($post_id, '_grt_states_accepted', $states);
        } else {
            update_post_meta($post_id, '_grt_states_accepted', array());
        }
    }
    
    /**
     * Enqueue admin scripts
     */
    public function enqueue_admin_scripts($hook) {
        if ('post.php' == $hook || 'post-new.php' == $hook) {
            global $post_type;
            if ('grt_site_block' == $post_type) {
                wp_enqueue_script('jquery');
            }
        }
    }
    
    /**
     * Custom updated messages for site blocks
     */
    public function updated_messages($messages) {
        $post = get_post();
        
        $messages['grt_site_block'] = array(
            0  => '', // Unused. Messages start at index 1.
            1  => __('Site block updated.', 'geo-ranking-tables'),
            2  => __('Custom field updated.', 'geo-ranking-tables'),
            3  => __('Custom field deleted.', 'geo-ranking-tables'),
            4  => __('Site block updated.', 'geo-ranking-tables'),
            5  => isset($_GET['revision']) ? sprintf(__('Site block restored to revision from %s', 'geo-ranking-tables'), wp_post_revision_title((int) $_GET['revision'], false)) : false,
            6  => __('Site block published.', 'geo-ranking-tables'),
            7  => __('Site block saved.', 'geo-ranking-tables'),
            8  => __('Site block submitted.', 'geo-ranking-tables'),
            9  => sprintf(__('Site block scheduled for: <strong>%1$s</strong>.', 'geo-ranking-tables'), date_i18n(__('M j, Y @ G:i'), strtotime($post->post_date))),
            10 => __('Site block draft updated.', 'geo-ranking-tables')
        );
        
        return $messages;
    }
    
    /**
     * Add custom columns to site blocks list
     */
    public function add_custom_columns($columns) {
        $new_columns = array();
        $new_columns['cb'] = $columns['cb'];
        $new_columns['title'] = $columns['title'];
        $new_columns['site_name'] = __('Site Name', 'geo-ranking-tables');
        $new_columns['rating'] = __('Rating', 'geo-ranking-tables');
        $new_columns['states_count'] = __('States', 'geo-ranking-tables');
        $new_columns['date'] = $columns['date'];
        
        return $new_columns;
    }
    
    /**
     * Custom column content
     */
    public function custom_column_content($column, $post_id) {
        switch ($column) {
            case 'site_name':
                $site_name = get_post_meta($post_id, '_grt_site_name', true);
                echo esc_html($site_name ? $site_name : '-');
                break;
                
            case 'rating':
                $rating = get_post_meta($post_id, '_grt_star_rating', true);
                if ($rating) {
                    echo esc_html($rating . '/5 ⭐');
                } else {
                    echo '-';
                }
                break;
                
            case 'states_count':
                $states = get_post_meta($post_id, '_grt_states_accepted', true);
                if (is_array($states)) {
                    echo esc_html(count($states) . ' states');
                } else {
                    echo '0 states';
                }
                break;
        }
    }
}