<?php
/**
 * Table Renderer Class
 * Unified rendering engine for ranking tables
 */

if (!defined('ABSPATH')) {
    exit;
}

class GRT_TableRenderer {
    
    /**
     * Render a ranking table
     * 
     * @param array $config Table configuration
     * @param array $overrides Optional overrides for the configuration
     * @return string HTML output
     */
    public function render($config, $overrides = array()) {
        // Merge config with overrides
        $final_config = $this->merge_config($config, $overrides);
        
        // Validate configuration
        if (empty($final_config['selectedSites']) || !is_array($final_config['selectedSites'])) {
            return $this->render_no_sites_message();
        }
        
        // Generate unique block ID
        $block_id = 'grt-table-' . md5(serialize($final_config) . time());
        
        // Get user location for debug info
        $user_location = $this->get_user_location_for_debug();
        
        // Render the table HTML
        ob_start();
        ?>
        <div class="grt-ranking-table-wrapper" id="<?php echo esc_attr($block_id); ?>">
            <div class="grt-table-header">
                <h3 class="grt-table-title"><?php echo esc_html($final_config['tableName']); ?></h3>
                <div class="grt-location-info">
                    <span class="grt-location-text">
                        <?php _e('Showing sites available in:', 'geo-ranking-tables'); ?> 
                        <strong class="grt-user-location"><?php _e('Detecting location...', 'geo-ranking-tables'); ?></strong>
                    </span>
                </div>
            </div>
            
            <div class="grt-table-loading">
                <div class="grt-loading-spinner"></div>
                <p><?php _e('Loading available gambling sites for your location...', 'geo-ranking-tables'); ?></p>
            </div>
            
            <div class="grt-table-content" style="display: none;">
                <!-- Content will be populated by JavaScript -->
            </div>
            
            <div class="grt-table-fallback" style="display: none;">
                <div class="grt-fallback-message">
                    <h4><?php _e('No sites available in your location', 'geo-ranking-tables'); ?></h4>
                    <p><?php _e('Unfortunately, none of our featured gambling sites accept players from your current location.', 'geo-ranking-tables'); ?></p>
                </div>
            </div>
            
            <?php if ($final_config['showDebugInfo']): ?>
                <?php echo $this->render_debug_info($final_config, $block_id, $user_location); ?>
            <?php endif; ?>
        </div>
        
        <script type="application/json" class="grt-block-config" data-block-id="<?php echo esc_attr($block_id); ?>">
        <?php echo wp_json_encode($final_config); ?>
        </script>
        <?php
        
        return ob_get_clean();
    }
    
    /**
     * Get managed table configuration
     * 
     * @param int $table_id Managed table post ID
     * @return array|false Configuration array or false if not found
     */
    public function get_managed_config($table_id) {
        return GRT_ManagedTables::get_config($table_id);
    }
    
    /**
     * Get managed table configuration by slug
     * 
     * @param string $slug Managed table slug
     * @return array|false Configuration array or false if not found
     */
    public function get_managed_config_by_slug($slug) {
        return GRT_ManagedTables::get_by_slug($slug);
    }
    
    /**
     * Merge configuration with overrides
     * 
     * @param array $config Base configuration
     * @param array $overrides Override values
     * @return array Merged configuration
     */
    private function merge_config($config, $overrides) {
        $defaults = array(
            'selectedSites' => array(),
            'maxDisplay' => 5,
            'tableName' => __('Top Gambling Sites', 'geo-ranking-tables'),
            'showRatings' => true,
            'showBonuses' => true,
            'showLicenses' => false,
            'showDebugInfo' => false,
            'fallbackTableId' => 0,
        );
        
        // Start with defaults
        $final = wp_parse_args($config, $defaults);
        
        // Apply overrides
        if (!empty($overrides)) {
            foreach ($overrides as $key => $value) {
                $final[$key] = $value;
            }
        }
        
        return $final;
    }
    
    /**
     * Render debug information panel
     */
    private function render_debug_info($config, $block_id, $user_location) {
        ob_start();
        ?>
        <div class="grt-debug-info" style="margin-top: 20px; padding: 15px; background: #e6f3ff; border: 1px solid #0073aa; border-radius: 5px; font-size: 12px; font-family: monospace;">
            <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                <strong style="color: #0073aa; font-size: 14px;">🔍 <?php _e('Debug Information', 'geo-ranking-tables'); ?></strong>
                <small style="color: #666;"><?php _e('Render Time:', 'geo-ranking-tables'); ?> <?php echo date('Y-m-d H:i:s'); ?></small>
            </div>
            
            <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px;">
                <div>
                    <strong style="color: #333;"><?php _e('Table Configuration:', 'geo-ranking-tables'); ?></strong><br>
                    <?php _e('Block ID:', 'geo-ranking-tables'); ?> <code><?php echo esc_html($block_id); ?></code><br>
                    <?php _e('Table Name:', 'geo-ranking-tables'); ?> <strong><?php echo esc_html($config['tableName']); ?></strong><br>
                    <?php _e('Selected Sites:', 'geo-ranking-tables'); ?> <code>[<?php echo esc_html(implode(', ', $config['selectedSites'])); ?>]</code><br>
                    <?php _e('Max Display:', 'geo-ranking-tables'); ?> <strong><?php echo esc_html($config['maxDisplay']); ?></strong><br>
                    <?php _e('Show Ratings:', 'geo-ranking-tables'); ?> <?php echo $config['showRatings'] ? '✅' : '❌'; ?><br>
                    <?php _e('Show Bonuses:', 'geo-ranking-tables'); ?> <?php echo $config['showBonuses'] ? '✅' : '❌'; ?><br>
                    <?php _e('Show Licenses:', 'geo-ranking-tables'); ?> <?php echo $config['showLicenses'] ? '✅' : '❌'; ?>
                </div>
                
                <div>
                    <strong style="color: #333;"><?php _e('User Location:', 'geo-ranking-tables'); ?></strong><br>
                    <?php if (isset($user_location['forced_ip']) && $user_location['forced_ip']): ?>
                        <div style="background: #fff3cd; border: 1px solid #ffeaa7; padding: 8px; border-radius: 4px; margin-bottom: 8px; font-size: 11px;">
                            🧪 <strong><?php _e('Test Mode Active', 'geo-ranking-tables'); ?></strong><br>
                            <code><?php _e('Original IP:', 'geo-ranking-tables'); ?> <?php echo esc_html($user_location['original_ip']); ?></code>
                        </div>
                    <?php endif; ?>
                    <?php _e('IP Address:', 'geo-ranking-tables'); ?> <code><?php echo esc_html($user_location['ip']); ?></code>
                    <?php if (isset($user_location['forced_ip']) && $user_location['forced_ip']): ?>
                        <span style="color: #d68910; font-weight: bold;"> (<?php _e('FORCED', 'geo-ranking-tables'); ?>)</span>
                    <?php endif; ?><br>
                    <?php _e('Country:', 'geo-ranking-tables'); ?> <strong><?php echo esc_html($user_location['country']); ?></strong><br>
                    <?php _e('State:', 'geo-ranking-tables'); ?> <strong><?php echo esc_html($user_location['state']); ?> (<?php echo esc_html($user_location['state_code']); ?>)</strong><br>
                    <?php _e('City:', 'geo-ranking-tables'); ?> <?php echo esc_html($user_location['city']); ?><br>
                    <?php _e('Detection Service:', 'geo-ranking-tables'); ?> <code><?php echo esc_html($user_location['service']); ?></code>
                </div>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }
    
    /**
     * Render "no sites" message
     */
    private function render_no_sites_message() {
        ob_start();
        ?>
        <div class="grt-no-sites" style="padding: 20px; border: 2px solid #d63638; text-align: center; background: #ffe6e6; margin: 20px 0; border-radius: 4px;">
            <p style="margin: 0; color: #d63638; font-weight: bold;">
                ⚠️ <?php _e('No sites selected for this ranking table. Please configure the table settings.', 'geo-ranking-tables'); ?>
            </p>
        </div>
        <?php
        return ob_get_clean();
    }
    
    /**
     * Get user location for debug display
     */
    private function get_user_location_for_debug() {
        // Include the geo detector if available
        if (class_exists('GRT_GeoDetector')) {
            $detector = new GRT_GeoDetector();
            $location = $detector->get_user_location();
            return $location;
        }
        
        // Fallback to basic detection
        $ip = $this->get_forced_test_ip() ?: $this->get_user_ip();
        $is_forced = $this->get_forced_test_ip() !== false;
        
        $location = array(
            'ip' => $ip,
            'country' => 'Unknown',
            'state' => 'Unknown',
            'state_code' => 'N/A',
            'city' => 'Unknown',
            'service' => 'basic'
        );
        
        if ($is_forced) {
            $location['forced_ip'] = true;
            $location['original_ip'] = $this->get_user_ip();
        }
        
        return $location;
    }
    
    /**
     * Get forced test IP from URL parameter
     */
    private function get_forced_test_ip() {
        if (!isset($_GET['ftm-ip'])) {
            return false;
        }
        
        $test_ip = sanitize_text_field($_GET['ftm-ip']);
        
        if (filter_var($test_ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4 | FILTER_FLAG_IPV6)) {
            return $test_ip;
        }
        
        return false;
    }
    
    /**
     * Get user IP
     */
    private function get_user_ip() {
        $ip_headers = array(
            'HTTP_CF_CONNECTING_IP',
            'HTTP_CLIENT_IP',
            'HTTP_X_FORWARDED_FOR',
            'HTTP_X_FORWARDED',
            'HTTP_X_CLUSTER_CLIENT_IP',
            'HTTP_FORWARDED_FOR',
            'HTTP_FORWARDED',
            'REMOTE_ADDR'
        );
        
        foreach ($ip_headers as $header) {
            if (!empty($_SERVER[$header])) {
                $ip = $_SERVER[$header];
                if (strpos($ip, ',') !== false) {
                    $ip_list = explode(',', $ip);
                    $ip = trim($ip_list[0]);
                }
                if (filter_var($ip, FILTER_VALIDATE_IP)) {
                    return $ip;
                }
            }
        }
        
        return isset($_SERVER['REMOTE_ADDR']) ? $_SERVER['REMOTE_ADDR'] : '127.0.0.1';
    }
}




