<?php

/**
 * Plugin Name: Geo-Ranking Tables
 * Plugin URI: https://your-site.com/geo-ranking-tables
 * Description: A WordPress plugin that creates geo-targeted ranking tables for gambling sites with dynamic filtering based on user location.
 * Version: 1.0.2
 * Author: Your Name
 * License: GPL v2 or later
 * Text Domain: geo-ranking-tables
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('GRT_PLUGIN_URL', plugin_dir_url(__FILE__));
define('GRT_PLUGIN_PATH', plugin_dir_path(__FILE__));
define('GRT_VERSION', '1.0.2');

/**
 * Main Geo Ranking Tables Class
 */
class GeoRankingTables
{

    private static $instance = null;

    public static function get_instance()
    {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct()
    {
        add_action('plugins_loaded', array($this, 'init'));
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
    }

    /**
     * Initialize the plugin
     */
    public function init()
    {
        // Load includes first
        $this->includes();

        // Initialize on init hook with proper priority
        add_action('init', array($this, 'register_post_types'), 0);
        add_action('init', array($this, 'init_components'), 10);

        // Load textdomain for translations
        load_plugin_textdomain('geo-ranking-tables', false, dirname(plugin_basename(__FILE__)) . '/languages');
    }

    /**
     * Register custom post types
     */
    public function register_post_types()
    {
        // Register Site Blocks CPT
        $labels = array(
            'name' => __('Site Blocks', 'geo-ranking-tables'),
            'singular_name' => __('Site Block', 'geo-ranking-tables'),
            'menu_name' => __('Site Blocks', 'geo-ranking-tables'),
            'add_new' => __('Add New Site', 'geo-ranking-tables'),
            'add_new_item' => __('Add New Site Block', 'geo-ranking-tables'),
            'edit_item' => __('Edit Site Block', 'geo-ranking-tables'),
            'new_item' => __('New Site Block', 'geo-ranking-tables'),
            'view_item' => __('View Site Block', 'geo-ranking-tables'),
            'search_items' => __('Search Site Blocks', 'geo-ranking-tables'),
            'not_found' => __('No site blocks found', 'geo-ranking-tables'),
            'not_found_in_trash' => __('No site blocks found in trash', 'geo-ranking-tables'),
        );

        $args = array(
            'labels' => $labels,
            'public' => false,
            'show_ui' => true,
            'show_in_menu' => 'geo-ranking-tables',
            'menu_icon' => 'dashicons-star-filled',
            'supports' => array('title', 'editor', 'thumbnail'),
            'has_archive' => false,
            'rewrite' => false,
            'capability_type' => 'post',
            'show_in_rest' => true,
        );

        register_post_type('grt_site_block', $args);

        // Register Managed Tables CPT
        $managed_labels = array(
            'name' => __('Managed Tables', 'geo-ranking-tables'),
            'singular_name' => __('Managed Table', 'geo-ranking-tables'),
            'menu_name' => __('Managed Tables', 'geo-ranking-tables'),
            'add_new' => __('Add New Table', 'geo-ranking-tables'),
            'add_new_item' => __('Add New Managed Table', 'geo-ranking-tables'),
            'edit_item' => __('Edit Managed Table', 'geo-ranking-tables'),
            'new_item' => __('New Managed Table', 'geo-ranking-tables'),
            'view_item' => __('View Managed Table', 'geo-ranking-tables'),
            'search_items' => __('Search Managed Tables', 'geo-ranking-tables'),
            'not_found' => __('No managed tables found', 'geo-ranking-tables'),
            'not_found_in_trash' => __('No managed tables found in trash', 'geo-ranking-tables'),
        );

        $managed_args = array(
            'labels' => $managed_labels,
            'public' => false,
            'show_ui' => true,
            'show_in_menu' => false, // We add it manually via submenu
            'menu_icon' => 'dashicons-editor-table',
            'supports' => array('title'),
            'has_archive' => false,
            'rewrite' => false,
            'capability_type' => 'post',
            'show_in_rest' => false,
        );

        register_post_type('grt_managed_table', $managed_args);
    }

    /**
     * Exclude managed tables from ACF
     */
    public function exclude_managed_tables_from_acf($post_types, $args)
    {
        // Remove grt_managed_table from ACF post types
        if (is_array($post_types)) {
            $post_types = array_diff($post_types, array('grt_managed_table'));
        }

        return $post_types;
    }

    /**
     * Initialize components after post types are registered
     */
    public function init_components()
    {
        // Register themes first
        if (class_exists('GRT_ThemeRegistry')) {
            GRT_ThemeRegistry::register_default_themes();
        }

        // Initialize components after includes are loaded
        if (class_exists('GRT_SiteBlocks')) {
            new GRT_SiteBlocks();
        }
        if (class_exists('GRT_TableBlocks')) {
            new GRT_TableBlocks();
        }
        if (class_exists('GRT_Admin')) {
            new GRT_Admin();
        }
        if (class_exists('GRT_Frontend')) {
            new GRT_Frontend();
        }
        if (class_exists('GRT_ManagedTables')) {
            new GRT_ManagedTables();
        }
        if (class_exists('GRT_ImportExport')) {
            new GRT_ImportExport();
        }

        // Register blocks directly here to ensure proper timing
        $this->register_blocks();
    }

    /**
     * Register Gutenberg blocks
     */
    public function register_blocks()
    {
        // Check if Gutenberg is available
        if (!function_exists('register_block_type')) {
            return;
        }

        // Get the table blocks instance for the render callback
        global $grt_table_blocks_instance;

        // Register the ranking table block
        register_block_type('geo-ranking-tables/ranking-table', array(
            'render_callback' => array($this, 'render_ranking_table_block'),
            'attributes' => array(
                'useManaged' => array(
                    'type' => 'boolean',
                    'default' => false,
                ),
                'managedTableId' => array(
                    'type' => 'number',
                    'default' => 0,
                ),
                'overrideSettings' => array(
                    'type' => 'object',
                    'default' => array(),
                ),
                'selectedSites' => array(
                    'type' => 'array',
                    'default' => array(),
                ),
                'maxDisplay' => array(
                    'type' => 'number',
                    'default' => 5,
                ),
                'fallbackBlockId' => array(
                    'type' => 'number',
                    'default' => 0,
                ),
                'tableName' => array(
                    'type' => 'string',
                    'default' => 'Top Gambling Sites',
                ),
                'showRatings' => array(
                    'type' => 'boolean',
                    'default' => true,
                ),
                'showBonuses' => array(
                    'type' => 'boolean',
                    'default' => true,
                ),
                'showLicenses' => array(
                    'type' => 'boolean',
                    'default' => false,
                ),
                'showDebugInfo' => array(
                    'type' => 'boolean',
                    'default' => false,
                ),
            ),
        ));
    }

    /**
     * Render callback for ranking table block
     */
    public function render_ranking_table_block($attributes)
    {

        // Check if using managed mode
        $use_managed = isset($attributes['useManaged']) && $attributes['useManaged'];
        $managed_table_id = isset($attributes['managedTableId']) ? intval($attributes['managedTableId']) : 0;

        if ($use_managed && $managed_table_id > 0) {
            // Get managed table configuration
            $config = GRT_ManagedTables::get_config($managed_table_id);

            if (!$config) {
                return '<div class="grt-error" style="padding: 20px; border: 2px solid #d63638; text-align: center; background: #ffe6e6; margin: 20px 0; border-radius: 4px;">' .
                    __('Managed table not found or not published.', 'geo-ranking-tables') .
                    '</div>';
            }

            // Apply any overrides from block attributes
            $overrides = isset($attributes['overrideSettings']) && is_array($attributes['overrideSettings'])
                ? $attributes['overrideSettings']
                : array();

            // Use the table renderer for unified rendering
            $renderer = new GRT_TableRenderer();
            return $renderer->render($config, $overrides);
        }

        // Fall back to custom/local mode (original behavior)
        $selected_sites = isset($attributes['selectedSites']) ? $attributes['selectedSites'] : array();
        $max_display = isset($attributes['maxDisplay']) ? intval($attributes['maxDisplay']) : 5;
        $table_name = isset($attributes['tableName']) ? $attributes['tableName'] : 'Top Gambling Sites';
        $show_debug = isset($attributes['showDebugInfo']) ? $attributes['showDebugInfo'] : false;

        // Detect user location for debug info
        $user_location = $this->get_user_location_for_debug();

        if (empty($selected_sites)) {
            $no_sites_html = '<div class="grt-no-sites" style="padding: 20px; border: 2px solid red; text-align: center; background: #ffe6e6; margin: 20px 0;">⚠️ No sites selected for this ranking table. Please edit the block and select some sites.</div>';

            if ($show_debug) {
                $no_sites_html .= $this->render_debug_info($attributes, 'no-sites-debug', $user_location);
            }

            return $no_sites_html;
        }

        // Generate unique ID for this table block
        $block_id = 'grt-table-' . md5(serialize($attributes));

        ob_start();
?>
        <!-- GRT DEBUG: Main plugin render called -->
        <div class="grt-ranking-table-wrapper" id="<?php echo esc_attr($block_id); ?>" style="border: 2px solid blue; margin: 20px 0; padding: 20px;">
            <div class="grt-table-header">
                <h3 class="grt-table-title"><?php echo esc_html($table_name); ?></h3>
                <div class="grt-location-info">
                    <span class="grt-location-text">Showing sites available in: <strong class="grt-user-location">Detecting location...</strong></span>
                </div>
            </div>

            <div class="grt-table-loading">
                <div class="grt-loading-spinner"></div>
                <p>Loading available gambling sites for your location...</p>
            </div>

            <div class="grt-table-content" style="display: none;">
                <!-- Content will be populated by JavaScript -->
            </div>

            <div class="grt-table-fallback" style="display: none;">
                <div class="grt-fallback-message">
                    <h4>No sites available in your location</h4>
                    <p>Unfortunately, none of our featured gambling sites accept players from your current location.</p>
                </div>
            </div>

            <?php if ($show_debug): ?>
                <?php echo $this->render_debug_info($attributes, $block_id, $user_location); ?>
            <?php endif; ?>
        </div>

        <script type="application/json" class="grt-block-config" data-block-id="<?php echo esc_attr($block_id); ?>">
            <?php echo wp_json_encode(array(
                'selectedSites' => $selected_sites,
                'maxDisplay' => $max_display,
                'fallbackBlockId' => 0,
                'showRatings' => isset($attributes['showRatings']) ? $attributes['showRatings'] : true,
                'showBonuses' => isset($attributes['showBonuses']) ? $attributes['showBonuses'] : true,
                'showLicenses' => isset($attributes['showLicenses']) ? $attributes['showLicenses'] : false,
                'showDebugInfo' => $show_debug,
            )); ?>
        </script>
    <?php
        return ob_get_clean();
    }

    /**
     * Get user location for debug display
     */
    private function get_user_location_for_debug()
    {
        // Include the geo detector if available
        if (class_exists('GRT_GeoDetector')) {
            $detector = new GRT_GeoDetector();
            $location = $detector->get_user_location();
            return $location;
        }

        // Fallback to basic IP detection
        $ip = $this->get_forced_test_ip() ?: $this->get_user_ip();
        $is_forced = $this->get_forced_test_ip() !== false;

        $location = array(
            'ip' => $ip,
            'country' => 'Unknown',
            'state' => 'Unknown',
            'state_code' => 'N/A',
            'city' => 'Unknown',
            'service' => 'basic'
        );

        if ($is_forced) {
            $location['forced_ip'] = true;
            $location['original_ip'] = $this->get_user_ip();
        }

        return $location;
    }

    /**
     * Get forced test IP from URL parameter
     */
    private function get_forced_test_ip()
    {
        if (!isset($_GET['ftm-ip'])) {
            return false;
        }

        $test_ip = sanitize_text_field($_GET['ftm-ip']);

        // Validate IPv4 or IPv6 address
        if (filter_var($test_ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4 | FILTER_FLAG_IPV6)) {
            return $test_ip;
        }

        return false;
    }

    /**
     * Get user IP for debug
     */
    private function get_user_ip()
    {
        $ip_headers = array(
            'HTTP_CF_CONNECTING_IP',
            'HTTP_CLIENT_IP',
            'HTTP_X_FORWARDED_FOR',
            'HTTP_X_FORWARDED',
            'HTTP_X_CLUSTER_CLIENT_IP',
            'HTTP_FORWARDED_FOR',
            'HTTP_FORWARDED',
            'REMOTE_ADDR'
        );

        foreach ($ip_headers as $header) {
            if (!empty($_SERVER[$header])) {
                $ip = $_SERVER[$header];
                if (strpos($ip, ',') !== false) {
                    $ip_list = explode(',', $ip);
                    $ip = trim($ip_list[0]);
                }
                if (filter_var($ip, FILTER_VALIDATE_IP)) {
                    return $ip;
                }
            }
        }

        return isset($_SERVER['REMOTE_ADDR']) ? $_SERVER['REMOTE_ADDR'] : '127.0.0.1';
    }

    /**
     * Render debug information
     */
    private function render_debug_info($attributes, $block_id, $user_location)
    {
        $selected_sites = isset($attributes['selectedSites']) ? $attributes['selectedSites'] : array();
        $max_display = isset($attributes['maxDisplay']) ? $attributes['maxDisplay'] : 5;
        $table_name = isset($attributes['tableName']) ? $attributes['tableName'] : 'Top Gambling Sites';

        ob_start();
    ?>
        <div class="grt-debug-info" style="margin-top: 20px; padding: 15px; background: #e6f3ff; border: 1px solid #0073aa; border-radius: 5px; font-size: 12px; font-family: monospace;">
            <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px;">
                <strong style="color: #0073aa; font-size: 14px;">🔍 Debug Information</strong>
                <small style="color: #666;">Render Time: <?php echo date('Y-m-d H:i:s'); ?></small>
            </div>

            <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px;">
                <div>
                    <strong style="color: #333;">Block Configuration:</strong><br>
                    Block ID: <code><?php echo esc_html($block_id); ?></code><br>
                    Table Name: <strong><?php echo esc_html($table_name); ?></strong><br>
                    Selected Sites: <code>[<?php echo esc_html(implode(', ', $selected_sites)); ?>]</code><br>
                    Max Display: <strong><?php echo esc_html($max_display); ?></strong><br>
                    Show Ratings: <?php echo isset($attributes['showRatings']) && $attributes['showRatings'] ? '✅' : '❌'; ?><br>
                    Show Bonuses: <?php echo isset($attributes['showBonuses']) && $attributes['showBonuses'] ? '✅' : '❌'; ?><br>
                    Show Licenses: <?php echo isset($attributes['showLicenses']) && $attributes['showLicenses'] ? '✅' : '❌'; ?>
                </div>

                <div>
                    <strong style="color: #333;">User Location:</strong><br>
                    <?php if (isset($user_location['forced_ip']) && $user_location['forced_ip']): ?>
                        <div style="background: #fff3cd; border: 1px solid #ffeaa7; padding: 8px; border-radius: 4px; margin-bottom: 8px; font-size: 11px;">
                            🧪 <strong>Test Mode Active:</strong> Using forced IP from URL parameter<br>
                            <code>Original IP: <?php echo esc_html($user_location['original_ip']); ?></code>
                        </div>
                    <?php endif; ?>
                    IP Address: <code><?php echo esc_html($user_location['ip']); ?></code>
                    <?php if (isset($user_location['forced_ip']) && $user_location['forced_ip']): ?>
                        <span style="color: #d68910; font-weight: bold;"> (FORCED)</span>
                    <?php endif; ?><br>
                    Country: <strong><?php echo esc_html($user_location['country']); ?></strong><br>
                    State: <strong><?php echo esc_html($user_location['state']); ?> (<?php echo esc_html($user_location['state_code']); ?>)</strong><br>
                    City: <?php echo esc_html($user_location['city']); ?><br>
                    Detection Service: <code><?php echo esc_html($user_location['service']); ?></code><br>
                    <small style="color: #666;">
                        <?php if ($user_location['service'] === 'basic'): ?>
                            ⚠️ Using basic detection (no API)
                        <?php else: ?>
                            ✅ Using geo-location API
                        <?php endif; ?>
                    </small>
                </div>
            </div>

            <?php if (!empty($selected_sites)): ?>
                <div style="margin-top: 10px; padding-top: 10px; border-top: 1px solid #ccc;">
                    <strong style="color: #333;">Selected Sites Status:</strong><br>
                    <div style="margin-top: 5px;">
                        <?php foreach ($selected_sites as $site_id): ?>
                            <?php
                            $site_post = get_post($site_id);
                            $site_name = $site_post ? $site_post->post_title : 'Unknown Site';
                            $states_accepted = get_post_meta($site_id, '_grt_states_accepted', true);
                            $is_available = is_array($states_accepted) && in_array($user_location['state_code'], $states_accepted);
                            ?>
                            <span style="display: inline-block; margin-right: 10px; margin-bottom: 5px; padding: 2px 6px; background: <?php echo $is_available ? '#d4edda' : '#f8d7da'; ?>; border-radius: 3px; font-size: 11px;">
                                <?php echo $is_available ? '✅' : '❌'; ?> <?php echo esc_html($site_name); ?> (ID: <?php echo esc_html($site_id); ?>)
                            </span>
                        <?php endforeach; ?>
                    </div>
                </div>
            <?php endif; ?>

            <div style="margin-top: 10px; padding-top: 10px; border-top: 1px solid #ccc; font-size: 11px; color: #666;">
                <strong>💡 Debug Tips:</strong>
                Check browser console for JavaScript logs |
                Look for "GRT:" messages |
                This debug info can be hidden in block settings
            </div>
        </div>
<?php
        return ob_get_clean();
    }

    /**
     * Include required files
     */
    private function includes()
    {
        require_once GRT_PLUGIN_PATH . 'includes/class-site-blocks.php';
        require_once GRT_PLUGIN_PATH . 'includes/class-table-blocks.php';
        require_once GRT_PLUGIN_PATH . 'includes/class-admin.php';
        require_once GRT_PLUGIN_PATH . 'includes/class-frontend.php';
        require_once GRT_PLUGIN_PATH . 'includes/class-geo-detector.php';
        require_once GRT_PLUGIN_PATH . 'includes/class-managed-tables.php';
        require_once GRT_PLUGIN_PATH . 'includes/class-table-renderer.php';
        require_once GRT_PLUGIN_PATH . 'includes/class-exporter.php';
        require_once GRT_PLUGIN_PATH . 'includes/class-importer.php';
        require_once GRT_PLUGIN_PATH . 'includes/class-import-export.php';
        require_once GRT_PLUGIN_PATH . 'includes/sample-data.php';

        // Theme system
        require_once GRT_PLUGIN_PATH . 'includes/class-theme-registry.php';
        require_once GRT_PLUGIN_PATH . 'includes/themes/class-theme-template.php';
        require_once GRT_PLUGIN_PATH . 'includes/themes/class-theme-review-style.php';
        require_once GRT_PLUGIN_PATH . 'includes/themes/class-theme-gs-theme.php';
    }

    /**
     * Plugin activation
     */
    public function activate()
    {
        // Register post types first
        $this->register_post_types();

        // Create database tables
        $this->create_tables();

        // Set default options
        add_option('grt_settings', array(
            'geo_api_key' => '',
            'fallback_enabled' => true,
            'cache_duration' => 3600
        ));

        // Flush rewrite rules
        flush_rewrite_rules();
    }

    /**
     * Plugin deactivation
     */
    public function deactivate()
    {
        // Flush rewrite rules
        flush_rewrite_rules();
    }

    /**
     * Create database tables
     */
    private function create_tables()
    {
        global $wpdb;

        $charset_collate = $wpdb->get_charset_collate();

        // Sites table
        $sites_table = $wpdb->prefix . 'grt_sites';
        $sites_sql = "CREATE TABLE $sites_table (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            site_name varchar(255) NOT NULL,
            site_description text,
            star_rating decimal(2,1) DEFAULT 0.0,
            states_accepted text,
            additional_fields longtext,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id)
        ) $charset_collate;";

        // Table blocks configuration
        $table_blocks_table = $wpdb->prefix . 'grt_table_blocks';
        $table_blocks_sql = "CREATE TABLE $table_blocks_table (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            block_name varchar(255) NOT NULL,
            selected_sites text,
            max_display int(11) DEFAULT 5,
            fallback_block_id int(11),
            settings longtext,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id)
        ) $charset_collate;";

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sites_sql);
        dbDelta($table_blocks_sql);
    }
}

// Initialize the plugin
GeoRankingTables::get_instance();

/**
 * Template Functions for Theme Developers
 */

/**
 * Display a managed ranking table by ID
 * 
 * @param int $table_id Managed table post ID
 * @param array $overrides Optional overrides for the configuration
 * @return void Echoes the HTML output
 */
function grt_display_table($table_id, $overrides = array())
{
    $config = GRT_ManagedTables::get_config($table_id);

    if (!$config) {
        echo '<div class="grt-error" style="padding: 15px; background: #ffe6e6; border: 1px solid #d63638; border-radius: 4px; color: #d63638;">' .
            __('Managed table not found or not published.', 'geo-ranking-tables') .
            '</div>';
        return;
    }

    $renderer = new GRT_TableRenderer();
    echo $renderer->render($config, $overrides);
}

/**
 * Display a managed ranking table by slug
 * 
 * @param string $slug Managed table slug
 * @param array $overrides Optional overrides for the configuration
 * @return void Echoes the HTML output
 */
function grt_display_table_by_slug($slug, $overrides = array())
{
    $config = GRT_ManagedTables::get_by_slug($slug);

    if (!$config) {
        echo '<div class="grt-error" style="padding: 15px; background: #ffe6e6; border: 1px solid #d63638; border-radius: 4px; color: #d63638;">' .
            __('Managed table not found or not published.', 'geo-ranking-tables') .
            '</div>';
        return;
    }

    $renderer = new GRT_TableRenderer();
    echo $renderer->render($config, $overrides);
}

/**
 * Get a managed ranking table HTML by ID (returns instead of echoing)
 * 
 * @param int $table_id Managed table post ID
 * @param array $overrides Optional overrides for the configuration
 * @return string HTML output
 */
function grt_get_table($table_id, $overrides = array())
{
    $config = GRT_ManagedTables::get_config($table_id);

    if (!$config) {
        return '<div class="grt-error" style="padding: 15px; background: #ffe6e6; border: 1px solid #d63638; border-radius: 4px; color: #d63638;">' .
            __('Managed table not found or not published.', 'geo-ranking-tables') .
            '</div>';
    }

    $renderer = new GRT_TableRenderer();
    return $renderer->render($config, $overrides);
}

/**
 * Get a managed ranking table HTML by slug (returns instead of echoing)
 * 
 * @param string $slug Managed table slug
 * @param array $overrides Optional overrides for the configuration
 * @return string HTML output
 */
function grt_get_table_by_slug($slug, $overrides = array())
{
    $config = GRT_ManagedTables::get_by_slug($slug);

    if (!$config) {
        return '<div class="grt-error" style="padding: 15px; background: #ffe6e6; border: 1px solid #d63638; border-radius: 4px; color: #d63638;">' .
            __('Managed table not found or not published.', 'geo-ranking-tables') .
            '</div>';
    }

    $renderer = new GRT_TableRenderer();
    return $renderer->render($config, $overrides);
}
