<?php
/**
 * Click Tracker Class
 * Handles click tracking for affiliate links and review links
 */

if (!defined('ABSPATH')) {
    exit;
}

class GRT_ClickTracker {

    private $table_name;

    public function __construct() {
        global $wpdb;
        $this->table_name = $wpdb->prefix . 'grt_click_tracking';

        // Register AJAX handlers
        add_action('wp_ajax_grt_track_click', array($this, 'ajax_track_click'));
        add_action('wp_ajax_nopriv_grt_track_click', array($this, 'ajax_track_click'));
    }

    /**
     * Create the click tracking table on plugin activation
     */
    public static function create_table() {
        global $wpdb;

        $table_name = $wpdb->prefix . 'grt_click_tracking';
        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE $table_name (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            site_id BIGINT(20) UNSIGNED NOT NULL,
            table_id BIGINT(20) UNSIGNED DEFAULT NULL,
            page_id BIGINT(20) UNSIGNED DEFAULT NULL,
            click_type VARCHAR(20) NOT NULL,
            rank_position TINYINT UNSIGNED DEFAULT NULL,
            user_state VARCHAR(5) DEFAULT NULL,
            device_type VARCHAR(20) DEFAULT 'desktop',
            theme VARCHAR(50) DEFAULT 'default',
            referrer_url VARCHAR(500) DEFAULT NULL,
            ip_hash VARCHAR(64) DEFAULT NULL,
            session_id VARCHAR(64) DEFAULT NULL,
            created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY idx_site_id (site_id),
            KEY idx_table_id (table_id),
            KEY idx_created_at (created_at),
            KEY idx_user_state (user_state),
            KEY idx_click_type (click_type),
            KEY idx_site_date (site_id, created_at)
        ) $charset_collate;";

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }

    /**
     * AJAX handler for tracking clicks
     */
    public function ajax_track_click() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'grt_nonce')) {
            wp_send_json_error('Security check failed');
            return;
        }

        $data = array(
            'site_id'       => isset($_POST['site_id']) ? intval($_POST['site_id']) : 0,
            'table_id'      => isset($_POST['table_id']) ? intval($_POST['table_id']) : 0,
            'page_id'       => isset($_POST['page_id']) ? intval($_POST['page_id']) : 0,
            'click_type'    => isset($_POST['click_type']) ? sanitize_text_field($_POST['click_type']) : '',
            'rank_position' => isset($_POST['rank']) ? intval($_POST['rank']) : 0,
            'user_state'    => isset($_POST['user_state']) ? sanitize_text_field($_POST['user_state']) : '',
            'theme'         => isset($_POST['theme']) ? sanitize_text_field($_POST['theme']) : 'default',
            'referrer_url'  => isset($_POST['url']) ? esc_url_raw($_POST['url']) : ''
        );

        $result = $this->record_click($data);

        // Return minimal response for performance
        wp_send_json_success(array('recorded' => $result));
    }

    /**
     * Record a click to the database
     *
     * @param array $data Click data
     * @return bool Success status
     */
    public function record_click($data) {
        global $wpdb;

        // Validate required fields
        if (empty($data['site_id']) || empty($data['click_type'])) {
            return false;
        }

        // Validate click type
        $valid_types = array('visit', 'review', 'overlay');
        if (!in_array($data['click_type'], $valid_types)) {
            return false;
        }

        $insert_data = array(
            'site_id'       => $data['site_id'],
            'table_id'      => !empty($data['table_id']) ? $data['table_id'] : null,
            'page_id'       => !empty($data['page_id']) ? $data['page_id'] : null,
            'click_type'    => $data['click_type'],
            'rank_position' => !empty($data['rank_position']) ? $data['rank_position'] : null,
            'user_state'    => !empty($data['user_state']) ? strtoupper(substr($data['user_state'], 0, 5)) : null,
            'device_type'   => $this->detect_device_type(),
            'theme'         => !empty($data['theme']) ? substr($data['theme'], 0, 50) : 'default',
            'referrer_url'  => !empty($data['referrer_url']) ? substr($data['referrer_url'], 0, 500) : null,
            'ip_hash'       => $this->hash_ip($this->get_user_ip()),
            'session_id'    => $this->get_session_id(),
            'created_at'    => current_time('mysql')
        );

        $result = $wpdb->insert($this->table_name, $insert_data);

        return $result !== false;
    }

    /**
     * Get or create anonymous session ID from cookie
     *
     * @return string Session ID
     */
    private function get_session_id() {
        $cookie_name = 'grt_session';

        if (isset($_COOKIE[$cookie_name])) {
            return sanitize_text_field($_COOKIE[$cookie_name]);
        }

        // Generate new session ID
        $session_id = hash('sha256', uniqid(mt_rand(), true) . time());

        // Set cookie for 30 days (won't work in AJAX, but will be set on next page load)
        if (!headers_sent()) {
            setcookie($cookie_name, $session_id, time() + (86400 * 30), '/');
        }

        return $session_id;
    }

    /**
     * Detect device type from user agent
     *
     * @return string Device type (desktop, mobile, tablet)
     */
    private function detect_device_type() {
        $user_agent = isset($_SERVER['HTTP_USER_AGENT']) ? $_SERVER['HTTP_USER_AGENT'] : '';

        // Check for tablet first (before mobile check)
        if (preg_match('/tablet|ipad|playbook|silk/i', $user_agent)) {
            return 'tablet';
        }

        // Check for mobile
        if (preg_match('/mobile|android|iphone|ipod|blackberry|opera mini|iemobile|windows phone/i', $user_agent)) {
            return 'mobile';
        }

        return 'desktop';
    }

    /**
     * Hash IP address for privacy-safe storage
     *
     * @param string $ip IP address
     * @return string Hashed IP
     */
    private function hash_ip($ip) {
        // Add salt for security
        $salt = defined('AUTH_SALT') ? AUTH_SALT : 'grt_default_salt';
        return hash('sha256', $ip . $salt);
    }

    /**
     * Get user IP address
     *
     * @return string IP address
     */
    private function get_user_ip() {
        $ip_headers = array(
            'HTTP_CF_CONNECTING_IP',  // Cloudflare
            'HTTP_CLIENT_IP',
            'HTTP_X_FORWARDED_FOR',
            'REMOTE_ADDR'
        );

        foreach ($ip_headers as $header) {
            if (!empty($_SERVER[$header])) {
                $ip = $_SERVER[$header];
                // Handle comma-separated IPs (X-Forwarded-For can have multiple)
                if (strpos($ip, ',') !== false) {
                    $ip_list = explode(',', $ip);
                    $ip = trim($ip_list[0]);
                }
                if (filter_var($ip, FILTER_VALIDATE_IP)) {
                    return $ip;
                }
            }
        }

        return '127.0.0.1';
    }

    /**
     * Get the database table name
     *
     * @return string Table name
     */
    public function get_table_name() {
        return $this->table_name;
    }
}
