<?php
/**
 * Geo Detection Class
 * Handles user location detection for geo-filtering
 */

if (!defined('ABSPATH')) {
    exit;
}

class GRT_GeoDetector {
    
    private $api_services = array(
        'ipapi' => 'http://ip-api.com/json/',
        'ipinfo' => 'https://ipinfo.io/json',
        'ipgeolocation' => 'https://api.ipgeolocation.io/ipgeo',
    );
    
    /**
     * Get user's location based on IP address
     */
    public function get_user_location($ip_address = null) {
        if (!$ip_address) {
            $ip_address = $this->get_user_ip();
        }
        
        // Check for forced test mode IP from URL parameter
        $forced_ip = $this->get_forced_test_ip();
        if ($forced_ip) {
            $ip_address = $forced_ip;
            error_log('GRT: Using forced test IP: ' . $ip_address);
        }
        
        // Don't process local/private IPs unless forced
        if ($this->is_private_ip($ip_address) && !$forced_ip) {
            return $this->get_fallback_location();
        }
        
        // Try to get from cache first (include forced IP in cache key)
        $cache_key = 'grt_geo_' . md5($ip_address . ($forced_ip ? '_forced' : ''));
        $cached_result = get_transient($cache_key);
        
        if ($cached_result !== false && !$forced_ip) {
            // Don't use cache for forced IPs to allow real-time testing
            return $cached_result;
        }
        
        // Try different API services
        foreach ($this->api_services as $service => $base_url) {
            $location = $this->query_geo_service($service, $base_url, $ip_address);
            
            if ($location && isset($location['state_code'])) {
                // Add forced IP indicator to location data
                if ($forced_ip) {
                    $location['forced_ip'] = true;
                    $location['original_ip'] = $this->get_user_ip();
                }
                
                // Cache the result for 1 hour (shorter cache for forced IPs)
                $cache_duration = $forced_ip ? 300 : HOUR_IN_SECONDS; // 5 minutes for forced, 1 hour for normal
                set_transient($cache_key, $location, $cache_duration);
                return $location;
            }
        }
        
        // Fallback if all services fail
        $fallback = $this->get_fallback_location();
        if ($forced_ip) {
            $fallback['forced_ip'] = true;
            $fallback['original_ip'] = $this->get_user_ip();
            $fallback['ip'] = $ip_address;
        }
        return $fallback;
    }
    
    /**
     * Get forced test IP from URL parameter
     */
    private function get_forced_test_ip() {
        // Check if ftm-ip parameter is present
        if (!isset($_GET['ftm-ip'])) {
            return false;
        }
        
        $test_ip = sanitize_text_field($_GET['ftm-ip']);
        
        // Validate IPv4 or IPv6 address
        if (filter_var($test_ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4 | FILTER_FLAG_IPV6)) {
            return $test_ip;
        }
        
        // Log invalid IP attempts
        error_log('GRT: Invalid ftm-ip parameter provided: ' . $test_ip);
        return false;
    }
    
    /**
     * Get user's IP address
     */
    private function get_user_ip() {
        // Check for various headers that might contain the real IP
        $ip_headers = array(
            'HTTP_CF_CONNECTING_IP',     // Cloudflare
            'HTTP_CLIENT_IP',            // Proxy
            'HTTP_X_FORWARDED_FOR',      // Load balancer/proxy
            'HTTP_X_FORWARDED',          // Proxy
            'HTTP_X_CLUSTER_CLIENT_IP',  // Cluster
            'HTTP_FORWARDED_FOR',        // Proxy
            'HTTP_FORWARDED',            // Proxy
            'REMOTE_ADDR'                // Standard
        );
        
        foreach ($ip_headers as $header) {
            if (!empty($_SERVER[$header])) {
                $ip = $_SERVER[$header];
                
                // Handle comma-separated IPs (X-Forwarded-For can contain multiple IPs)
                if (strpos($ip, ',') !== false) {
                    $ip_list = explode(',', $ip);
                    $ip = trim($ip_list[0]);
                }
                
                // Validate IP address
                if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
                    return $ip;
                }
            }
        }
        
        // Fallback to REMOTE_ADDR even if it's private (for local testing)
        return isset($_SERVER['REMOTE_ADDR']) ? $_SERVER['REMOTE_ADDR'] : '127.0.0.1';
    }
    
    /**
     * Check if IP is private/local
     */
    private function is_private_ip($ip) {
        return !filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE);
    }
    
    /**
     * Query a geo location service
     */
    private function query_geo_service($service, $base_url, $ip_address) {
        $url = $this->build_service_url($service, $base_url, $ip_address);
        
        if (!$url) {
            return false;
        }
        
        $response = wp_remote_get($url, array(
            'timeout' => 5,
            'headers' => array(
                'User-Agent' => 'WordPress GeoRankingTables Plugin'
            )
        ));
        
        if (is_wp_error($response) || wp_remote_retrieve_response_code($response) !== 200) {
            return false;
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if (!$data) {
            return false;
        }
        
        return $this->normalize_location_data($service, $data);
    }
    
    /**
     * Build service-specific URL
     */
    private function build_service_url($service, $base_url, $ip_address) {
        switch ($service) {
            case 'ipapi':
                return $base_url . $ip_address . '?fields=status,country,countryCode,region,regionName,city,lat,lon';
                
            case 'ipinfo':
                return $base_url . $ip_address;
                
            case 'ipgeolocation':
                $api_key = get_option('grt_ipgeolocation_api_key', '');
                if (empty($api_key)) {
                    return false;
                }
                return $base_url . '?apiKey=' . $api_key . '&ip=' . $ip_address;
                
            default:
                return false;
        }
    }
    
    /**
     * Normalize location data from different services
     */
    private function normalize_location_data($service, $data) {
        $location = array(
            'ip' => '',
            'country' => '',
            'country_code' => '',
            'state' => '',
            'state_code' => '',
            'city' => '',
            'latitude' => 0,
            'longitude' => 0,
            'service' => $service
        );
        
        switch ($service) {
            case 'ipapi':
                if (isset($data['status']) && $data['status'] === 'success') {
                    $location['country'] = isset($data['country']) ? $data['country'] : '';
                    $location['country_code'] = isset($data['countryCode']) ? $data['countryCode'] : '';
                    $location['state'] = isset($data['regionName']) ? $data['regionName'] : '';
                    $location['state_code'] = isset($data['region']) ? $this->convert_state_name_to_code($data['region']) : '';
                    $location['city'] = isset($data['city']) ? $data['city'] : '';
                    $location['latitude'] = isset($data['lat']) ? $data['lat'] : 0;
                    $location['longitude'] = isset($data['lon']) ? $data['lon'] : 0;
                }
                break;
                
            case 'ipinfo':
                $location['country_code'] = isset($data['country']) ? $data['country'] : '';
                $location['state_code'] = isset($data['region']) ? $data['region'] : '';
                $location['city'] = isset($data['city']) ? $data['city'] : '';
                if (isset($data['loc'])) {
                    $coords = explode(',', $data['loc']);
                    $location['latitude'] = isset($coords[0]) ? floatval($coords[0]) : 0;
                    $location['longitude'] = isset($coords[1]) ? floatval($coords[1]) : 0;
                }
                break;
                
            case 'ipgeolocation':
                $location['country'] = isset($data['country_name']) ? $data['country_name'] : '';
                $location['country_code'] = isset($data['country_code2']) ? $data['country_code2'] : '';
                $location['state'] = isset($data['state_prov']) ? $data['state_prov'] : '';
                $location['state_code'] = isset($data['state_code']) ? $data['state_code'] : '';
                $location['city'] = isset($data['city']) ? $data['city'] : '';
                $location['latitude'] = isset($data['latitude']) ? floatval($data['latitude']) : 0;
                $location['longitude'] = isset($data['longitude']) ? floatval($data['longitude']) : 0;
                break;
        }
        
        // Ensure we have a state code for US locations
        if ($location['country_code'] === 'US' && empty($location['state_code']) && !empty($location['state'])) {
            $location['state_code'] = $this->convert_state_name_to_code($location['state']);
        }
        
        return $location;
    }
    
    /**
     * Convert full state name to 2-letter code
     */
    private function convert_state_name_to_code($state_name) {
        $states = array(
            'Alabama' => 'AL', 'Alaska' => 'AK', 'Arizona' => 'AZ', 'Arkansas' => 'AR', 'California' => 'CA',
            'Colorado' => 'CO', 'Connecticut' => 'CT', 'Delaware' => 'DE', 'Florida' => 'FL', 'Georgia' => 'GA',
            'Hawaii' => 'HI', 'Idaho' => 'ID', 'Illinois' => 'IL', 'Indiana' => 'IN', 'Iowa' => 'IA',
            'Kansas' => 'KS', 'Kentucky' => 'KY', 'Louisiana' => 'LA', 'Maine' => 'ME', 'Maryland' => 'MD',
            'Massachusetts' => 'MA', 'Michigan' => 'MI', 'Minnesota' => 'MN', 'Mississippi' => 'MS', 'Missouri' => 'MO',
            'Montana' => 'MT', 'Nebraska' => 'NE', 'Nevada' => 'NV', 'New Hampshire' => 'NH', 'New Jersey' => 'NJ',
            'New Mexico' => 'NM', 'New York' => 'NY', 'North Carolina' => 'NC', 'North Dakota' => 'ND', 'Ohio' => 'OH',
            'Oklahoma' => 'OK', 'Oregon' => 'OR', 'Pennsylvania' => 'PA', 'Rhode Island' => 'RI', 'South Carolina' => 'SC',
            'South Dakota' => 'SD', 'Tennessee' => 'TN', 'Texas' => 'TX', 'Utah' => 'UT', 'Vermont' => 'VT',
            'Virginia' => 'VA', 'Washington' => 'WA', 'West Virginia' => 'WV', 'Wisconsin' => 'WI', 'Wyoming' => 'WY',
            'District of Columbia' => 'DC'
        );
        
        return isset($states[$state_name]) ? $states[$state_name] : '';
    }
    
    /**
     * Get fallback location (for testing or when detection fails)
     */
    private function get_fallback_location() {
        return array(
            'ip' => '127.0.0.1',
            'country' => 'United States',
            'country_code' => 'US',
            'state' => 'New York',
            'state_code' => 'NY',
            'city' => 'New York',
            'latitude' => 40.7128,
            'longitude' => -74.0060,
            'service' => 'fallback'
        );
    }
    
    /**
     * AJAX endpoint for frontend geo detection
     */
    public static function ajax_detect_location() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'grt_nonce')) {
            wp_die('Security check failed');
        }
        
        $detector = new self();
        
        // Check if a forced IP is provided
        $forced_ip = null;
        if (isset($_POST['forced_ip']) && !empty($_POST['forced_ip'])) {
            $forced_ip = sanitize_text_field($_POST['forced_ip']);
            // Validate the forced IP
            if (!filter_var($forced_ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4 | FILTER_FLAG_IPV6)) {
                wp_send_json_error('Invalid forced IP address');
                return;
            }
        }
        
        $location = $detector->get_user_location($forced_ip);
        
        wp_send_json_success($location);
    }
}

// Register AJAX handlers
add_action('wp_ajax_grt_detect_location', array('GRT_GeoDetector', 'ajax_detect_location'));
add_action('wp_ajax_nopriv_grt_detect_location', array('GRT_GeoDetector', 'ajax_detect_location'));