<?php
/**
 * Import/Export Class
 * Handles import and export of site blocks and managed tables
 */

if (!defined('ABSPATH')) {
    exit;
}

class GRT_ImportExport {

    public function __construct() {
        add_action('admin_menu', array($this, 'add_submenu_page'), 15);
        add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('wp_ajax_grt_export_data', array($this, 'ajax_export'));
        add_action('wp_ajax_grt_import_preview', array($this, 'ajax_import_preview'));
        add_action('wp_ajax_grt_import_execute', array($this, 'ajax_import_execute'));
    }

    /**
     * Add Import/Export submenu page
     */
    public function add_submenu_page() {
        add_submenu_page(
            'geo-ranking-tables',
            __('Import/Export', 'geo-ranking-tables'),
            __('Import/Export', 'geo-ranking-tables'),
            'manage_options',
            'grt-import-export',
            array($this, 'render_page')
        );
    }

    /**
     * Enqueue scripts for import/export page
     */
    public function enqueue_scripts($hook) {
        // Not needed - script data is output directly in render_page()
    }

    /**
     * Render the Import/Export admin page
     */
    public function render_page() {
        // Get site blocks for export selection
        $site_blocks = get_posts(array(
            'post_type' => 'grt_site_block',
            'post_status' => 'publish',
            'numberposts' => -1,
            'orderby' => 'title',
            'order' => 'ASC',
        ));

        // Get managed tables for export selection
        $managed_tables = get_posts(array(
            'post_type' => 'grt_managed_table',
            'post_status' => 'publish',
            'numberposts' => -1,
            'orderby' => 'title',
            'order' => 'ASC',
        ));

        ?>
        <div class="wrap grt-import-export-container">
            <h1><?php _e('Import/Export', 'geo-ranking-tables'); ?></h1>
            <p class="description"><?php _e('Export your site blocks and managed tables to transfer them to another site, or import data from another installation.', 'geo-ranking-tables'); ?></p>

            <!-- Export Section -->
            <div class="grt-import-export-section grt-export-section">
                <h2><?php _e('Export', 'geo-ranking-tables'); ?></h2>
                <p><?php _e('Select the data you want to export. The export will be downloaded as a JSON file.', 'geo-ranking-tables'); ?></p>

                <form id="grt-export-form">
                    <div class="grt-export-options">
                        <div class="grt-export-type">
                            <h3><?php _e('Export Type', 'geo-ranking-tables'); ?></h3>
                            <label>
                                <input type="radio" name="export_type" value="full" checked>
                                <?php _e('Export All (Site Blocks + Managed Tables)', 'geo-ranking-tables'); ?>
                            </label><br>
                            <label>
                                <input type="radio" name="export_type" value="sites_only">
                                <?php _e('Site Blocks Only', 'geo-ranking-tables'); ?>
                            </label><br>
                            <label>
                                <input type="radio" name="export_type" value="tables_only">
                                <?php _e('Managed Tables Only', 'geo-ranking-tables'); ?>
                            </label><br>
                            <label>
                                <input type="radio" name="export_type" value="selected">
                                <?php _e('Selected Items Only', 'geo-ranking-tables'); ?>
                            </label>
                        </div>

                        <div class="grt-export-selection" style="display: none;">
                            <div class="grt-selection-columns">
                                <div class="grt-selection-column">
                                    <h4><?php _e('Site Blocks', 'geo-ranking-tables'); ?></h4>
                                    <div class="grt-select-actions">
                                        <button type="button" class="button button-small grt-select-all-sites"><?php _e('Select All', 'geo-ranking-tables'); ?></button>
                                        <button type="button" class="button button-small grt-deselect-all-sites"><?php _e('Deselect All', 'geo-ranking-tables'); ?></button>
                                    </div>
                                    <div class="grt-item-selector">
                                        <?php if (empty($site_blocks)): ?>
                                            <p class="description"><?php _e('No site blocks found.', 'geo-ranking-tables'); ?></p>
                                        <?php else: ?>
                                            <?php foreach ($site_blocks as $site): ?>
                                                <label class="grt-item-checkbox">
                                                    <input type="checkbox" name="selected_sites[]" value="<?php echo esc_attr($site->ID); ?>">
                                                    <?php echo esc_html($site->post_title); ?>
                                                </label>
                                            <?php endforeach; ?>
                                        <?php endif; ?>
                                    </div>
                                </div>

                                <div class="grt-selection-column">
                                    <h4><?php _e('Managed Tables', 'geo-ranking-tables'); ?></h4>
                                    <div class="grt-select-actions">
                                        <button type="button" class="button button-small grt-select-all-tables"><?php _e('Select All', 'geo-ranking-tables'); ?></button>
                                        <button type="button" class="button button-small grt-deselect-all-tables"><?php _e('Deselect All', 'geo-ranking-tables'); ?></button>
                                    </div>
                                    <div class="grt-item-selector">
                                        <?php if (empty($managed_tables)): ?>
                                            <p class="description"><?php _e('No managed tables found.', 'geo-ranking-tables'); ?></p>
                                        <?php else: ?>
                                            <?php foreach ($managed_tables as $table): ?>
                                                <label class="grt-item-checkbox">
                                                    <input type="checkbox" name="selected_tables[]" value="<?php echo esc_attr($table->ID); ?>">
                                                    <?php echo esc_html($table->post_title); ?>
                                                </label>
                                            <?php endforeach; ?>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="grt-export-summary">
                        <p>
                            <strong><?php _e('Available for export:', 'geo-ranking-tables'); ?></strong>
                            <?php echo count($site_blocks); ?> <?php _e('site blocks', 'geo-ranking-tables'); ?>,
                            <?php echo count($managed_tables); ?> <?php _e('managed tables', 'geo-ranking-tables'); ?>
                        </p>
                    </div>

                    <p class="submit">
                        <button type="submit" class="button button-primary" id="grt-export-btn">
                            <?php _e('Export to JSON', 'geo-ranking-tables'); ?>
                        </button>
                        <span class="spinner" style="float: none; margin-left: 10px;"></span>
                    </p>
                </form>
            </div>

            <!-- Import Section -->
            <div class="grt-import-export-section grt-import-section">
                <h2><?php _e('Import', 'geo-ranking-tables'); ?></h2>
                <p><?php _e('Import site blocks and managed tables from a JSON export file.', 'geo-ranking-tables'); ?></p>

                <form id="grt-import-form" enctype="multipart/form-data">
                    <div class="grt-dropzone" id="grt-import-dropzone">
                        <div class="grt-dropzone-content">
                            <span class="dashicons dashicons-upload"></span>
                            <p><?php _e('Drag & drop your JSON file here', 'geo-ranking-tables'); ?></p>
                            <p class="description"><?php _e('or', 'geo-ranking-tables'); ?></p>
                            <label class="button" for="grt-import-file"><?php _e('Browse Files', 'geo-ranking-tables'); ?></label>
                            <input type="file" id="grt-import-file" name="import_file" accept=".json" style="display: none;">
                        </div>
                        <div class="grt-dropzone-file" style="display: none;">
                            <span class="dashicons dashicons-media-code"></span>
                            <span class="grt-filename"></span>
                            <button type="button" class="grt-remove-file">&times;</button>
                        </div>
                    </div>

                    <div class="grt-import-preview" id="grt-import-preview" style="display: none;">
                        <h3><?php _e('Import Preview', 'geo-ranking-tables'); ?></h3>
                        <div class="grt-preview-content"></div>

                        <div class="grt-duplicate-options">
                            <h4><?php _e('Duplicate Handling', 'geo-ranking-tables'); ?></h4>
                            <p class="description"><?php _e('Choose how to handle items that already exist (matched by slug or title).', 'geo-ranking-tables'); ?></p>
                            <label>
                                <input type="radio" name="duplicate_handling" value="skip" checked>
                                <?php _e('Skip duplicates', 'geo-ranking-tables'); ?>
                            </label><br>
                            <label>
                                <input type="radio" name="duplicate_handling" value="overwrite">
                                <?php _e('Overwrite existing', 'geo-ranking-tables'); ?>
                            </label><br>
                            <label>
                                <input type="radio" name="duplicate_handling" value="create_new">
                                <?php _e('Create as new (with modified slug)', 'geo-ranking-tables'); ?>
                            </label>
                        </div>
                    </div>

                    <div class="grt-import-results" id="grt-import-results" style="display: none;">
                        <h3><?php _e('Import Results', 'geo-ranking-tables'); ?></h3>
                        <div class="grt-results-content"></div>
                    </div>

                    <p class="submit">
                        <button type="submit" class="button button-primary" id="grt-import-btn" disabled>
                            <?php _e('Import Data', 'geo-ranking-tables'); ?>
                        </button>
                        <span class="spinner" style="float: none; margin-left: 10px;"></span>
                    </p>
                </form>
            </div>
        </div>

        <script type="text/javascript">
        var grtImportExport = <?php echo wp_json_encode(array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('grt_import_export_nonce'),
            'strings' => array(
                'exporting' => __('Exporting...', 'geo-ranking-tables'),
                'importing' => __('Importing...', 'geo-ranking-tables'),
                'parsing' => __('Parsing file...', 'geo-ranking-tables'),
                'success' => __('Success!', 'geo-ranking-tables'),
                'error' => __('An error occurred.', 'geo-ranking-tables'),
                'selectFile' => __('Please select a file to import.', 'geo-ranking-tables'),
                'invalidJson' => __('Invalid JSON file.', 'geo-ranking-tables'),
                'confirmImport' => __('Are you sure you want to import this data?', 'geo-ranking-tables'),
            ),
        )); ?>;
        </script>
        <?php
    }

    /**
     * AJAX handler for export
     */
    public function ajax_export() {
        check_ajax_referer('grt_import_export_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('Permission denied.', 'geo-ranking-tables'));
        }

        $export_type = isset($_POST['export_type']) ? sanitize_text_field($_POST['export_type']) : 'full';
        $selected_sites = isset($_POST['selected_sites']) ? array_map('intval', $_POST['selected_sites']) : array();
        $selected_tables = isset($_POST['selected_tables']) ? array_map('intval', $_POST['selected_tables']) : array();

        $exporter = new GRT_Exporter();
        $exporter->set_options(array(
            'export_type' => $export_type,
            'selected_sites' => $selected_sites,
            'selected_tables' => $selected_tables,
        ));

        $data = $exporter->generate_export_data();

        if (is_wp_error($data)) {
            wp_send_json_error($data->get_error_message());
        }

        $filename = 'grt-export-' . date('Y-m-d-His') . '.json';

        wp_send_json_success(array(
            'filename' => $filename,
            'content' => $data,
        ));
    }

    /**
     * AJAX handler for import preview
     */
    public function ajax_import_preview() {
        check_ajax_referer('grt_import_export_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('Permission denied.', 'geo-ranking-tables'));
        }

        $json_content = isset($_POST['json_content']) ? wp_unslash($_POST['json_content']) : '';

        if (empty($json_content)) {
            wp_send_json_error(__('No data provided.', 'geo-ranking-tables'));
        }

        $importer = new GRT_Importer();
        $parsed = $importer->parse_json($json_content);

        if (is_wp_error($parsed)) {
            wp_send_json_error($parsed->get_error_message());
        }

        $preview = $importer->generate_preview($parsed);

        if (is_wp_error($preview)) {
            wp_send_json_error($preview->get_error_message());
        }

        wp_send_json_success($preview);
    }

    /**
     * AJAX handler for import execution
     */
    public function ajax_import_execute() {
        check_ajax_referer('grt_import_export_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('Permission denied.', 'geo-ranking-tables'));
        }

        $json_content = isset($_POST['json_content']) ? wp_unslash($_POST['json_content']) : '';
        $duplicate_handling = isset($_POST['duplicate_handling']) ? sanitize_text_field($_POST['duplicate_handling']) : 'skip';

        if (empty($json_content)) {
            wp_send_json_error(__('No data provided.', 'geo-ranking-tables'));
        }

        $importer = new GRT_Importer();
        $parsed = $importer->parse_json($json_content);

        if (is_wp_error($parsed)) {
            wp_send_json_error($parsed->get_error_message());
        }

        $result = $importer->execute_import($parsed, array(
            'duplicate_handling' => $duplicate_handling,
        ));

        if (is_wp_error($result)) {
            wp_send_json_error($result->get_error_message());
        }

        wp_send_json_success($result);
    }
}
