// Admin JavaScript for Geo Ranking Tables
/* global jQuery */
(function($) {
    'use strict';
    
    $(document).ready(function() {
        initializeAdmin();
    });
    
    function initializeAdmin() {
        // Initialize select all/deselect all for states
        initializeStateSelectors();

        // Initialize admin dashboard functionality
        initializeDashboard();

        // Initialize settings page functionality
        initializeSettings();

        // Initialize import/export functionality
        initializeImportExport();
    }
    
    function initializeStateSelectors() {
        // Select all states button
        $(document).on('click', '#grt-select-all-states', function(e) {
            e.preventDefault();
            $('.grt-states-grid input[type="checkbox"]').prop('checked', true);
            updateStateCount();
        });
        
        // Deselect all states button
        $(document).on('click', '#grt-deselect-all-states', function(e) {
            e.preventDefault();
            $('.grt-states-grid input[type="checkbox"]').prop('checked', false);
            updateStateCount();
        });
        
        // Update count when individual checkboxes change
        $(document).on('change', '.grt-states-grid input[type="checkbox"]', function() {
            updateStateCount();
        });
        
        // Initial count update
        updateStateCount();
    }
    
    function updateStateCount() {
        var totalStates = $('.grt-states-grid input[type="checkbox"]').length;
        var selectedStates = $('.grt-states-grid input[type="checkbox"]:checked').length;
        var $counter = $('.grt-states-counter');
        
        if ($counter.length === 0) {
            $counter = $('<div class="grt-states-counter" style="margin-top: 10px; font-weight: bold; color: #0073aa;"></div>');
            $('.grt-select-all-controls').after($counter);
        }
        
        $counter.text(selectedStates + ' of ' + totalStates + ' states selected');
        
        // Update visual feedback
        if (selectedStates === 0) {
            $counter.css('color', '#d63638');
        } else if (selectedStates === totalStates) {
            $counter.css('color', '#00a32a');
        } else {
            $counter.css('color', '#0073aa');
        }
    }
    
    function initializeDashboard() {
        // Add hover effects to stat cards
        $('.grt-stat-card').hover(
            function() {
                $(this).css('transform', 'translateY(-2px)');
            },
            function() {
                $(this).css('transform', 'translateY(0)');
            }
        );
        
        // Add click handlers for quick action buttons
        $('.grt-action-buttons .button').on('click', function() {
            var $btn = $(this);
            $btn.addClass('loading');
            
            // Remove loading class after navigation
            setTimeout(function() {
                $btn.removeClass('loading');
            }, 1000);
        });
        
        // Initialize sample data buttons
        initializeSampleDataButtons();
    }
    
    function initializeSampleDataButtons() {
        // Create sample data button
        $(document).on('click', '#grt-create-sample-data', function(e) {
            e.preventDefault();
            createSampleData($(this));
        });
        
        // Reset sample data button
        $(document).on('click', '#grt-reset-sample-data', function(e) {
            e.preventDefault();
            resetSampleData($(this));
        });
    }
    
    function createSampleData($btn) {
        // Disable button and show loading state
        $btn.prop('disabled', true).addClass('grt-loading');
        var originalText = $btn.html();
        $btn.html('<span class="dashicons dashicons-update grt-spin" style="margin-top: 3px;"></span> ' + 
                  (typeof grtAdminAjax !== 'undefined' ? grtAdminAjax.strings.creating : 'Creating...'));
        
        // Make AJAX request
        $.ajax({
            url: typeof grtAdminAjax !== 'undefined' ? grtAdminAjax.ajaxUrl : ajaxurl,
            type: 'POST',
            data: {
                action: 'grt_create_sample_data',
                nonce: typeof grtAdminAjax !== 'undefined' ? grtAdminAjax.nonce : ''
            },
            success: function(response) {
                if (response.success) {
                    // Show success message
                    showAdminNotice('success', response.data.message);
                    
                    // Reload page after 1 second
                    setTimeout(function() {
                        window.location.reload();
                    }, 1000);
                } else {
                    // Show error message
                    showAdminNotice('error', response.data.message);
                    $btn.prop('disabled', false).removeClass('grt-loading').html(originalText);
                }
            },
            error: function(xhr, status, error) {
                showAdminNotice('error', 'An error occurred while creating sample data. Please try again.');
                $btn.prop('disabled', false).removeClass('grt-loading').html(originalText);
            }
        });
    }
    
    function resetSampleData($btn) {
        // Confirm deletion
        var confirmMessage = typeof grtAdminAjax !== 'undefined' ? 
            grtAdminAjax.strings.confirmReset : 
            'Are you sure you want to delete all sample data and recreate it?';
        
        if (!confirm(confirmMessage)) {
            return;
        }
        
        // Disable button and show loading state
        $btn.prop('disabled', true).addClass('grt-loading');
        var originalText = $btn.html();
        $btn.html('<span class="dashicons dashicons-update grt-spin" style="margin-top: 3px;"></span> ' + 
                  (typeof grtAdminAjax !== 'undefined' ? grtAdminAjax.strings.deleting : 'Deleting...'));
        
        // Make AJAX request to delete
        $.ajax({
            url: typeof grtAdminAjax !== 'undefined' ? grtAdminAjax.ajaxUrl : ajaxurl,
            type: 'POST',
            data: {
                action: 'grt_delete_sample_data',
                nonce: typeof grtAdminAjax !== 'undefined' ? grtAdminAjax.nonce : ''
            },
            success: function(response) {
                if (response.success) {
                    // Show success message
                    showAdminNotice('success', response.data.message);
                    
                    // Now create new sample data
                    $btn.html('<span class="dashicons dashicons-update grt-spin" style="margin-top: 3px;"></span> ' + 
                              (typeof grtAdminAjax !== 'undefined' ? grtAdminAjax.strings.creating : 'Creating...'));
                    
                    // Small delay before creating
                    setTimeout(function() {
                        createSampleDataAfterDelete($btn, originalText);
                    }, 500);
                } else {
                    // Show error message
                    showAdminNotice('error', response.data.message);
                    $btn.prop('disabled', false).removeClass('grt-loading').html(originalText);
                }
            },
            error: function(xhr, status, error) {
                showAdminNotice('error', 'An error occurred while deleting sample data. Please try again.');
                $btn.prop('disabled', false).removeClass('grt-loading').html(originalText);
            }
        });
    }
    
    function createSampleDataAfterDelete($btn, originalText) {
        // Make AJAX request to create
        $.ajax({
            url: typeof grtAdminAjax !== 'undefined' ? grtAdminAjax.ajaxUrl : ajaxurl,
            type: 'POST',
            data: {
                action: 'grt_create_sample_data',
                nonce: typeof grtAdminAjax !== 'undefined' ? grtAdminAjax.nonce : ''
            },
            success: function(response) {
                if (response.success) {
                    // Show success message
                    showAdminNotice('success', response.data.message);
                    
                    // Reload page after 1 second
                    setTimeout(function() {
                        window.location.reload();
                    }, 1000);
                } else {
                    // Show error message
                    showAdminNotice('error', response.data.message);
                    $btn.prop('disabled', false).removeClass('grt-loading').html(originalText);
                }
            },
            error: function(xhr, status, error) {
                showAdminNotice('error', 'An error occurred while creating sample data. Please try again.');
                $btn.prop('disabled', false).removeClass('grt-loading').html(originalText);
            }
        });
    }
    
    function showAdminNotice(type, message) {
        // Remove existing notices
        $('.grt-admin-notice').remove();
        
        // Create notice
        var noticeClass = type === 'success' ? 'notice-success' : 'notice-error';
        var $notice = $('<div class="notice ' + noticeClass + ' is-dismissible grt-admin-notice"><p>' + message + '</p></div>');
        
        // Insert notice
        if ($('.wrap h1').length > 0) {
            $('.wrap h1').after($notice);
        } else {
            $('.wrap').prepend($notice);
        }
        
        // Initialize dismiss button
        $notice.on('click', '.notice-dismiss', function() {
            $notice.fadeOut(300, function() {
                $(this).remove();
            });
        });
        
        // Scroll to top
        $('html, body').animate({ scrollTop: 0 }, 300);
    }
    
    function initializeSettings() {
        // Settings form validation
        $('form[action="options.php"]').on('submit', function(e) {
            var isValid = validateSettings();
            
            if (!isValid) {
                e.preventDefault();
                showSettingsError('Please correct the errors before saving.');
            }
        });
        
        // Cache duration validation
        $('input[name="grt_settings[cache_duration]"]').on('blur', function() {
            var value = parseInt($(this).val());
            var $field = $(this);
            
            if (isNaN(value) || value < 300 || value > 86400) {
                $field.addClass('error');
                showFieldError($field, 'Cache duration must be between 300 and 86400 seconds');
            } else {
                $field.removeClass('error');
                hideFieldError($field);
            }
        });
        
        // API key validation
        $('input[name="grt_settings[geo_api_key]"]').on('blur', function() {
            var value = $(this).val().trim();
            var $field = $(this);
            
            if (value && value.length < 10) {
                $field.addClass('error');
                showFieldError($field, 'API key appears to be too short');
            } else {
                $field.removeClass('error');
                hideFieldError($field);
            }
        });
    }
    
    function validateSettings() {
        var isValid = true;
        
        // Validate cache duration
        var cacheDuration = parseInt($('input[name="grt_settings[cache_duration]"]').val());
        if (isNaN(cacheDuration) || cacheDuration < 300 || cacheDuration > 86400) {
            isValid = false;
        }
        
        return isValid;
    }
    
    function showSettingsError(message) {
        var $notice = $('.grt-settings-error');
        
        if ($notice.length === 0) {
            $notice = $('<div class="notice notice-error grt-settings-error"><p></p></div>');
            $('h1').after($notice);
        }
        
        $notice.find('p').text(message);
        $notice.show();
        
        // Scroll to top to show error
        $('html, body').animate({ scrollTop: 0 }, 300);
    }
    
    function showFieldError($field, message) {
        var $error = $field.siblings('.grt-field-error');
        
        if ($error.length === 0) {
            $error = $('<div class="grt-field-error" style="color: #d63638; font-size: 12px; margin-top: 4px;"></div>');
            $field.after($error);
        }
        
        $error.text(message).show();
    }
    
    function hideFieldError($field) {
        $field.siblings('.grt-field-error').hide();
    }
    
    // Site blocks list table enhancements
    function enhanceSiteBlocksList() {
        // Add quick actions
        $('.wp-list-table .row-actions').each(function() {
            var $actions = $(this);
            var postId = $actions.closest('tr').attr('id').replace('post-', '');
            
            // Add quick duplicate action
            if ($actions.find('.duplicate').length === 0) {
                $actions.append(' | <span class="duplicate"><a href="#" data-post-id="' + postId + '">Duplicate</a></span>');
            }
        });
        
        // Handle duplicate clicks
        $(document).on('click', '.row-actions .duplicate a', function(e) {
            e.preventDefault();
            var postId = $(this).data('post-id');
            duplicateSiteBlock(postId);
        });
    }
    
    function duplicateSiteBlock(postId) {
        if (!confirm('Are you sure you want to duplicate this site block?')) {
            return;
        }
        
        // This would require an AJAX endpoint to handle duplication
        console.log('Duplicating site block:', postId);
        // Implementation would go here
    }
    
    // Initialize site blocks list enhancements if on the right page
    if (window.location.href.indexOf('post_type=grt_site_block') !== -1) {
        enhanceSiteBlocksList();
    }
    
    // Add CSS for loading states
    $('<style>')
        .prop('type', 'text/css')
        .html(
            '.button.loading { opacity: 0.6; pointer-events: none; }' +
            '.button.loading::after { content: "..."; }' +
            'input.error { border-color: #d63638 !important; box-shadow: 0 0 0 1px #d63638; }' +
            '.grt-stat-card { transition: transform 0.2s ease; }' +
            '.grt-field-error { animation: grt-shake 0.5s; }' +
            '@keyframes grt-shake { 0%, 100% { transform: translateX(0); } 25% { transform: translateX(-5px); } 75% { transform: translateX(5px); } }'
        )
        .appendTo('head');

    // ===== Import/Export Functionality =====

    var importJsonContent = null;

    function initializeImportExport() {
        // Only initialize on import/export page
        if ($('.grt-import-export-container').length === 0) {
            return;
        }

        // Export type toggle
        $('input[name="export_type"]').on('change', function() {
            var value = $(this).val();
            if (value === 'selected') {
                $('.grt-export-selection').slideDown();
            } else {
                $('.grt-export-selection').slideUp();
            }
        });

        // Export selection buttons
        $('.grt-select-all-sites').on('click', function(e) {
            e.preventDefault();
            $('input[name="selected_sites[]"]').prop('checked', true);
        });

        $('.grt-deselect-all-sites').on('click', function(e) {
            e.preventDefault();
            $('input[name="selected_sites[]"]').prop('checked', false);
        });

        $('.grt-select-all-tables').on('click', function(e) {
            e.preventDefault();
            $('input[name="selected_tables[]"]').prop('checked', true);
        });

        $('.grt-deselect-all-tables').on('click', function(e) {
            e.preventDefault();
            $('input[name="selected_tables[]"]').prop('checked', false);
        });

        // Export form submission
        $('#grt-export-form').on('submit', function(e) {
            e.preventDefault();
            handleExport();
        });

        // Import file handling
        $('#grt-import-file').on('change', function(e) {
            var file = e.target.files[0];
            if (file) {
                handleFileSelect(file);
            }
        });

        // Drag and drop handling
        var $dropzone = $('#grt-import-dropzone');

        $dropzone.on('dragover', function(e) {
            e.preventDefault();
            e.stopPropagation();
            $(this).addClass('dragover');
        });

        $dropzone.on('dragleave', function(e) {
            e.preventDefault();
            e.stopPropagation();
            $(this).removeClass('dragover');
        });

        $dropzone.on('drop', function(e) {
            e.preventDefault();
            e.stopPropagation();
            $(this).removeClass('dragover');

            var files = e.originalEvent.dataTransfer.files;
            if (files.length > 0) {
                handleFileSelect(files[0]);
            }
        });

        // Remove file button
        $(document).on('click', '.grt-remove-file', function(e) {
            e.preventDefault();
            clearImportFile();
        });

        // Import form submission
        $('#grt-import-form').on('submit', function(e) {
            e.preventDefault();
            handleImport();
        });
    }

    function handleExport() {
        var $btn = $('#grt-export-btn');
        var $spinner = $btn.siblings('.spinner');

        $btn.prop('disabled', true);
        $spinner.addClass('is-active');

        var formData = {
            action: 'grt_export_data',
            nonce: typeof grtImportExport !== 'undefined' ? grtImportExport.nonce : '',
            export_type: $('input[name="export_type"]:checked').val(),
            selected_sites: [],
            selected_tables: []
        };

        // Get selected items if export type is 'selected'
        if (formData.export_type === 'selected') {
            $('input[name="selected_sites[]"]:checked').each(function() {
                formData.selected_sites.push($(this).val());
            });
            $('input[name="selected_tables[]"]:checked').each(function() {
                formData.selected_tables.push($(this).val());
            });
        }

        $.ajax({
            url: typeof grtImportExport !== 'undefined' ? grtImportExport.ajaxUrl : ajaxurl,
            type: 'POST',
            data: formData,
            success: function(response) {
                $btn.prop('disabled', false);
                $spinner.removeClass('is-active');

                if (response.success) {
                    // Trigger download
                    downloadJson(response.data.filename, response.data.content);
                    showAdminNotice('success', 'Export completed successfully!');
                } else {
                    showAdminNotice('error', response.data || 'Export failed.');
                }
            },
            error: function() {
                $btn.prop('disabled', false);
                $spinner.removeClass('is-active');
                showAdminNotice('error', 'An error occurred during export.');
            }
        });
    }

    function downloadJson(filename, content) {
        var jsonStr = JSON.stringify(content, null, 2);
        var blob = new Blob([jsonStr], { type: 'application/json' });
        var url = URL.createObjectURL(blob);

        var a = document.createElement('a');
        a.href = url;
        a.download = filename;
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
        URL.revokeObjectURL(url);
    }

    function handleFileSelect(file) {
        // Validate file type
        if (!file.name.endsWith('.json')) {
            showAdminNotice('error', 'Please select a JSON file.');
            return;
        }

        // Update UI to show selected file
        $('.grt-dropzone-content').hide();
        $('.grt-dropzone-file').show();
        $('.grt-filename').text(file.name);

        // Read file content
        var reader = new FileReader();
        reader.onload = function(e) {
            importJsonContent = e.target.result;
            previewImport();
        };
        reader.onerror = function() {
            showAdminNotice('error', 'Failed to read file.');
            clearImportFile();
        };
        reader.readAsText(file);
    }

    function clearImportFile() {
        importJsonContent = null;
        $('#grt-import-file').val('');
        $('.grt-dropzone-content').show();
        $('.grt-dropzone-file').hide();
        $('#grt-import-preview').hide();
        $('#grt-import-results').hide();
        $('#grt-import-btn').prop('disabled', true);
    }

    function previewImport() {
        var $btn = $('#grt-import-btn');
        var $spinner = $btn.siblings('.spinner');
        var $preview = $('#grt-import-preview');

        $spinner.addClass('is-active');

        $.ajax({
            url: typeof grtImportExport !== 'undefined' ? grtImportExport.ajaxUrl : ajaxurl,
            type: 'POST',
            data: {
                action: 'grt_import_preview',
                nonce: typeof grtImportExport !== 'undefined' ? grtImportExport.nonce : '',
                json_content: importJsonContent
            },
            success: function(response) {
                $spinner.removeClass('is-active');

                if (response.success) {
                    renderPreview(response.data);
                    $preview.show();
                    $btn.prop('disabled', false);
                } else {
                    showAdminNotice('error', response.data || 'Failed to parse import file.');
                    clearImportFile();
                }
            },
            error: function() {
                $spinner.removeClass('is-active');
                showAdminNotice('error', 'An error occurred while parsing the file.');
                clearImportFile();
            }
        });
    }

    function renderPreview(data) {
        var html = '<div class="grt-preview-info">';

        // Export info
        html += '<p><strong>Source:</strong> ' + escapeHtml(data.export_info.source_site) + '</p>';
        html += '<p><strong>Export Date:</strong> ' + escapeHtml(data.export_info.export_date) + '</p>';
        html += '<p><strong>Plugin Version:</strong> ' + escapeHtml(data.export_info.plugin_version) + '</p>';
        html += '</div>';

        // Site blocks summary
        html += '<div class="grt-preview-summary">';
        html += '<h4>Site Blocks</h4>';
        html += '<p>' + data.site_blocks.total + ' total (' +
                data.site_blocks.new + ' new, ' +
                data.site_blocks.duplicates + ' duplicates)</p>';

        if (data.site_blocks.items.length > 0) {
            html += '<ul class="grt-preview-list">';
            data.site_blocks.items.forEach(function(item) {
                var status = item.is_duplicate ? '<span class="grt-duplicate">Duplicate</span>' : '<span class="grt-new">New</span>';
                html += '<li>' + escapeHtml(item.title) + ' ' + status + '</li>';
            });
            html += '</ul>';
        }
        html += '</div>';

        // Managed tables summary
        html += '<div class="grt-preview-summary">';
        html += '<h4>Managed Tables</h4>';
        html += '<p>' + data.managed_tables.total + ' total (' +
                data.managed_tables.new + ' new, ' +
                data.managed_tables.duplicates + ' duplicates)</p>';

        if (data.managed_tables.items.length > 0) {
            html += '<ul class="grt-preview-list">';
            data.managed_tables.items.forEach(function(item) {
                var status = item.is_duplicate ? '<span class="grt-duplicate">Duplicate</span>' : '<span class="grt-new">New</span>';
                html += '<li>' + escapeHtml(item.title) + ' (' + item.sites_count + ' sites) ' + status + '</li>';
            });
            html += '</ul>';
        }
        html += '</div>';

        $('.grt-preview-content').html(html);
    }

    function handleImport() {
        if (!importJsonContent) {
            showAdminNotice('error', 'No file selected for import.');
            return;
        }

        var $btn = $('#grt-import-btn');
        var $spinner = $btn.siblings('.spinner');
        var $results = $('#grt-import-results');

        $btn.prop('disabled', true);
        $spinner.addClass('is-active');

        $.ajax({
            url: typeof grtImportExport !== 'undefined' ? grtImportExport.ajaxUrl : ajaxurl,
            type: 'POST',
            data: {
                action: 'grt_import_execute',
                nonce: typeof grtImportExport !== 'undefined' ? grtImportExport.nonce : '',
                json_content: importJsonContent,
                duplicate_handling: $('input[name="duplicate_handling"]:checked').val()
            },
            success: function(response) {
                $spinner.removeClass('is-active');

                if (response.success) {
                    renderResults(response.data);
                    $results.show();
                    showAdminNotice('success', 'Import completed successfully!');
                } else {
                    $btn.prop('disabled', false);
                    showAdminNotice('error', response.data || 'Import failed.');
                }
            },
            error: function() {
                $spinner.removeClass('is-active');
                $btn.prop('disabled', false);
                showAdminNotice('error', 'An error occurred during import.');
            }
        });
    }

    function renderResults(data) {
        var html = '<div class="grt-results-summary">';

        html += '<h4>Site Blocks</h4>';
        html += '<ul>';
        html += '<li><strong>Created:</strong> ' + data.sites_created + '</li>';
        html += '<li><strong>Updated:</strong> ' + data.sites_updated + '</li>';
        html += '<li><strong>Skipped:</strong> ' + data.sites_skipped + '</li>';
        html += '</ul>';

        html += '<h4>Managed Tables</h4>';
        html += '<ul>';
        html += '<li><strong>Created:</strong> ' + data.tables_created + '</li>';
        html += '<li><strong>Updated:</strong> ' + data.tables_updated + '</li>';
        html += '<li><strong>Skipped:</strong> ' + data.tables_skipped + '</li>';
        html += '</ul>';

        html += '<h4>Images</h4>';
        html += '<ul>';
        html += '<li><strong>Imported:</strong> ' + data.images_imported + '</li>';
        html += '<li><strong>Failed:</strong> ' + data.images_failed + '</li>';
        html += '</ul>';

        if (data.errors && data.errors.length > 0) {
            html += '<h4 class="grt-errors-heading">Errors</h4>';
            html += '<ul class="grt-errors-list">';
            data.errors.forEach(function(error) {
                html += '<li>' + escapeHtml(error) + '</li>';
            });
            html += '</ul>';
        }

        if (data.warnings && data.warnings.length > 0) {
            html += '<h4 class="grt-warnings-heading">Warnings</h4>';
            html += '<ul class="grt-warnings-list">';
            data.warnings.forEach(function(warning) {
                html += '<li>' + escapeHtml(warning) + '</li>';
            });
            html += '</ul>';
        }

        html += '</div>';

        $('.grt-results-content').html(html);
    }

    function escapeHtml(text) {
        if (!text) return '';
        var div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

})(jQuery);