// Frontend JavaScript for Geo Ranking Tables
/* global grtAjax, jQuery */
(function($) {
    'use strict';

    // Conditional debug logging
    function debugLog() {
        if (grtAjax && grtAjax.debugMode) {
            console.log.apply(console, ['GRT:'].concat(Array.prototype.slice.call(arguments)));
        }
    }

    function debugError() {
        if (grtAjax && grtAjax.debugMode) {
            console.error.apply(console, ['GRT:'].concat(Array.prototype.slice.call(arguments)));
        }
    }

    // Main initialization
    $(document).ready(function() {
        initializeGRTTables();
    });
    
    function initializeGRTTables() {
        var wrappers = $('.grt-ranking-table-wrapper');
        debugLog('Initializing tables, found', wrappers.length, 'wrapper(s)');

        wrappers.each(function() {
            var $wrapper = $(this);
            var blockId = $wrapper.attr('id');
            if (!blockId) return;

            var $configScript = $('.grt-block-config[data-block-id="' + blockId + '"]');
            if ($configScript.length === 0) return;

            try {
                var config = JSON.parse($configScript.text());
                debugLog('Config loaded for', blockId, config);
                loadTableData($wrapper, config);
            } catch (e) {
                debugError('Error parsing block config', e);
                showError($wrapper, 'Configuration error');
            }
        });
    }
    
    function loadTableData($wrapper, config) {
        var $loading = $wrapper.find('.grt-table-loading');
        var $content = $wrapper.find('.grt-table-content');
        var $fallback = $wrapper.find('.grt-table-fallback');
        var $locationText = $wrapper.find('.grt-user-location');

        // Show loading state
        $loading.show();
        $content.hide();
        $fallback.hide();

        // Step 1: Detect user location
        debugLog('Detecting user location...');
        detectUserLocation()
            .then(function(location) {
                debugLog('Location detected:', location);
                var userState = '';

                if (location && location.state_code) {
                    // Update location display
                    var locationString = location.state;
                    if (location.city) {
                        locationString = location.city + ', ' + location.state;
                    }
                    $locationText.text(locationString + ' (' + location.state_code + ')');
                    userState = location.state_code;
                } else {
                    // Non-US or unknown location - will use fallback table
                    debugLog('Non-US location, using fallback');
                    $locationText.text('International');
                }

                // Step 2: Load filtered sites (will use fallback if no matches)
                debugLog('Loading filtered sites for state:', userState || '(empty - international)');
                return loadFilteredSites(config, userState);
            })
            .then(function(response) {
                $loading.hide();
                debugLog('Sites loaded, count:', response.count);

                if (response.count > 0) {
                    $content.html(response.html).show();

                    // Build table config for click tracking
                    var tableConfig = {
                        tableId: config.managedTableId || 0,
                        theme: config.theme || response.theme || 'default',
                        userState: response.user_state || ''
                    };

                    // Store user state globally for tracking
                    grtAjax.userState = response.user_state || '';

                    // Add click tracking with full config
                    trackSiteClicks($content, tableConfig);
                } else {
                    debugLog('No sites available, showing fallback message');
                    showFallback($wrapper, 'No gambling sites are available in your location.');
                }
            })
            .catch(function(error) {
                debugError('Error loading table data', error);
                $loading.hide();

                // Even on error, try to load with empty state to trigger fallback
                loadFilteredSites(config, '')
                    .then(function(response) {
                        if (response.count > 0) {
                            $locationText.text('International');
                            $content.html(response.html).show();

                            // Build table config for click tracking
                            var tableConfig = {
                                tableId: config.managedTableId || 0,
                                theme: config.theme || response.theme || 'default',
                                userState: ''
                            };

                            trackSiteClicks($content, tableConfig);
                        } else {
                            showFallback($wrapper, 'No gambling sites are available in your location.');
                        }
                    })
                    .catch(function() {
                        showFallback($wrapper, 'Unable to load ranking data. Please try refreshing the page.');
                    });
            });
    }
    
    function detectUserLocation() {
        return new Promise(function(resolve, reject) {
            var postData = {
                action: 'grt_detect_location',
                nonce: grtAjax.nonce
            };

            // Include forced IP if set via URL parameter
            if (grtAjax.forcedIp) {
                postData.forced_ip = grtAjax.forcedIp;
            }

            $.post(grtAjax.ajaxUrl, postData)
            .done(function(response) {
                if (response.success && response.data) {
                    resolve(response.data);
                } else {
                    reject(new Error('Location detection failed'));
                }
            })
            .fail(function(xhr, status, error) {
                reject(new Error('Location detection request failed: ' + error));
            });
        });
    }
    
    function loadFilteredSites(config, userState) {
        return new Promise(function(resolve, reject) {
            $.post(grtAjax.ajaxUrl, {
                action: 'grt_get_filtered_sites',
                nonce: grtAjax.nonce,
                user_state: userState,
                selected_sites: config.selectedSites || [],
                max_display: config.maxDisplay || 5,
                show_ratings: config.showRatings !== false,
                show_bonuses: config.showBonuses !== false,
                show_licenses: config.showLicenses === true,
                fallback_table_id: config.fallbackTableId || 0,
                theme: config.theme || 'default'
            })
            .done(function(response) {
                if (response.success) {
                    resolve(response.data);
                } else {
                    reject(new Error('Failed to load sites'));
                }
            })
            .fail(function(xhr, status, error) {
                reject(new Error('Sites request failed: ' + error));
            });
        });
    }
    
    function showFallback($wrapper, message) {
        var $fallback = $wrapper.find('.grt-table-fallback');
        var $fallbackMessage = $fallback.find('.grt-fallback-message p');
        
        if (message) {
            $fallbackMessage.text(message);
        }
        
        $fallback.show();
    }
    
    function showError($wrapper, message) {
        var $loading = $wrapper.find('.grt-table-loading');
        var $content = $wrapper.find('.grt-table-content');
        var $fallback = $wrapper.find('.grt-table-fallback');
        
        $loading.hide();
        $content.hide();
        
        var $fallbackMessage = $fallback.find('.grt-fallback-message');
        $fallbackMessage.find('h4').text('Error Loading Table');
        $fallbackMessage.find('p').text(message || 'An error occurred while loading the ranking table.');
        
        $fallback.show();
    }
    
    /**
     * Track clicks on site elements
     * @param {jQuery} $content - The table content container
     * @param {Object} tableConfig - Table configuration including tableId, theme, userState
     */
    function trackSiteClicks($content, tableConfig) {
        tableConfig = tableConfig || {};

        // CSS selectors for all clickable elements across themes
        var visitSelectors = '.grt-visit-site-btn, .grt-play-now-btn, .gs-visit-btn';
        var reviewSelectors = '.grt-read-review-link, .gs-review-link';
        var overlaySelectors = '.gs-site-overlay';

        // Track visit/play button clicks
        $content.find(visitSelectors).on('click', function() {
            trackClick($(this), 'visit', tableConfig);
        });

        // Track review link clicks
        $content.find(reviewSelectors).on('click', function() {
            trackClick($(this), 'review', tableConfig);
        });

        // Track overlay clicks (GS Theme - full card clickable)
        $content.find(overlaySelectors).on('click', function() {
            trackClick($(this), 'overlay', tableConfig);
        });
    }

    /**
     * Track a single click event
     * @param {jQuery} $element - The clicked element
     * @param {string} clickType - Type of click: 'visit', 'review', 'overlay'
     * @param {Object} tableConfig - Table configuration
     */
    function trackClick($element, clickType, tableConfig) {
        // Find the parent row/card to get site data
        var $row = $element.closest('.grt-site-row, .grt-site-card, .gs-site');

        // Extract data attributes
        var siteId = $row.data('site-id') || 0;
        var rank = $row.data('rank') || 0;

        // Don't track if no site ID
        if (!siteId) {
            debugLog('Click not tracked - no site ID found');
            return;
        }

        // Build tracking payload
        var trackingData = {
            action: 'grt_track_click',
            nonce: grtAjax.nonce,
            site_id: siteId,
            table_id: tableConfig.tableId || 0,
            page_id: grtAjax.pageId || 0,
            click_type: clickType,
            rank: rank,
            user_state: tableConfig.userState || grtAjax.userState || '',
            theme: tableConfig.theme || 'default',
            url: $element.attr('href') || window.location.href
        };

        debugLog('Click tracked:', trackingData);

        // Send tracking data without blocking navigation
        sendTrackingData(trackingData);
    }

    /**
     * Send tracking data using Beacon API (non-blocking)
     * Falls back to XMLHttpRequest for older browsers
     * @param {Object} data - Tracking data to send
     */
    function sendTrackingData(data) {
        // Convert data to FormData for Beacon API
        var formData = new FormData();
        for (var key in data) {
            if (data.hasOwnProperty(key)) {
                formData.append(key, data[key]);
            }
        }

        // Prefer Beacon API - works even when user navigates away
        if (navigator.sendBeacon) {
            navigator.sendBeacon(grtAjax.ajaxUrl, formData);
            debugLog('Tracking sent via Beacon API');
        } else {
            // Fallback to async XMLHttpRequest
            var xhr = new XMLHttpRequest();
            xhr.open('POST', grtAjax.ajaxUrl, true);

            // Convert data to URL-encoded string
            var params = [];
            for (var k in data) {
                if (data.hasOwnProperty(k)) {
                    params.push(encodeURIComponent(k) + '=' + encodeURIComponent(data[k]));
                }
            }

            xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
            xhr.send(params.join('&'));
            debugLog('Tracking sent via XHR fallback');
        }
    }
    
    // Refresh tables function
    window.grtRefreshTables = function() {
        $('.grt-ranking-table-wrapper').each(function() {
            var $wrapper = $(this);
            var $configScript = $('.grt-block-config[data-block-id="' + $wrapper.attr('id') + '"]');
            
            if ($configScript.length > 0) {
                try {
                    var config = JSON.parse($configScript.text());
                    loadTableData($wrapper, config);
                } catch (e) {
                    debugError('Error refreshing table', e);
                }
            }
        });
    };
    
})(jQuery);