<?php
/**
 * Click Reports Class
 * Handles reporting queries for click tracking data
 */

if (!defined('ABSPATH')) {
    exit;
}

class GRT_ClickReports {

    private $table_name;

    public function __construct() {
        global $wpdb;
        $this->table_name = $wpdb->prefix . 'grt_click_tracking';
    }

    /**
     * Get clicks aggregated by site
     *
     * @param string $start_date Start date (Y-m-d)
     * @param string $end_date End date (Y-m-d)
     * @param int $limit Max results
     * @return array Site click data
     */
    public function get_clicks_by_site($start_date, $end_date, $limit = 20) {
        global $wpdb;

        $sql = $wpdb->prepare(
            "SELECT
                c.site_id,
                p.post_title as site_name,
                COUNT(*) as total_clicks,
                SUM(CASE WHEN c.click_type = 'visit' THEN 1 ELSE 0 END) as visit_clicks,
                SUM(CASE WHEN c.click_type = 'review' THEN 1 ELSE 0 END) as review_clicks,
                SUM(CASE WHEN c.click_type = 'overlay' THEN 1 ELSE 0 END) as overlay_clicks,
                COUNT(DISTINCT c.session_id) as unique_sessions
            FROM {$this->table_name} c
            LEFT JOIN {$wpdb->posts} p ON c.site_id = p.ID
            WHERE c.created_at BETWEEN %s AND %s
            GROUP BY c.site_id
            ORDER BY total_clicks DESC
            LIMIT %d",
            $start_date . ' 00:00:00',
            $end_date . ' 23:59:59',
            $limit
        );

        return $wpdb->get_results($sql, ARRAY_A);
    }

    /**
     * Get clicks aggregated by US state
     *
     * @param string $start_date Start date (Y-m-d)
     * @param string $end_date End date (Y-m-d)
     * @return array State click data
     */
    public function get_clicks_by_state($start_date, $end_date) {
        global $wpdb;

        $sql = $wpdb->prepare(
            "SELECT
                user_state,
                COUNT(*) as total_clicks,
                COUNT(DISTINCT session_id) as unique_sessions,
                SUM(CASE WHEN click_type = 'visit' THEN 1 ELSE 0 END) as visit_clicks,
                SUM(CASE WHEN click_type = 'review' THEN 1 ELSE 0 END) as review_clicks
            FROM {$this->table_name}
            WHERE created_at BETWEEN %s AND %s
            AND user_state IS NOT NULL AND user_state != ''
            GROUP BY user_state
            ORDER BY total_clicks DESC",
            $start_date . ' 00:00:00',
            $end_date . ' 23:59:59'
        );

        return $wpdb->get_results($sql, ARRAY_A);
    }

    /**
     * Get clicks aggregated by device type
     *
     * @param string $start_date Start date (Y-m-d)
     * @param string $end_date End date (Y-m-d)
     * @return array Device click data
     */
    public function get_clicks_by_device($start_date, $end_date) {
        global $wpdb;

        $total_sql = $wpdb->prepare(
            "SELECT COUNT(*) FROM {$this->table_name} WHERE created_at BETWEEN %s AND %s",
            $start_date . ' 00:00:00',
            $end_date . ' 23:59:59'
        );
        $total = $wpdb->get_var($total_sql);

        if ($total == 0) {
            return array();
        }

        $sql = $wpdb->prepare(
            "SELECT
                device_type,
                COUNT(*) as total_clicks,
                ROUND(COUNT(*) * 100.0 / %d, 1) as percentage
            FROM {$this->table_name}
            WHERE created_at BETWEEN %s AND %s
            GROUP BY device_type
            ORDER BY total_clicks DESC",
            $total,
            $start_date . ' 00:00:00',
            $end_date . ' 23:59:59'
        );

        return $wpdb->get_results($sql, ARRAY_A);
    }

    /**
     * Get summary statistics
     *
     * @param string $start_date Start date (Y-m-d)
     * @param string $end_date End date (Y-m-d)
     * @return array Summary stats
     */
    public function get_summary_stats($start_date, $end_date) {
        global $wpdb;

        $sql = $wpdb->prepare(
            "SELECT
                COUNT(*) as total_clicks,
                COUNT(DISTINCT session_id) as unique_sessions,
                SUM(CASE WHEN click_type = 'visit' THEN 1 ELSE 0 END) as visit_clicks,
                SUM(CASE WHEN click_type = 'review' THEN 1 ELSE 0 END) as review_clicks,
                SUM(CASE WHEN click_type = 'overlay' THEN 1 ELSE 0 END) as overlay_clicks,
                COUNT(DISTINCT site_id) as unique_sites,
                COUNT(DISTINCT user_state) as unique_states
            FROM {$this->table_name}
            WHERE created_at BETWEEN %s AND %s",
            $start_date . ' 00:00:00',
            $end_date . ' 23:59:59'
        );

        $result = $wpdb->get_row($sql, ARRAY_A);

        // Handle null values
        if ($result) {
            foreach ($result as $key => $value) {
                $result[$key] = $value ?: 0;
            }
        }

        return $result ?: array(
            'total_clicks' => 0,
            'unique_sessions' => 0,
            'visit_clicks' => 0,
            'review_clicks' => 0,
            'overlay_clicks' => 0,
            'unique_sites' => 0,
            'unique_states' => 0
        );
    }

    /**
     * Get total record count
     *
     * @return int Total records
     */
    public function get_record_count() {
        global $wpdb;
        return (int) $wpdb->get_var("SELECT COUNT(*) FROM {$this->table_name}");
    }

    /**
     * Get record count older than specified days
     *
     * @param int $days Number of days
     * @return int Record count
     */
    public function get_old_record_count($days) {
        global $wpdb;

        $cutoff_date = date('Y-m-d H:i:s', strtotime("-{$days} days"));

        return (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$this->table_name} WHERE created_at < %s",
            $cutoff_date
        ));
    }

    /**
     * Purge records older than specified days
     *
     * @param int $days Number of days
     * @return int Number of deleted records
     */
    public function purge_old_data($days) {
        global $wpdb;

        $days = max(1, intval($days)); // Ensure at least 1 day
        $cutoff_date = date('Y-m-d H:i:s', strtotime("-{$days} days"));

        $deleted = $wpdb->query($wpdb->prepare(
            "DELETE FROM {$this->table_name} WHERE created_at < %s",
            $cutoff_date
        ));

        return $deleted !== false ? $deleted : 0;
    }

    /**
     * Export data to CSV format
     *
     * @param string $start_date Start date (Y-m-d)
     * @param string $end_date End date (Y-m-d)
     * @return array Data rows for CSV export
     */
    public function export_csv($start_date, $end_date) {
        global $wpdb;

        $sql = $wpdb->prepare(
            "SELECT
                c.created_at,
                p.post_title as site_name,
                c.click_type,
                c.rank_position,
                c.user_state,
                c.device_type,
                c.theme,
                c.referrer_url
            FROM {$this->table_name} c
            LEFT JOIN {$wpdb->posts} p ON c.site_id = p.ID
            WHERE c.created_at BETWEEN %s AND %s
            ORDER BY c.created_at DESC",
            $start_date . ' 00:00:00',
            $end_date . ' 23:59:59'
        );

        return $wpdb->get_results($sql, ARRAY_A);
    }

    /**
     * Get clicks by theme
     *
     * @param string $start_date Start date (Y-m-d)
     * @param string $end_date End date (Y-m-d)
     * @return array Theme click data
     */
    public function get_clicks_by_theme($start_date, $end_date) {
        global $wpdb;

        $sql = $wpdb->prepare(
            "SELECT
                theme,
                COUNT(*) as total_clicks,
                SUM(CASE WHEN click_type = 'visit' THEN 1 ELSE 0 END) as visit_clicks,
                SUM(CASE WHEN click_type = 'review' THEN 1 ELSE 0 END) as review_clicks
            FROM {$this->table_name}
            WHERE created_at BETWEEN %s AND %s
            GROUP BY theme
            ORDER BY total_clicks DESC",
            $start_date . ' 00:00:00',
            $end_date . ' 23:59:59'
        );

        return $wpdb->get_results($sql, ARRAY_A);
    }

    /**
     * Get clicks by rank position
     *
     * @param string $start_date Start date (Y-m-d)
     * @param string $end_date End date (Y-m-d)
     * @return array Rank position click data
     */
    public function get_clicks_by_rank($start_date, $end_date) {
        global $wpdb;

        $sql = $wpdb->prepare(
            "SELECT
                rank_position,
                COUNT(*) as total_clicks,
                SUM(CASE WHEN click_type = 'visit' THEN 1 ELSE 0 END) as visit_clicks,
                SUM(CASE WHEN click_type = 'review' THEN 1 ELSE 0 END) as review_clicks
            FROM {$this->table_name}
            WHERE created_at BETWEEN %s AND %s
            AND rank_position IS NOT NULL AND rank_position > 0
            GROUP BY rank_position
            ORDER BY rank_position ASC",
            $start_date . ' 00:00:00',
            $end_date . ' 23:59:59'
        );

        return $wpdb->get_results($sql, ARRAY_A);
    }
}
