<?php
/**
 * Table Blocks Management Class
 * Handles Gutenberg blocks for displaying geo-filtered ranking tables
 */

if (!defined('ABSPATH')) {
    exit;
}

class GRT_TableBlocks {
    
    public function __construct() {
        // Remove block registration from here - it's now in main plugin
        // add_action('init', array($this, 'register_block'));
        add_action('enqueue_block_editor_assets', array($this, 'enqueue_block_editor_assets'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_frontend_scripts'));
        add_action('wp_ajax_grt_get_filtered_sites', array($this, 'ajax_get_filtered_sites'));
        add_action('wp_ajax_nopriv_grt_get_filtered_sites', array($this, 'ajax_get_filtered_sites'));
    }
    
    /**
     * Register the Gutenberg block
     */
    public function register_block() {
        // Check if Gutenberg is available
        if (!function_exists('register_block_type')) {
            return;
        }
        
        // Register the block
        $result = register_block_type('geo-ranking-tables/ranking-table', array(
            'render_callback' => array($this, 'render_table_block'),
            'attributes' => array(
                'selectedSites' => array(
                    'type' => 'array',
                    'default' => array(),
                ),
                'maxDisplay' => array(
                    'type' => 'number',
                    'default' => 5,
                ),
                'fallbackTableId' => array(
                    'type' => 'number',
                    'default' => 0,
                ),
                'tableName' => array(
                    'type' => 'string',
                    'default' => 'Top Gambling Sites',
                ),
                'showRatings' => array(
                    'type' => 'boolean',
                    'default' => true,
                ),
                'showBonuses' => array(
                    'type' => 'boolean',
                    'default' => true,
                ),
                'showLicenses' => array(
                    'type' => 'boolean',
                    'default' => false,
                ),
            ),
        ));
    }
    
    /**
     * Enqueue block editor assets
     */
    public function enqueue_block_editor_assets() {
        // Enqueue the block script
        wp_enqueue_script(
            'grt-block-editor',
            GRT_PLUGIN_URL . 'assets/js/block-editor.js',
            array('wp-blocks', 'wp-i18n', 'wp-element', 'wp-components', 'wp-block-editor', 'wp-editor'),
            GRT_VERSION,
            true
        );
        
        // Enqueue the block editor styles
        wp_enqueue_style(
            'grt-block-editor',
            GRT_PLUGIN_URL . 'assets/css/block-editor.css',
            array('wp-edit-blocks'),
            GRT_VERSION
        );
        
        // Localize script with site blocks data
        wp_localize_script('grt-block-editor', 'grtBlockData', array(
            'siteBlocks' => $this->get_all_site_blocks(),
            'managedTables' => $this->get_all_managed_tables(),
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('grt_nonce'),
        ));
    }
    
    /**
     * Enqueue frontend scripts
     */
    public function enqueue_frontend_scripts() {
        wp_enqueue_script(
            'grt-frontend',
            GRT_PLUGIN_URL . 'assets/js/frontend.js',
            array('jquery'),
            GRT_VERSION,
            true
        );
        
        wp_enqueue_style(
            'grt-frontend',
            GRT_PLUGIN_URL . 'assets/css/frontend.css',
            array(),
            GRT_VERSION
        );
        
        wp_localize_script('grt-frontend', 'grtAjax', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('grt_nonce'),
        ));
    }
    
    /**
     * Render the table block on frontend
     */
    public function render_table_block($attributes) {
        $selected_sites = isset($attributes['selectedSites']) ? $attributes['selectedSites'] : array();
        $max_display = isset($attributes['maxDisplay']) ? intval($attributes['maxDisplay']) : 5;
        $fallback_block_id = isset($attributes['fallbackTableId']) ? intval($attributes['fallbackTableId']) : 0;
        $table_name = isset($attributes['tableName']) ? $attributes['tableName'] : 'Top Gambling Sites';
        $show_ratings = isset($attributes['showRatings']) ? $attributes['showRatings'] : true;
        $show_bonuses = isset($attributes['showBonuses']) ? $attributes['showBonuses'] : true;
        $show_licenses = isset($attributes['showLicenses']) ? $attributes['showLicenses'] : false;

        if (empty($selected_sites)) {
            return '<div class="grt-no-sites" style="padding: 20px; border: 2px solid red; text-align: center; background: #ffe6e6; margin: 20px 0;">⚠️ No sites selected for this ranking table. Please edit the block and select some sites.</div>';
        }
        
        // Generate unique ID for this table block
        $block_id = 'grt-table-' . md5(serialize($attributes));
        
        ob_start();
        ?>
        <div class="grt-ranking-table-wrapper" id="<?php echo esc_attr($block_id); ?>" style="border: 2px solid green; margin: 20px 0;">
            <div class="grt-table-header">
                <h3 class="grt-table-title"><?php echo esc_html($table_name); ?></h3>
                <div class="grt-location-info">
                    <span class="grt-location-text">Showing sites available in: <strong class="grt-user-location">Detecting location...</strong></span>
                </div>
            </div>
            
            <div class="grt-table-loading">
                <div class="grt-loading-spinner"></div>
                <p>Loading available gambling sites for your location...</p>
            </div>
            
            <div class="grt-table-content" style="display: none;">
                <!-- Content will be populated by JavaScript -->
            </div>
            
            <div class="grt-table-fallback" style="display: none;">
                <div class="grt-fallback-message">
                    <h4>No sites available in your location</h4>
                    <p>Unfortunately, none of our featured gambling sites accept players from your current location.</p>
                </div>
            </div>
        </div>
        
        <script type="application/json" class="grt-block-config" data-block-id="<?php echo esc_attr($block_id); ?>">
        <?php echo wp_json_encode(array(
            'selectedSites' => $selected_sites,
            'maxDisplay' => $max_display,
            'fallbackTableId' => $fallback_block_id,
            'showRatings' => $show_ratings,
            'showBonuses' => $show_bonuses,
            'showLicenses' => $show_licenses,
        )); ?>
        </script>
        <?php
        return ob_get_clean();
    }
    
    /**
     * Get all site blocks for the block editor
     */
    private function get_all_site_blocks() {
        $site_blocks = get_posts(array(
            'post_type' => 'grt_site_block',
            'post_status' => 'publish',
            'numberposts' => -1,
            'orderby' => 'title',
            'order' => 'ASC',
        ));
        
        $blocks_data = array();
        foreach ($site_blocks as $block) {
            $blocks_data[] = array(
                'id' => $block->ID,
                'title' => $block->post_title,
                'site_name' => get_post_meta($block->ID, '_grt_site_name', true),
                'description' => get_post_meta($block->ID, '_grt_site_description', true),
                'rating' => get_post_meta($block->ID, '_grt_star_rating', true),
                'states' => get_post_meta($block->ID, '_grt_states_accepted', true),
            );
        }
        
        return $blocks_data;
    }
    
    /**
     * Get all managed tables for the block editor
     */
    private function get_all_managed_tables() {
        $managed_tables = get_posts(array(
            'post_type' => 'grt_managed_table',
            'post_status' => 'publish',
            'numberposts' => -1,
            'orderby' => 'title',
            'order' => 'ASC',
        ));
        
        $tables_data = array();
        foreach ($managed_tables as $table) {
            $sites = get_post_meta($table->ID, '_grt_managed_sites', true);
            $tables_data[] = array(
                'id' => $table->ID,
                'title' => $table->post_title,
                'slug' => get_post_meta($table->ID, '_grt_managed_slug', true),
                'sites_count' => is_array($sites) ? count($sites) : 0,
            );
        }
        
        return $tables_data;
    }
    
    /**
     * AJAX handler for getting filtered sites based on user location
     */
    public function ajax_get_filtered_sites() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'grt_nonce')) {
            wp_die('Security check failed');
        }
        
        $user_state = sanitize_text_field($_POST['user_state']);
        $selected_sites = isset($_POST['selected_sites']) ? array_map('intval', $_POST['selected_sites']) : array();
        $max_display = isset($_POST['max_display']) ? intval($_POST['max_display']) : 5;
        $show_ratings = isset($_POST['show_ratings']) ? (bool)$_POST['show_ratings'] : true;
        $show_bonuses = isset($_POST['show_bonuses']) ? (bool)$_POST['show_bonuses'] : true;
        $show_licenses = isset($_POST['show_licenses']) ? (bool)$_POST['show_licenses'] : false;
        $fallback_table_id = isset($_POST['fallback_table_id']) ? intval($_POST['fallback_table_id']) : 0;
        $theme = isset($_POST['theme']) ? sanitize_text_field($_POST['theme']) : 'default';

        if (empty($selected_sites)) {
            wp_send_json_error('No sites selected');
        }
        
        // Get site blocks data
        $sites_data = array();
        foreach ($selected_sites as $site_id) {
            $post = get_post($site_id);
            if (!$post || $post->post_type !== 'grt_site_block' || $post->post_status !== 'publish') {
                continue;
            }
            
            $states_accepted = get_post_meta($site_id, '_grt_states_accepted', true);
            if (!is_array($states_accepted)) {
                $states_accepted = array();
            }
            
            // Check if site accepts users from this state
            // If user_state is empty (non-US), skip all sites to trigger fallback
            // If user_state is set but not in accepted list, skip the site
            if (empty($user_state) || !in_array($user_state, $states_accepted)) {
                continue;
            }
            
            $certified_badge_id = get_post_meta($site_id, '_grt_certified_badge', true);
            $certified_badge_url = $certified_badge_id ? wp_get_attachment_url($certified_badge_id) : '';

            $site_data = array(
                'id' => $site_id,
                'title' => $post->post_title,
                'site_name' => get_post_meta($site_id, '_grt_site_name', true),
                'description' => get_post_meta($site_id, '_grt_site_description', true),
                'rating' => floatval(get_post_meta($site_id, '_grt_star_rating', true)),
                'affiliate_link' => get_post_meta($site_id, '_grt_affiliate_link', true),
                'bonus_info' => get_post_meta($site_id, '_grt_bonus_info', true),
                'license_info' => get_post_meta($site_id, '_grt_license_info', true),
                'featured_image' => get_the_post_thumbnail_url($site_id, 'medium'),
                // New fields
                'pros' => get_post_meta($site_id, '_grt_pros', true) ?: array(),
                'cons' => get_post_meta($site_id, '_grt_cons', true) ?: array(),
                'payout_speed' => get_post_meta($site_id, '_grt_payout_speed', true),
                'compatibility' => get_post_meta($site_id, '_grt_compatibility', true) ?: array(),
                'payout_percentage' => get_post_meta($site_id, '_grt_payout_percentage', true),
                'review_link' => get_post_meta($site_id, '_grt_review_link', true),
                'certified_badge' => $certified_badge_url,
            );

            $sites_data[] = $site_data;
        }
        
        // Sort by rating (highest first)
        usort($sites_data, function($a, $b) {
            return $b['rating'] <=> $a['rating'];
        });
        
        // Limit results
        if (count($sites_data) > $max_display) {
            $sites_data = array_slice($sites_data, 0, $max_display);
        }

        // If no sites match and fallback table exists, use fallback sites WITHOUT geo-filtering
        if (empty($sites_data) && $fallback_table_id > 0) {
            $fallback_config = GRT_ManagedTables::get_config($fallback_table_id);

            if ($fallback_config && !empty($fallback_config['selectedSites'])) {
                foreach ($fallback_config['selectedSites'] as $site_id) {
                    $post = get_post($site_id);
                    if (!$post || $post->post_type !== 'grt_site_block' || $post->post_status !== 'publish') {
                        continue;
                    }

                    // NO geo-filtering - include all sites from fallback table
                    $fb_certified_badge_id = get_post_meta($site_id, '_grt_certified_badge', true);
                    $fb_certified_badge_url = $fb_certified_badge_id ? wp_get_attachment_url($fb_certified_badge_id) : '';

                    $sites_data[] = array(
                        'id' => $site_id,
                        'title' => $post->post_title,
                        'site_name' => get_post_meta($site_id, '_grt_site_name', true),
                        'description' => get_post_meta($site_id, '_grt_site_description', true),
                        'rating' => floatval(get_post_meta($site_id, '_grt_star_rating', true)),
                        'affiliate_link' => get_post_meta($site_id, '_grt_affiliate_link', true),
                        'bonus_info' => get_post_meta($site_id, '_grt_bonus_info', true),
                        'license_info' => get_post_meta($site_id, '_grt_license_info', true),
                        'featured_image' => get_the_post_thumbnail_url($site_id, 'medium'),
                        // New fields
                        'pros' => get_post_meta($site_id, '_grt_pros', true) ?: array(),
                        'cons' => get_post_meta($site_id, '_grt_cons', true) ?: array(),
                        'payout_speed' => get_post_meta($site_id, '_grt_payout_speed', true),
                        'compatibility' => get_post_meta($site_id, '_grt_compatibility', true) ?: array(),
                        'payout_percentage' => get_post_meta($site_id, '_grt_payout_percentage', true),
                        'review_link' => get_post_meta($site_id, '_grt_review_link', true),
                        'certified_badge' => $fb_certified_badge_url,
                    );
                }

                // Sort by rating and limit to original max_display
                usort($sites_data, function($a, $b) {
                    return $b['rating'] <=> $a['rating'];
                });

                if (count($sites_data) > $max_display) {
                    $sites_data = array_slice($sites_data, 0, $max_display);
                }
            }
        }

        // Build config for template
        $config = array(
            'showRatings' => $show_ratings,
            'showBonuses' => $show_bonuses,
            'showLicenses' => $show_licenses,
            'theme' => $theme,
        );

        // Generate HTML using theme template if available
        $html = '';
        $theme_config = GRT_ThemeRegistry::get_theme($theme);

        if ($theme_config && $theme_config['has_custom_layout'] && !empty($theme_config['template_class'])) {
            $template = GRT_ThemeRegistry::get_template_instance($theme);
            if ($template) {
                $html = $template->render_sites_table($sites_data, $config);
            }
        }

        // Fall back to default generator if no template or template failed
        if (empty($html)) {
            $html = $this->generate_table_html($sites_data, $show_ratings, $show_bonuses, $show_licenses);
        }

        wp_send_json_success(array(
            'html' => $html,
            'count' => count($sites_data),
            'user_state' => $user_state,
            'theme' => $theme,
        ));
    }
    
    /**
     * Generate HTML for the ranking table
     */
    private function generate_table_html($sites_data, $show_ratings = true, $show_bonuses = true, $show_licenses = false) {
        if (empty($sites_data)) {
            return '<div class="grt-no-results">No gambling sites available for your location.</div>';
        }
        
        ob_start();
        ?>
        <div class="grt-sites-table">
            <?php foreach ($sites_data as $index => $site): ?>
                <div class="grt-site-row" data-rank="<?php echo ($index + 1); ?>" data-site-id="<?php echo esc_attr($site['id']); ?>">
                    <div class="grt-site-rank">
                        <span class="grt-rank-number"><?php echo ($index + 1); ?></span>
                    </div>
                    
                    <div class="grt-site-logo">
                        <?php if (!empty($site['featured_image'])): ?>
                            <img src="<?php echo esc_url($site['featured_image']); ?>" alt="<?php echo esc_attr($site['site_name']); ?>" />
                        <?php else: ?>
                            <div class="grt-site-initial"><?php echo esc_html(substr($site['site_name'], 0, 1)); ?></div>
                        <?php endif; ?>
                    </div>
                    
                    <div class="grt-site-info">
                        <h4 class="grt-site-name"><?php echo esc_html($site['site_name'] ?: $site['title']); ?></h4>
                        
                        <?php if ($show_ratings && $site['rating'] > 0): ?>
                            <div class="grt-site-rating">
                                <?php $this->render_star_rating($site['rating']); ?>
                                <span class="grt-rating-text"><?php echo esc_html($site['rating']); ?>/5</span>
                            </div>
                        <?php endif; ?>
                        
                        <?php if (!empty($site['description'])): ?>
                            <p class="grt-site-description"><?php echo esc_html($site['description']); ?></p>
                        <?php endif; ?>
                        
                        <?php if ($show_bonuses && !empty($site['bonus_info'])): ?>
                            <div class="grt-site-bonus">
                                <strong>Bonus:</strong> <?php echo esc_html($site['bonus_info']); ?>
                            </div>
                        <?php endif; ?>
                        
                        <?php if ($show_licenses && !empty($site['license_info'])): ?>
                            <div class="grt-site-license">
                                <strong>License:</strong> <?php echo esc_html($site['license_info']); ?>
                            </div>
                        <?php endif; ?>
                    </div>
                    
                    <div class="grt-site-action">
                        <?php if (!empty($site['affiliate_link'])): ?>
                            <a href="<?php echo esc_url($site['affiliate_link']); ?>" target="_blank" class="grt-visit-site-btn" rel="nofollow noopener">
                                Visit Site
                            </a>
                        <?php else: ?>
                            <span class="grt-no-link">Visit Site</span>
                        <?php endif; ?>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>
        <?php
        return ob_get_clean();
    }
    
    /**
     * Render star rating
     */
    private function render_star_rating($rating) {
        $full_stars = floor($rating);
        $half_star = ($rating - $full_stars) >= 0.5;
        $empty_stars = 5 - $full_stars - ($half_star ? 1 : 0);
        
        echo '<div class="grt-stars">';
        
        // Full stars
        for ($i = 0; $i < $full_stars; $i++) {
            echo '<span class="grt-star grt-star-full">★</span>';
        }
        
        // Half star
        if ($half_star) {
            echo '<span class="grt-star grt-star-half">★</span>';
        }
        
        // Empty stars
        for ($i = 0; $i < $empty_stars; $i++) {
            echo '<span class="grt-star grt-star-empty">☆</span>';
        }
        
        echo '</div>';
    }
}