<?php
/**
 * Installation and Setup Script for Geo-Ranking Tables Plugin
 * 
 * This file can be used to set up sample data for testing the plugin
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Create sample site blocks for testing
 */
function grt_create_sample_data() {
    // Sample gambling sites data - Top 5 most recognizable brands
    $sample_sites = array(
        array(
            'title' => 'BetMGM Casino',
            'site_name' => 'BetMGM',
            'description' => 'Premier online casino and sportsbook with a wide variety of games and competitive odds.',
            'rating' => 4.5,
            'states' => array('NJ', 'PA', 'WV', 'MI', 'VA', 'TN', 'CO', 'IN', 'IA', 'IL'),
            'affiliate_link' => 'https://example.com/betmgm',
            'bonus_info' => 'Up to $1,000 deposit bonus + $25 free play',
            'license_info' => 'Licensed by New Jersey Division of Gaming Enforcement'
        ),
        array(
            'title' => 'DraftKings Casino',
            'site_name' => 'DraftKings',
            'description' => 'Leading daily fantasy sports platform now offering full casino and sportsbook services.',
            'rating' => 4.3,
            'states' => array('NJ', 'PA', 'WV', 'MI', 'CT', 'VA', 'TN', 'CO', 'IN', 'IA', 'IL', 'AZ', 'NH'),
            'affiliate_link' => 'https://example.com/draftkings',
            'bonus_info' => '$2,000 deposit bonus + $50 free bet',
            'license_info' => 'Licensed by Pennsylvania Gaming Control Board'
        ),
        array(
            'title' => 'FanDuel Casino',
            'site_name' => 'FanDuel',
            'description' => 'Top-rated fantasy sports and casino platform with excellent mobile app and user experience.',
            'rating' => 4.4,
            'states' => array('NJ', 'PA', 'WV', 'MI', 'CT', 'VA', 'TN', 'CO', 'IN', 'IA', 'IL', 'AZ'),
            'affiliate_link' => 'https://example.com/fanduel',
            'bonus_info' => 'Play $5, Get $150 in bonus bets',
            'license_info' => 'Licensed by West Virginia Lottery Commission'
        ),
        array(
            'title' => 'Caesars Casino',
            'site_name' => 'Caesars',
            'description' => 'Iconic Las Vegas brand bringing premium casino experience online with rewards program.',
            'rating' => 4.2,
            'states' => array('NJ', 'PA', 'WV', 'MI', 'VA', 'TN', 'CO', 'IN', 'IA', 'IL', 'AZ', 'LA'),
            'affiliate_link' => 'https://example.com/caesars',
            'bonus_info' => '$1,250 deposit match + 2,500 reward credits',
            'license_info' => 'Licensed by Michigan Gaming Control Board'
        ),
        array(
            'title' => 'PointsBet Casino',
            'site_name' => 'PointsBet',
            'description' => 'Australian-based sportsbook with unique PointsBetting and traditional fixed-odds betting.',
            'rating' => 4.0,
            'states' => array('NJ', 'PA', 'WV', 'MI', 'VA', 'CO', 'IN', 'IA', 'IL'),
            'affiliate_link' => 'https://example.com/pointsbet',
            'bonus_info' => 'Up to $500 in free bets',
            'license_info' => 'Licensed by Indiana Gaming Commission'
        )
    );
    
    $created_ids = array();
    
    foreach ($sample_sites as $site_data) {
        // Create the post
        $post_data = array(
            'post_title' => $site_data['title'],
            'post_type' => 'grt_site_block',
            'post_status' => 'publish',
            'post_content' => $site_data['description']
        );
        
        $post_id = wp_insert_post($post_data);
        
        if ($post_id && !is_wp_error($post_id)) {
            // Add meta data
            update_post_meta($post_id, '_grt_site_name', $site_data['site_name']);
            update_post_meta($post_id, '_grt_site_description', $site_data['description']);
            update_post_meta($post_id, '_grt_star_rating', $site_data['rating']);
            update_post_meta($post_id, '_grt_states_accepted', $site_data['states']);
            update_post_meta($post_id, '_grt_affiliate_link', $site_data['affiliate_link']);
            update_post_meta($post_id, '_grt_bonus_info', $site_data['bonus_info']);
            update_post_meta($post_id, '_grt_license_info', $site_data['license_info']);
            
            // Mark as sample data
            update_post_meta($post_id, '_grt_is_sample_data', '1');
            
            $created_ids[] = $post_id;
        }
    }
    
    return $created_ids;
}

/**
 * Create sample page with ranking table
 */
function grt_create_sample_page($site_ids) {
    if (empty($site_ids)) {
        return false;
    }
    
    // Create block content
    $block_content = '<!-- wp:geo-ranking-tables/ranking-table {"selectedSites":[' . implode(',', $site_ids) . '],"maxDisplay":5,"tableName":"Top Online Casinos","showRatings":true,"showBonuses":true,"showLicenses":false} /-->';
    
    // Create the page
    $page_data = array(
        'post_title' => 'Best Online Casinos',
        'post_content' => '<!-- wp:paragraph -->' . "\n" . 
                         '<p>Find the best online casinos available in your state. Our geo-targeted rankings show you only the gambling sites that accept players from your location.</p>' . "\n" .
                         '<!-- /wp:paragraph -->' . "\n\n" .
                         $block_content . "\n\n" .
                         '<!-- wp:paragraph -->' . "\n" . 
                         '<p><strong>Important:</strong> Please gamble responsibly. These rankings are based on our editorial review and user feedback. Always verify that online gambling is legal in your jurisdiction before playing.</p>' . "\n" .
                         '<!-- /wp:paragraph -->',
        'post_type' => 'page',
        'post_status' => 'publish',
        'post_excerpt' => 'Discover the top-rated online casinos available in your state with our geo-filtered ranking table.'
    );
    
    $page_id = wp_insert_post($page_data);
    
    // Mark page as sample data
    if ($page_id && !is_wp_error($page_id)) {
        update_post_meta($page_id, '_grt_is_sample_data', '1');
    }
    
    return $page_id;
}

/**
 * Setup function to create all sample data
 */
function grt_setup_sample_data() {
    // Create sample site blocks
    $site_ids = grt_create_sample_data();
    
    if (!empty($site_ids)) {
        // Create sample page
        $page_id = grt_create_sample_page($site_ids);
        
        return array(
            'sites_created' => count($site_ids),
            'site_ids' => $site_ids,
            'page_id' => $page_id,
            'page_url' => $page_id ? get_permalink($page_id) : null
        );
    }
    
    return false;
}

// Add admin notice with setup option
add_action('admin_notices', function() {
    if (get_option('grt_sample_data_created')) {
        return;
    }
    
    $current_screen = get_current_screen();
    if ($current_screen && strpos($current_screen->id, 'geo-ranking-tables') !== false) {
        ?>
        <div class="notice notice-info is-dismissible">
            <p><strong>Geo-Ranking Tables:</strong> Would you like to create sample gambling site data for testing? 
               <a href="<?php echo admin_url('admin.php?page=geo-ranking-tables&action=create-sample-data&nonce=' . wp_create_nonce('grt_sample_data')); ?>" class="button button-primary" style="margin-left: 10px;">Create Sample Data</a>
            </p>
        </div>
        <?php
    }
});

/**
 * Check if sample data exists
 */
function grt_check_sample_data_exists() {
    $sample_posts = get_posts(array(
        'post_type' => 'grt_site_block',
        'meta_key' => '_grt_is_sample_data',
        'meta_value' => '1',
        'posts_per_page' => 1,
        'post_status' => 'publish'
    ));
    
    return !empty($sample_posts);
}

/**
 * Get IDs of sample data posts
 */
function grt_get_sample_data_ids() {
    $sample_posts = get_posts(array(
        'post_type' => 'grt_site_block',
        'meta_key' => '_grt_is_sample_data',
        'meta_value' => '1',
        'posts_per_page' => -1,
        'post_status' => 'any',
        'fields' => 'ids'
    ));
    
    // Also get sample pages
    $sample_pages = get_posts(array(
        'post_type' => 'page',
        'meta_key' => '_grt_is_sample_data',
        'meta_value' => '1',
        'posts_per_page' => -1,
        'post_status' => 'any',
        'fields' => 'ids'
    ));
    
    return array_merge($sample_posts, $sample_pages);
}

/**
 * Delete all sample data
 */
function grt_delete_sample_data() {
    $sample_ids = grt_get_sample_data_ids();
    $deleted_count = 0;
    
    foreach ($sample_ids as $post_id) {
        if (wp_delete_post($post_id, true)) {
            $deleted_count++;
        }
    }
    
    // Reset the option
    delete_option('grt_sample_data_created');
    
    return $deleted_count;
}

// Handle sample data creation via URL (legacy support)
add_action('admin_init', function() {
    if (isset($_GET['action']) && $_GET['action'] === 'create-sample-data' && isset($_GET['nonce'])) {
        if (!wp_verify_nonce($_GET['nonce'], 'grt_sample_data')) {
            wp_die('Security check failed');
        }
        
        if (!current_user_can('manage_options')) {
            wp_die('Insufficient permissions');
        }
        
        $result = grt_setup_sample_data();
        
        if ($result) {
            update_option('grt_sample_data_created', true);
            
            $message = sprintf(
                'Sample data created successfully! %d gambling sites and 1 demo page created. <a href="%s" target="_blank">View Demo Page</a>',
                $result['sites_created'],
                $result['page_url']
            );
            
            add_action('admin_notices', function() use ($message) {
                echo '<div class="notice notice-success is-dismissible"><p>' . $message . '</p></div>';
            });
        } else {
            add_action('admin_notices', function() {
                echo '<div class="notice notice-error is-dismissible"><p>Failed to create sample data. Please try again.</p></div>';
            });
        }
        
        wp_redirect(admin_url('admin.php?page=geo-ranking-tables'));
        exit;
    }
});