// Frontend JavaScript for Geo Ranking Tables
/* global grtAjax, jQuery */
(function($) {
    'use strict';
    
    // Main initialization
    $(document).ready(function() {
        initializeGRTTables();
    });
    
    function initializeGRTTables() {
        $('.grt-ranking-table-wrapper').each(function() {
            var $wrapper = $(this);
            var blockId = $wrapper.attr('id');
            if (!blockId) return;
            
            var $configScript = $('.grt-block-config[data-block-id="' + blockId + '"]');
            if ($configScript.length === 0) return;
            
            try {
                var config = JSON.parse($configScript.text());
                loadTableData($wrapper, config);
            } catch (e) {
                console.error('GRT: Error parsing block config', e);
                showError($wrapper, 'Configuration error');
            }
        });
    }
    
    function loadTableData($wrapper, config) {
        var $loading = $wrapper.find('.grt-table-loading');
        var $content = $wrapper.find('.grt-table-content');
        var $fallback = $wrapper.find('.grt-table-fallback');
        var $locationText = $wrapper.find('.grt-user-location');
        
        // Show loading state
        $loading.show();
        $content.hide();
        $fallback.hide();
        
        // Step 1: Detect user location
        detectUserLocation()
            .then(function(location) {
                if (!location || !location.state_code) {
                    throw new Error('Location detection failed');
                }
                
                // Update location display
                var locationString = location.state;
                if (location.city) {
                    locationString = location.city + ', ' + location.state;
                }
                $locationText.text(locationString + ' (' + location.state_code + ')');
                
                // Step 2: Load filtered sites
                return loadFilteredSites(config, location.state_code);
            })
            .then(function(response) {
                $loading.hide();
                
                if (response.count > 0) {
                    $content.html(response.html).show();
                    // Add click tracking
                    trackSiteClicks($content);
                } else {
                    showFallback($wrapper, 'No gambling sites are available in your location.');
                }
            })
            .catch(function(error) {
                console.error('GRT: Error loading table data', error);
                $loading.hide();
                showFallback($wrapper, 'Unable to load ranking data. Please try refreshing the page.');
            });
    }
    
    function detectUserLocation() {
        return new Promise(function(resolve, reject) {
            $.post(grtAjax.ajaxUrl, {
                action: 'grt_detect_location',
                nonce: grtAjax.nonce
            })
            .done(function(response) {
                if (response.success && response.data) {
                    resolve(response.data);
                } else {
                    reject(new Error('Location detection failed'));
                }
            })
            .fail(function(xhr, status, error) {
                reject(new Error('Location detection request failed: ' + error));
            });
        });
    }
    
    function loadFilteredSites(config, userState) {
        return new Promise(function(resolve, reject) {
            $.post(grtAjax.ajaxUrl, {
                action: 'grt_get_filtered_sites',
                nonce: grtAjax.nonce,
                user_state: userState,
                selected_sites: config.selectedSites || [],
                max_display: config.maxDisplay || 5,
                show_ratings: config.showRatings !== false,
                show_bonuses: config.showBonuses !== false,
                show_licenses: config.showLicenses === true
            })
            .done(function(response) {
                if (response.success) {
                    resolve(response.data);
                } else {
                    reject(new Error('Failed to load sites'));
                }
            })
            .fail(function(xhr, status, error) {
                reject(new Error('Sites request failed: ' + error));
            });
        });
    }
    
    function showFallback($wrapper, message) {
        var $fallback = $wrapper.find('.grt-table-fallback');
        var $fallbackMessage = $fallback.find('.grt-fallback-message p');
        
        if (message) {
            $fallbackMessage.text(message);
        }
        
        $fallback.show();
    }
    
    function showError($wrapper, message) {
        var $loading = $wrapper.find('.grt-table-loading');
        var $content = $wrapper.find('.grt-table-content');
        var $fallback = $wrapper.find('.grt-table-fallback');
        
        $loading.hide();
        $content.hide();
        
        var $fallbackMessage = $fallback.find('.grt-fallback-message');
        $fallbackMessage.find('h4').text('Error Loading Table');
        $fallbackMessage.find('p').text(message || 'An error occurred while loading the ranking table.');
        
        $fallback.show();
    }
    
    function trackSiteClicks($content) {
        $content.find('.grt-visit-site-btn').on('click', function(e) {
            var $btn = $(this);
            var $row = $btn.closest('.grt-site-row');
            var rank = $row.data('rank') || 0;
            var siteName = $row.find('.grt-site-name').text() || 'Unknown';
            
            // Track click event
            console.log('GRT: Site click tracked', {
                rank: rank,
                siteName: siteName,
                url: $btn.attr('href')
            });
        });
    }
    
    // Refresh tables function
    window.grtRefreshTables = function() {
        $('.grt-ranking-table-wrapper').each(function() {
            var $wrapper = $(this);
            var $configScript = $('.grt-block-config[data-block-id="' + $wrapper.attr('id') + '"]');
            
            if ($configScript.length > 0) {
                try {
                    var config = JSON.parse($configScript.text());
                    loadTableData($wrapper, config);
                } catch (e) {
                    console.error('GRT: Error refreshing table', e);
                }
            }
        });
    };
    
})(jQuery);