<?php
/**
 * Frontend Class
 * Handles frontend functionality and scripts
 */

if (!defined('ABSPATH')) {
    exit;
}

class GRT_Frontend {
    
    public function __construct() {
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('wp_footer', array($this, 'add_geo_detection_script'));
        add_shortcode('grt_table', array($this, 'render_shortcode'));
    }
    
    /**
     * Enqueue frontend scripts and styles
     */
    public function enqueue_scripts() {
        // Always enqueue scripts if the plugin is active - we'll check for blocks in the footer
        wp_enqueue_script(
            'grt-frontend',
            GRT_PLUGIN_URL . 'assets/js/frontend.js',
            array('jquery'),
            GRT_VERSION,
            true
        );
        
        wp_enqueue_style(
            'grt-frontend',
            GRT_PLUGIN_URL . 'assets/css/frontend.css',
            array(),
            GRT_VERSION
        );
        
        // Localize script with AJAX data
        wp_localize_script('grt-frontend', 'grtAjax', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('grt_nonce'),
            'forcedIp' => $this->get_forced_test_ip(),
            'strings' => array(
                'loading' => __('Loading...', 'geo-ranking-tables'),
                'detecting_location' => __('Detecting your location...', 'geo-ranking-tables'),
                'location_detected' => __('Location detected:', 'geo-ranking-tables'),
                'no_sites_available' => __('No gambling sites available for your location.', 'geo-ranking-tables'),
                'error_loading' => __('Error loading ranking table. Please try again.', 'geo-ranking-tables'),
                'visit_site' => __('Visit Site', 'geo-ranking-tables'),
            )
        ));
    }
    
    /**
     * Check if current page has GRT blocks
     */
    private function page_has_grt_blocks() {
        global $post;
        
        if (!$post || !has_blocks($post->post_content)) {
            return false;
        }
        
        // Check if any block is our geo-ranking-tables block
        $blocks = parse_blocks($post->post_content);
        return $this->blocks_contain_grt($blocks);
    }
    
    /**
     * Recursively check if blocks contain our GRT blocks
     */
    private function blocks_contain_grt($blocks) {
        foreach ($blocks as $block) {
            if ($block['blockName'] === 'geo-ranking-tables/ranking-table') {
                return true;
            }
            
            // Check inner blocks
            if (!empty($block['innerBlocks'])) {
                if ($this->blocks_contain_grt($block['innerBlocks'])) {
                    return true;
                }
            }
        }
        
        return false;
    }
    
    /**
     * Get forced test IP from URL parameter
     */
    private function get_forced_test_ip() {
        if (!isset($_GET['ftm-ip'])) {
            return false;
        }
        
        $test_ip = sanitize_text_field($_GET['ftm-ip']);
        
        // Validate IPv4 or IPv6 address
        if (filter_var($test_ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4 | FILTER_FLAG_IPV6)) {
            return $test_ip;
        }
        
        return false;
    }
    
    /**
     * Add geo detection script to footer
     */
    public function add_geo_detection_script() {
        if (!$this->page_has_grt_blocks()) {
            return;
        }
        ?>
        <script>
        // Geo Detection and Table Loading
        jQuery(document).ready(function($) {
            console.log('GRT: Initializing geo-ranking tables...');
            console.log('GRT: Looking for elements with class .grt-ranking-table-wrapper');
            console.log('GRT: Found', $('.grt-ranking-table-wrapper').length, 'table wrappers');
            
            // Debug: Show all elements that might be related
            $('[class*="grt"]').each(function() {
                console.log('GRT: Found element with class:', this.className);
            });
            
            // Initialize all GRT tables on the page
            $('.grt-ranking-table-wrapper').each(function() {
                var $wrapper = $(this);
                var blockId = $wrapper.attr('id');
                console.log('GRT: Found table wrapper with ID:', blockId);
                
                var configScript = $('.grt-block-config[data-block-id="' + blockId + '"]');
                
                if (configScript.length > 0) {
                    try {
                        var config = JSON.parse(configScript.text());
                        console.log('GRT: Config loaded:', config);
                        initializeGRTTable($wrapper, config);
                    } catch (e) {
                        console.error('GRT: Error parsing block config:', e);
                        showTableError($wrapper);
                    }
                } else {
                    console.error('GRT: No config script found for block:', blockId);
                    showTableError($wrapper);
                }
            });
            
            function initializeGRTTable($wrapper, config) {
                console.log('GRT: Initializing table with config:', config);
                
                var $loading = $wrapper.find('.grt-table-loading');
                var $content = $wrapper.find('.grt-table-content');
                var $fallback = $wrapper.find('.grt-table-fallback');
                var $locationText = $wrapper.find('.grt-user-location');
                
                // Show loading state
                $loading.show();
                $content.hide();
                $fallback.hide();
                
                // Detect user location
                detectUserLocation(function(location) {
                    console.log('GRT: Location detected:', location);
                    
                    if (location && location.state_code) {
                        var locationString = location.state;
                        if (location.city) {
                            locationString = location.city + ', ' + location.state;
                        }
                        
                        // Add test mode indicator if using forced IP
                        if (location.test_mode || location.forced_ip) {
                            locationString += ' 🧪';
                            $locationText.html(locationString + ' (' + location.state_code + ') <span style="color: #d68910; font-size: 0.8em; font-weight: bold;">[TEST MODE]</span>');
                        } else {
                            $locationText.text(locationString + ' (' + location.state_code + ')');
                        }
                        
                        // Load filtered sites
                        loadFilteredSites($wrapper, config, location.state_code, function(success, data) {
                            $loading.hide();
                            
                            if (success && data.count > 0) {
                                console.log('GRT: Loaded', data.count, 'sites');
                                $content.html(data.html).show();
                            } else {
                                console.log('GRT: No sites available, showing fallback');
                                $fallback.show();
                            }
                        });
                    } else {
                        // Location detection failed, show fallback
                        console.log('GRT: Location detection failed, showing fallback');
                        $loading.hide();
                        $locationText.text('Unknown');
                        $fallback.show();
                    }
                });
            }
            
            function detectUserLocation(callback) {
                console.log('GRT: Detecting user location...');
                
                // Check if we have a forced IP from URL parameter
                if (grtAjax.forcedIp) {
                    console.log('GRT: Using forced test IP:', grtAjax.forcedIp);
                    
                    // Make the AJAX call with forced IP
                    $.post(grtAjax.ajaxUrl, {
                        action: 'grt_detect_location',
                        nonce: grtAjax.nonce,
                        forced_ip: grtAjax.forcedIp
                    }, function(response) {
                        console.log('GRT: Forced IP location response:', response);
                        
                        if (response.success) {
                            // Add test mode indicator to the response
                            response.data.test_mode = true;
                            response.data.forced_ip = grtAjax.forcedIp;
                            callback(response.data);
                        } else {
                            console.error('GRT: Forced IP location detection failed:', response);
                            callback(null);
                        }
                    }).fail(function(xhr, status, error) {
                        console.error('GRT: Forced IP location request failed:', error);
                        callback(null);
                    });
                } else {
                    // Normal location detection
                    $.post(grtAjax.ajaxUrl, {
                        action: 'grt_detect_location',
                        nonce: grtAjax.nonce
                    }, function(response) {
                        console.log('GRT: Location response:', response);
                        
                        if (response.success) {
                            callback(response.data);
                        } else {
                            console.error('GRT: Location detection failed:', response);
                            callback(null);
                        }
                    }).fail(function(xhr, status, error) {
                        console.error('GRT: Location request failed:', error);
                        callback(null);
                    });
                }
            }
            
            function loadFilteredSites($wrapper, config, userState, callback) {
                console.log('GRT: Loading filtered sites for state:', userState);
                
                $.post(grtAjax.ajaxUrl, {
                    action: 'grt_get_filtered_sites',
                    nonce: grtAjax.nonce,
                    user_state: userState,
                    selected_sites: config.selectedSites,
                    max_display: config.maxDisplay,
                    show_ratings: config.showRatings,
                    show_bonuses: config.showBonuses,
                    show_licenses: config.showLicenses
                }, function(response) {
                    console.log('GRT: Sites response:', response);
                    
                    if (response.success) {
                        callback(true, response.data);
                    } else {
                        console.error('GRT: Failed to load sites:', response);
                        callback(false, null);
                    }
                }).fail(function(xhr, status, error) {
                    console.error('GRT: Sites request failed:', error);
                    callback(false, null);
                });
            }
            
            function showTableError($wrapper) {
                var $loading = $wrapper.find('.grt-table-loading');
                var $content = $wrapper.find('.grt-table-content');
                var $fallback = $wrapper.find('.grt-table-fallback');
                
                $loading.hide();
                $content.hide();
                $fallback.find('.grt-fallback-message h4').text('Error Loading Table');
                $fallback.show();
            }
        });
        </script>
        <?php
    }
    
    /**
     * Render shortcode for managed tables
     * 
     * @param array $atts Shortcode attributes
     * @return string HTML output
     */
    public function render_shortcode($atts) {
        // Parse attributes
        $atts = shortcode_atts(array(
            'id' => 0,
            'slug' => '',
            'override_max' => 0,
            'override_name' => '',
            'override_ratings' => '',
            'override_bonuses' => '',
            'override_licenses' => '',
            'override_debug' => '',
        ), $atts, 'grt_table');
        
        // Get table ID from slug if provided
        $table_id = intval($atts['id']);
        if (empty($table_id) && !empty($atts['slug'])) {
            $table_by_slug = GRT_ManagedTables::get_by_slug($atts['slug']);
            if ($table_by_slug) {
                $table_id = $table_by_slug['id'];
            }
        }
        
        // If no table ID, show error
        if (empty($table_id)) {
            return '<div class="grt-error" style="padding: 15px; background: #ffe6e6; border: 1px solid #d63638; border-radius: 4px; color: #d63638;">' .
                   __('Invalid table ID or slug provided in shortcode.', 'geo-ranking-tables') .
                   '</div>';
        }
        
        // Get managed table configuration
        $config = GRT_ManagedTables::get_config($table_id);
        
        if (!$config) {
            return '<div class="grt-error" style="padding: 15px; background: #ffe6e6; border: 1px solid #d63638; border-radius: 4px; color: #d63638;">' .
                   __('Managed table not found or not published.', 'geo-ranking-tables') .
                   '</div>';
        }
        
        // Build overrides array
        $overrides = array();
        
        if (!empty($atts['override_max'])) {
            $overrides['maxDisplay'] = intval($atts['override_max']);
        }
        
        if (!empty($atts['override_name'])) {
            $overrides['tableName'] = sanitize_text_field($atts['override_name']);
        }
        
        if ($atts['override_ratings'] !== '') {
            $overrides['showRatings'] = filter_var($atts['override_ratings'], FILTER_VALIDATE_BOOLEAN);
        }
        
        if ($atts['override_bonuses'] !== '') {
            $overrides['showBonuses'] = filter_var($atts['override_bonuses'], FILTER_VALIDATE_BOOLEAN);
        }
        
        if ($atts['override_licenses'] !== '') {
            $overrides['showLicenses'] = filter_var($atts['override_licenses'], FILTER_VALIDATE_BOOLEAN);
        }
        
        if ($atts['override_debug'] !== '') {
            $overrides['showDebugInfo'] = filter_var($atts['override_debug'], FILTER_VALIDATE_BOOLEAN);
        }
        
        // Render using the table renderer
        $renderer = new GRT_TableRenderer();
        return $renderer->render($config, $overrides);
    }
}