<?php
/**
 * Frontend Class
 * Handles frontend functionality and scripts
 */

if (!defined('ABSPATH')) {
    exit;
}

class GRT_Frontend {
    
    public function __construct() {
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('wp_footer', array($this, 'add_geo_detection_script'));
        add_shortcode('grt_table', array($this, 'render_shortcode'));
    }
    
    /**
     * Enqueue frontend scripts and styles
     */
    public function enqueue_scripts() {
        // Always enqueue scripts if the plugin is active - we'll check for blocks in the footer
        wp_enqueue_script(
            'grt-frontend',
            GRT_PLUGIN_URL . 'assets/js/frontend.js',
            array('jquery'),
            GRT_VERSION,
            true
        );
        
        wp_enqueue_style(
            'grt-frontend',
            GRT_PLUGIN_URL . 'assets/css/frontend.css',
            array(),
            GRT_VERSION
        );
        
        // Get settings
        $settings = get_option('grt_settings', array());

        // Localize script with AJAX data
        wp_localize_script('grt-frontend', 'grtAjax', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('grt_nonce'),
            'forcedIp' => $this->get_forced_test_ip(),
            'debugMode' => isset($settings['debug_mode']) ? (bool)$settings['debug_mode'] : false,
            'pageId' => get_the_ID() ?: 0,
            'userState' => '', // Will be populated after geo-detection
            'strings' => array(
                'loading' => __('Loading...', 'geo-ranking-tables'),
                'detecting_location' => __('Detecting your location...', 'geo-ranking-tables'),
                'location_detected' => __('Location detected:', 'geo-ranking-tables'),
                'no_sites_available' => __('No gambling sites available for your location.', 'geo-ranking-tables'),
                'error_loading' => __('Error loading ranking table. Please try again.', 'geo-ranking-tables'),
                'visit_site' => __('Visit Site', 'geo-ranking-tables'),
            )
        ));
    }
    
    /**
     * Check if current page has GRT blocks
     */
    private function page_has_grt_blocks() {
        global $post;
        
        if (!$post || !has_blocks($post->post_content)) {
            return false;
        }
        
        // Check if any block is our geo-ranking-tables block
        $blocks = parse_blocks($post->post_content);
        return $this->blocks_contain_grt($blocks);
    }
    
    /**
     * Recursively check if blocks contain our GRT blocks
     */
    private function blocks_contain_grt($blocks) {
        foreach ($blocks as $block) {
            if ($block['blockName'] === 'geo-ranking-tables/ranking-table') {
                return true;
            }
            
            // Check inner blocks
            if (!empty($block['innerBlocks'])) {
                if ($this->blocks_contain_grt($block['innerBlocks'])) {
                    return true;
                }
            }
        }
        
        return false;
    }
    
    /**
     * Get forced test IP from URL parameter
     */
    private function get_forced_test_ip() {
        if (!isset($_GET['ftm-ip'])) {
            return false;
        }
        
        $test_ip = sanitize_text_field($_GET['ftm-ip']);
        
        // Validate IPv4 or IPv6 address
        if (filter_var($test_ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4 | FILTER_FLAG_IPV6)) {
            return $test_ip;
        }
        
        return false;
    }
    
    /**
     * Add geo detection script to footer
     *
     * Note: The main frontend logic is in assets/js/frontend.js
     * This method is kept for backwards compatibility but the inline script
     * has been removed to prevent race conditions with the main script.
     */
    public function add_geo_detection_script() {
        // Inline script removed - all functionality is now in assets/js/frontend.js
        // This prevents race conditions from having two scripts trying to
        // initialize the same table elements.
    }
    
    /**
     * Render shortcode for managed tables
     * 
     * @param array $atts Shortcode attributes
     * @return string HTML output
     */
    public function render_shortcode($atts) {
        // Parse attributes
        $atts = shortcode_atts(array(
            'id' => 0,
            'slug' => '',
            'override_max' => 0,
            'override_name' => '',
            'override_ratings' => '',
            'override_bonuses' => '',
            'override_licenses' => '',
            'override_debug' => '',
        ), $atts, 'grt_table');
        
        // Get table ID from slug if provided
        $table_id = intval($atts['id']);
        if (empty($table_id) && !empty($atts['slug'])) {
            $table_by_slug = GRT_ManagedTables::get_by_slug($atts['slug']);
            if ($table_by_slug) {
                $table_id = $table_by_slug['id'];
            }
        }
        
        // If no table ID, show error
        if (empty($table_id)) {
            return '<div class="grt-error" style="padding: 15px; background: #ffe6e6; border: 1px solid #d63638; border-radius: 4px; color: #d63638;">' .
                   __('Invalid table ID or slug provided in shortcode.', 'geo-ranking-tables') .
                   '</div>';
        }
        
        // Get managed table configuration
        $config = GRT_ManagedTables::get_config($table_id);
        
        if (!$config) {
            return '<div class="grt-error" style="padding: 15px; background: #ffe6e6; border: 1px solid #d63638; border-radius: 4px; color: #d63638;">' .
                   __('Managed table not found or not published.', 'geo-ranking-tables') .
                   '</div>';
        }
        
        // Build overrides array
        $overrides = array();
        
        if (!empty($atts['override_max'])) {
            $overrides['maxDisplay'] = intval($atts['override_max']);
        }
        
        if (!empty($atts['override_name'])) {
            $overrides['tableName'] = sanitize_text_field($atts['override_name']);
        }
        
        if ($atts['override_ratings'] !== '') {
            $overrides['showRatings'] = filter_var($atts['override_ratings'], FILTER_VALIDATE_BOOLEAN);
        }
        
        if ($atts['override_bonuses'] !== '') {
            $overrides['showBonuses'] = filter_var($atts['override_bonuses'], FILTER_VALIDATE_BOOLEAN);
        }
        
        if ($atts['override_licenses'] !== '') {
            $overrides['showLicenses'] = filter_var($atts['override_licenses'], FILTER_VALIDATE_BOOLEAN);
        }
        
        if ($atts['override_debug'] !== '') {
            $overrides['showDebugInfo'] = filter_var($atts['override_debug'], FILTER_VALIDATE_BOOLEAN);
        }
        
        // Render using the table renderer
        $renderer = new GRT_TableRenderer();
        return $renderer->render($config, $overrides);
    }
}