<?php
/**
 * Theme Registry Class
 * Manages theme registration and retrieval for ranking tables
 */

if (!defined('ABSPATH')) {
    exit;
}

class GRT_ThemeRegistry {

    /**
     * Registered themes
     *
     * @var array
     */
    private static $themes = array();

    /**
     * Register a theme
     *
     * @param string $id Theme identifier
     * @param array $config Theme configuration
     */
    public static function register($id, $config) {
        $defaults = array(
            'id' => $id,
            'name' => $id,
            'description' => '',
            'css_file' => '',
            'template_class' => null,
            'has_custom_layout' => false,
            'supports' => array('ratings', 'bonuses', 'licenses'),
        );

        self::$themes[$id] = wp_parse_args($config, $defaults);
    }

    /**
     * Get a theme by ID
     *
     * @param string $id Theme identifier
     * @return array|null Theme config or null if not found
     */
    public static function get_theme($id) {
        return isset(self::$themes[$id]) ? self::$themes[$id] : null;
    }

    /**
     * Get all registered themes
     *
     * @return array All themes
     */
    public static function get_all_themes() {
        return self::$themes;
    }

    /**
     * Get the default theme ID
     *
     * @return string Default theme ID
     */
    public static function get_default() {
        return 'default';
    }

    /**
     * Check if a theme exists
     *
     * @param string $id Theme identifier
     * @return bool
     */
    public static function exists($id) {
        return isset(self::$themes[$id]);
    }

    /**
     * Get theme CSS file URL
     *
     * @param string $id Theme identifier
     * @return string|null CSS file URL or null
     */
    public static function get_css_url($id) {
        $theme = self::get_theme($id);

        if (!$theme || empty($theme['css_file'])) {
            return null;
        }

        return GRT_PLUGIN_URL . 'assets/css/' . $theme['css_file'];
    }

    /**
     * Get theme template class instance
     *
     * @param string $id Theme identifier
     * @return GRT_Theme_Template|null Template instance or null
     */
    public static function get_template_instance($id) {
        $theme = self::get_theme($id);

        if (!$theme || empty($theme['template_class'])) {
            return null;
        }

        $class = $theme['template_class'];

        if (!class_exists($class)) {
            return null;
        }

        return new $class();
    }

    /**
     * Register default themes
     * Called during plugin initialization
     */
    public static function register_default_themes() {
        // Default theme (current purple design)
        self::register('default', array(
            'name' => __('Default', 'geo-ranking-tables'),
            'description' => __('Classic design with compact rows', 'geo-ranking-tables'),
            'css_file' => 'themes/default.css',
            'template_class' => null,
            'has_custom_layout' => false,
            'supports' => array('ratings', 'bonuses', 'licenses'),
        ));

        // Review Style theme
        self::register('review-style', array(
            'name' => __('Review Style', 'geo-ranking-tables'),
            'description' => __('Card layout with pros/cons and key facts', 'geo-ranking-tables'),
            'css_file' => 'themes/review-style.css',
            'template_class' => 'GRT_Theme_ReviewStyle',
            'has_custom_layout' => true,
            'supports' => array('ratings', 'bonuses', 'pros_cons', 'key_facts', 'badges', 'review_link'),
        ));

        // GS Theme
        self::register('gs-theme', array(
            'name' => __('GS Theme', 'geo-ranking-tables'),
            'description' => __('GamblingSite.com style card layout', 'geo-ranking-tables'),
            'css_file' => 'themes/gs-theme.css',
            'template_class' => 'GRT_Theme_GSTheme',
            'has_custom_layout' => true,
            'supports' => array('ratings', 'bonuses', 'pros_cons', 'key_facts', 'badges', 'review_link'),
        ));

        // Allow other plugins/themes to register custom themes
        do_action('grt_register_themes');
    }
}
