<?php
/**
 * Abstract Theme Template Class
 * Base class for all theme templates
 */

if (!defined('ABSPATH')) {
    exit;
}

abstract class GRT_Theme_Template {

    /**
     * Render a site row
     *
     * @param array $site Site data
     * @param int $index Site index (0-based)
     * @param array $config Table configuration
     * @return string HTML output
     */
    abstract public function render_site_row($site, $index, $config);

    /**
     * Render the table header
     *
     * @param array $config Table configuration
     * @return string HTML output
     */
    abstract public function render_table_header($config);

    /**
     * Render the table footer
     *
     * @param array $config Table configuration
     * @return string HTML output
     */
    abstract public function render_table_footer($config);

    /**
     * Render the complete sites table
     *
     * @param array $sites_data Array of site data
     * @param array $config Table configuration
     * @return string HTML output
     */
    public function render_sites_table($sites_data, $config) {
        if (empty($sites_data)) {
            return '<div class="grt-no-results">' . __('No gambling sites available for your location.', 'geo-ranking-tables') . '</div>';
        }

        ob_start();
        ?>
        <div class="grt-sites-table">
            <?php foreach ($sites_data as $index => $site): ?>
                <?php echo $this->render_site_row($site, $index, $config); ?>
            <?php endforeach; ?>
        </div>
        <?php
        return ob_get_clean();
    }

    /**
     * Render star rating
     *
     * @param float $rating Rating value (0-5)
     * @return string HTML output
     */
    protected function render_star_rating($rating) {
        $full_stars = floor($rating);
        $half_star = ($rating - $full_stars) >= 0.5;
        $empty_stars = 5 - $full_stars - ($half_star ? 1 : 0);

        ob_start();
        echo '<div class="grt-stars">';

        // Full stars
        for ($i = 0; $i < $full_stars; $i++) {
            echo '<span class="grt-star grt-star-full">&#9733;</span>';
        }

        // Half star
        if ($half_star) {
            echo '<span class="grt-star grt-star-half">&#9733;</span>';
        }

        // Empty stars
        for ($i = 0; $i < $empty_stars; $i++) {
            echo '<span class="grt-star grt-star-empty">&#9734;</span>';
        }

        echo '</div>';
        return ob_get_clean();
    }

    /**
     * Render site logo or fallback initial
     *
     * @param array $site Site data
     * @return string HTML output
     */
    protected function render_logo($site) {
        ob_start();

        if (!empty($site['featured_image'])) {
            ?>
            <img src="<?php echo esc_url($site['featured_image']); ?>" alt="<?php echo esc_attr($site['site_name']); ?>" />
            <?php
        } else {
            $initial = !empty($site['site_name']) ? substr($site['site_name'], 0, 1) : '?';
            ?>
            <div class="grt-site-initial"><?php echo esc_html($initial); ?></div>
            <?php
        }

        return ob_get_clean();
    }

    /**
     * Render pros list
     *
     * @param array $site Site data
     * @return string HTML output
     */
    protected function render_pros_list($site) {
        if (empty($site['pros']) || !is_array($site['pros'])) {
            return '';
        }

        ob_start();
        ?>
        <ul class="grt-pros-list">
            <?php foreach ($site['pros'] as $pro): ?>
                <li class="grt-pro-item">
                    <span class="grt-checkmark">&#10003;</span>
                    <?php echo esc_html($pro); ?>
                </li>
            <?php endforeach; ?>
        </ul>
        <?php
        return ob_get_clean();
    }

    /**
     * Render cons list
     *
     * @param array $site Site data
     * @return string HTML output
     */
    protected function render_cons_list($site) {
        if (empty($site['cons']) || !is_array($site['cons'])) {
            return '';
        }

        ob_start();
        ?>
        <ul class="grt-cons-list">
            <?php foreach ($site['cons'] as $con): ?>
                <li class="grt-con-item">
                    <span class="grt-xmark">&#10007;</span>
                    <?php echo esc_html($con); ?>
                </li>
            <?php endforeach; ?>
        </ul>
        <?php
        return ob_get_clean();
    }

    /**
     * Render key facts section
     *
     * @param array $site Site data
     * @return string HTML output
     */
    protected function render_key_facts($site) {
        $has_facts = !empty($site['payout_speed']) || !empty($site['compatibility']) || !empty($site['payout_percentage']);

        if (!$has_facts) {
            return '';
        }

        ob_start();
        ?>
        <div class="grt-key-facts">
            <span class="grt-key-facts-label"><?php _e('Key Facts:', 'geo-ranking-tables'); ?></span>

            <?php if (!empty($site['payout_speed'])): ?>
                <div class="grt-fact">
                    <span class="grt-fact-label"><?php _e('Payout Speed:', 'geo-ranking-tables'); ?></span>
                    <span class="grt-fact-value"><?php echo esc_html($site['payout_speed']); ?></span>
                </div>
            <?php endif; ?>

            <?php if (!empty($site['compatibility']) && is_array($site['compatibility'])): ?>
                <div class="grt-fact">
                    <span class="grt-fact-label"><?php _e('Compatibility:', 'geo-ranking-tables'); ?></span>
                    <span class="grt-fact-value"><?php echo esc_html($this->format_compatibility($site['compatibility'])); ?></span>
                </div>
            <?php endif; ?>

            <?php if (!empty($site['payout_percentage'])): ?>
                <div class="grt-fact">
                    <span class="grt-fact-label"><?php _e('Payout Percentage:', 'geo-ranking-tables'); ?></span>
                    <span class="grt-fact-value"><?php echo esc_html($site['payout_percentage']); ?></span>
                </div>
            <?php endif; ?>
        </div>
        <?php
        return ob_get_clean();
    }

    /**
     * Format compatibility array for display
     *
     * @param array $compatibility Platform codes
     * @return string Formatted string
     */
    protected function format_compatibility($compatibility) {
        $labels = array(
            'ios' => 'iOS',
            'android' => 'Android',
            'windows' => 'Windows',
            'mac' => 'Mac',
        );

        $formatted = array();
        foreach ($compatibility as $platform) {
            if (isset($labels[$platform])) {
                $formatted[] = $labels[$platform];
            }
        }

        return implode(', ', $formatted);
    }

    /**
     * Render disclaimer section with expandable toggle
     *
     * @param array $site Site data
     * @return string HTML output
     */
    protected function render_disclaimer($site) {
        if (empty($site['disclaimer'])) {
            return '';
        }

        ob_start();
        ?>
        <div class="grt-site-disclaimer">
            <a href="#" class="grt-disclaimer-toggle">
                <span><?php _e('More Details', 'geo-ranking-tables'); ?></span>
                <span class="grt-disclaimer-arrow">&#9660;</span>
            </a>
            <div class="grt-disclaimer-content" style="display: none;">
                <?php echo wp_kses_post($site['disclaimer']); ?>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }
}
