<?php
/**
 * Admin Class
 * Handles admin interface and settings
 */

if (!defined('ABSPATH')) {
    exit;
}

class GRT_Admin {
    
    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_menu', array($this, 'reorder_admin_menu'), 999);
        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        add_action('wp_ajax_grt_lookup_ip', array($this, 'ajax_lookup_ip'));
        add_action('wp_ajax_grt_create_sample_data', array($this, 'ajax_create_sample_data'));
        add_action('wp_ajax_grt_delete_sample_data', array($this, 'ajax_delete_sample_data'));
        add_action('wp_ajax_grt_purge_click_data', array($this, 'ajax_purge_click_data'));
        add_action('wp_ajax_grt_export_clicks', array($this, 'ajax_export_clicks'));

        // Cache invalidation for active tables count
        add_action('save_post', array($this, 'invalidate_active_tables_cache'));
        add_action('delete_post', array($this, 'invalidate_active_tables_cache'));
        add_action('trash_post', array($this, 'invalidate_active_tables_cache'));
    }
    
    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_menu_page(
            __('Geo Ranking Tables', 'geo-ranking-tables'),
            __('Geo Rankings', 'geo-ranking-tables'),
            'manage_options',
            'geo-ranking-tables',
            array($this, 'admin_page'),
            'dashicons-location-alt',
            30
        );
        
        // Add dashboard as first submenu item (same slug as parent to make it the default)
        add_submenu_page(
            'geo-ranking-tables',
            __('Dashboard', 'geo-ranking-tables'),
            __('Dashboard', 'geo-ranking-tables'),
            'manage_options',
            'geo-ranking-tables',
            array($this, 'admin_page')
        );
        
        // Add Managed Tables submenu - links to the custom post type
        add_submenu_page(
            'geo-ranking-tables',
            __('Managed Tables', 'geo-ranking-tables'),
            __('Managed Tables', 'geo-ranking-tables'),
            'manage_options',
            'edit.php?post_type=grt_managed_table'
        );
        
        // Add "Add New Managed Table" submenu
        add_submenu_page(
            'geo-ranking-tables',
            __('Add New Managed Table', 'geo-ranking-tables'),
            __('Add New Table', 'geo-ranking-tables'),
            'manage_options',
            'post-new.php?post_type=grt_managed_table'
        );
        
        add_submenu_page(
            'geo-ranking-tables',
            __('Settings', 'geo-ranking-tables'),
            __('Settings', 'geo-ranking-tables'),
            'manage_options',
            'grt-settings',
            array($this, 'settings_page')
        );
        
        add_submenu_page(
            'geo-ranking-tables',
            __('Troubleshoot', 'geo-ranking-tables'),
            __('Troubleshoot', 'geo-ranking-tables'),
            'manage_options',
            'grt-troubleshoot',
            array($this, 'troubleshoot_page')
        );

        add_submenu_page(
            'geo-ranking-tables',
            __('Table Usage', 'geo-ranking-tables'),
            __('Table Usage', 'geo-ranking-tables'),
            'manage_options',
            'grt-table-usage',
            array($this, 'table_usage_page')
        );

        add_submenu_page(
            'geo-ranking-tables',
            __('Click Analytics', 'geo-ranking-tables'),
            __('Click Analytics', 'geo-ranking-tables'),
            'manage_options',
            'grt-analytics',
            array($this, 'analytics_page')
        );
    }

    /**
     * Reorder admin menu to put Dashboard first
     */
    public function reorder_admin_menu() {
        global $submenu;
        
        if (!isset($submenu['geo-ranking-tables'])) {
            return;
        }
        
        // WordPress stores submenu items with numeric keys based on position
        // We need to reorder so Dashboard appears first
        $menu_items = $submenu['geo-ranking-tables'];
        $reordered = array();
        
        // Find and add Dashboard first (it has the same slug as parent)
        foreach ($menu_items as $key => $item) {
            if ($item[2] === 'geo-ranking-tables') {
                $reordered[0] = $item;
                unset($menu_items[$key]);
                break;
            }
        }
        
        // Add remaining items
        $position = 1;
        foreach ($menu_items as $item) {
            $reordered[$position] = $item;
            $position++;
        }
        
        // Replace the submenu
        $submenu['geo-ranking-tables'] = $reordered;
    }
    
    /**
     * Register plugin settings
     */
    public function register_settings() {
        register_setting('grt_settings_group', 'grt_settings', array($this, 'sanitize_settings'));
        
        add_settings_section(
            'grt_general_section',
            __('General Settings', 'geo-ranking-tables'),
            array($this, 'general_section_callback'),
            'grt-settings'
        );
        
        add_settings_field(
            'geo_api_key',
            __('IP Geolocation API Key', 'geo-ranking-tables'),
            array($this, 'geo_api_key_callback'),
            'grt-settings',
            'grt_general_section'
        );
        
        add_settings_field(
            'fallback_enabled',
            __('Enable Fallback Tables', 'geo-ranking-tables'),
            array($this, 'fallback_enabled_callback'),
            'grt-settings',
            'grt_general_section'
        );
        
        add_settings_field(
            'cache_duration',
            __('Location Cache Duration (seconds)', 'geo-ranking-tables'),
            array($this, 'cache_duration_callback'),
            'grt-settings',
            'grt_general_section'
        );

        add_settings_field(
            'debug_mode',
            __('Debug Mode', 'geo-ranking-tables'),
            array($this, 'debug_mode_callback'),
            'grt-settings',
            'grt_general_section'
        );
    }
    
    /**
     * Main admin page
     */
    public function admin_page() {
        $stats = $this->get_plugin_stats();
        ?>
        <div class="wrap">
            <h1><?php _e('Geo Ranking Tables', 'geo-ranking-tables'); ?></h1>
            
            <div class="grt-admin-dashboard">
                <div class="grt-stats-grid">
                    <div class="grt-stat-card">
                        <h3><?php _e('Site Blocks', 'geo-ranking-tables'); ?></h3>
                        <div class="grt-stat-number"><?php echo esc_html($stats['site_blocks']); ?></div>
                        <p><?php _e('Total gambling sites configured', 'geo-ranking-tables'); ?></p>
                    </div>
                    
                    <div class="grt-stat-card">
                        <h3><?php _e('Active Tables', 'geo-ranking-tables'); ?></h3>
                        <div class="grt-stat-number"><?php echo esc_html($stats['active_tables']); ?></div>
                        <p><?php _e('Ranking tables currently displayed', 'geo-ranking-tables'); ?></p>
                    </div>
                    
                    <div class="grt-stat-card">
                        <h3><?php _e('States Covered', 'geo-ranking-tables'); ?></h3>
                        <div class="grt-stat-number"><?php echo esc_html($stats['states_covered']); ?></div>
                        <p><?php _e('US states with available sites', 'geo-ranking-tables'); ?></p>
                    </div>
                </div>
                
                <div class="grt-quick-actions">
                    <h2><?php _e('Quick Actions', 'geo-ranking-tables'); ?></h2>
                    <div class="grt-action-buttons">
                        <a href="<?php echo admin_url('post-new.php?post_type=grt_site_block'); ?>" class="button button-primary">
                            <?php _e('Add New Site Block', 'geo-ranking-tables'); ?>
                        </a>
                        <a href="<?php echo admin_url('edit.php?post_type=grt_site_block'); ?>" class="button">
                            <?php _e('Manage Site Blocks', 'geo-ranking-tables'); ?>
                        </a>
                        <a href="<?php echo admin_url('admin.php?page=grt-settings'); ?>" class="button">
                            <?php _e('Plugin Settings', 'geo-ranking-tables'); ?>
                        </a>
                        <?php
                        // Check if sample data exists
                        $sample_data_exists = $this->check_sample_data_exists();
                        if ($sample_data_exists):
                        ?>
                            <button type="button" id="grt-reset-sample-data" class="button button-secondary grt-sample-data-btn">
                                <span class="dashicons dashicons-update" style="margin-top: 3px;"></span>
                                <?php _e('Reset Sample Data', 'geo-ranking-tables'); ?>
                            </button>
                        <?php else: ?>
                            <button type="button" id="grt-create-sample-data" class="button button-secondary grt-sample-data-btn">
                                <span class="dashicons dashicons-download" style="margin-top: 3px;"></span>
                                <?php _e('Create Sample Data', 'geo-ranking-tables'); ?>
                            </button>
                        <?php endif; ?>
                    </div>
                    <?php if ($sample_data_exists): ?>
                        <div class="grt-sample-data-status" style="margin-top: 15px; padding: 10px; background: #e7f3ff; border-left: 4px solid #0073aa; border-radius: 4px;">
                            <p style="margin: 0; color: #0073aa;">
                                <span class="dashicons dashicons-yes-alt" style="margin-top: 2px;"></span>
                                <?php _e('Sample data is currently loaded. You can reset it to recreate the 5 sample gambling sites.', 'geo-ranking-tables'); ?>
                            </p>
                        </div>
                    <?php endif; ?>
                </div>
                
                <div class="grt-usage-instructions">
                    <h2><?php _e('How to Use', 'geo-ranking-tables'); ?></h2>
                    <ol>
                        <li><?php _e('Create Site Blocks by adding new gambling sites with their details and accepted states', 'geo-ranking-tables'); ?></li>
                        <li><?php _e('In any post or page, add the "Geo Ranking Table" block from the Gutenberg editor', 'geo-ranking-tables'); ?></li>
                        <li><?php _e('Select which Site Blocks to include in your ranking table', 'geo-ranking-tables'); ?></li>
                        <li><?php _e('Configure display settings like max number of sites and fallback options', 'geo-ranking-tables'); ?></li>
                        <li><?php _e('The table will automatically filter sites based on user location', 'geo-ranking-tables'); ?></li>
                    </ol>
                </div>
            </div>
        </div>
        
        <style>
        .grt-admin-dashboard {
            max-width: 1200px;
        }
        
        .grt-stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin: 20px 0;
        }
        
        .grt-stat-card {
            background: #fff;
            padding: 20px;
            border: 1px solid #ddd;
            border-radius: 5px;
            text-align: center;
        }
        
        .grt-stat-number {
            font-size: 2.5em;
            font-weight: bold;
            color: #0073aa;
        }
        
        .grt-quick-actions {
            background: #fff;
            padding: 20px;
            border: 1px solid #ddd;
            border-radius: 5px;
            margin: 20px 0;
        }
        
        .grt-action-buttons {
            display: flex;
            gap: 10px;
            flex-wrap: wrap;
        }
        
        .grt-usage-instructions {
            background: #fff;
            padding: 20px;
            border: 1px solid #ddd;
            border-radius: 5px;
            margin: 20px 0;
        }
        
        .grt-usage-instructions ol {
            padding-left: 20px;
        }
        
        .grt-usage-instructions li {
            margin-bottom: 10px;
        }
        </style>
        <?php
    }
    
    /**
     * Settings page
     */
    public function settings_page() {
        ?>
        <div class="wrap">
            <h1><?php _e('Geo Ranking Tables Settings', 'geo-ranking-tables'); ?></h1>
            
            <form method="post" action="options.php">
                <?php
                settings_fields('grt_settings_group');
                do_settings_sections('grt-settings');
                submit_button();
                ?>
            </form>
            
            <div class="grt-settings-info">
                <h2><?php _e('API Information', 'geo-ranking-tables'); ?></h2>
                <p><?php _e('This plugin uses IP geolocation services to detect user location. While basic detection works without an API key, you can improve accuracy by providing an API key from:', 'geo-ranking-tables'); ?></p>
                <ul>
                    <li><strong>IPGeolocation.io</strong> - <a href="https://ipgeolocation.io/" target="_blank">Get Free API Key</a></li>
                </ul>
                <p><?php _e('The plugin will automatically fall back to free services if no API key is provided.', 'geo-ranking-tables'); ?></p>
            </div>
        </div>
        
        <style>
        .grt-settings-info {
            background: #fff;
            padding: 20px;
            border: 1px solid #ddd;
            border-radius: 5px;
            margin-top: 20px;
        }
        </style>
        <?php
    }
    
    /**
     * Troubleshoot page
     */
    public function troubleshoot_page() {
        $settings = get_option('grt_settings', array());
        $api_key = isset($settings['geo_api_key']) ? $settings['geo_api_key'] : '';
        ?>
        <div class="wrap">
            <h1><?php _e('Geo Ranking Tables - Troubleshoot', 'geo-ranking-tables'); ?></h1>
            
            <div class="grt-troubleshoot-container">
                <!-- IP Location Lookup -->
                <div class="grt-troubleshoot-section">
                    <h2><?php _e('IP Address Location Lookup', 'geo-ranking-tables'); ?></h2>
                    <p><?php _e('Test geo-location detection by entering any IP address below. This will help you verify that location detection is working correctly.', 'geo-ranking-tables'); ?></p>
                    
                    <form id="grt-ip-lookup-form" class="grt-lookup-form">
                        <table class="form-table">
                            <tr>
                                <th scope="row">
                                    <label for="grt_test_ip"><?php _e('IP Address', 'geo-ranking-tables'); ?></label>
                                </th>
                                <td>
                                    <input type="text" id="grt_test_ip" name="test_ip" class="regular-text" placeholder="e.g., 8.8.8.8" />
                                    <button type="submit" class="button button-primary" id="grt-lookup-btn">
                                        <?php _e('Lookup Location', 'geo-ranking-tables'); ?>
                                    </button>
                                    <p class="description">
                                        <?php _e('Enter any valid IP address to test location detection. Try: 8.8.8.8 (Google), 1.1.1.1 (Cloudflare), or your own IP.', 'geo-ranking-tables'); ?>
                                    </p>
                                </td>
                            </tr>
                        </table>
                    </form>
                    
                    <div id="grt-lookup-results" class="grt-results-container" style="display: none;">
                        <!-- Results will be populated by JavaScript -->
                    </div>
                </div>
                
                <!-- Force Test Mode -->
                <div class="grt-troubleshoot-section">
                    <h2><?php _e('Force Test Mode (ftm-ip)', 'geo-ranking-tables'); ?></h2>
                    <p><?php _e('Test how your ranking tables appear to users from different locations by adding the "ftm-ip" parameter to any page URL.', 'geo-ranking-tables'); ?></p>
                    
                    <div class="grt-test-mode-info">
                        <h4><?php _e('How to Use:', 'geo-ranking-tables'); ?></h4>
                        <ol>
                            <li><?php _e('Add', 'geo-ranking-tables'); ?> <code>?ftm-ip=IP_ADDRESS</code> <?php _e('to any page URL with a ranking table', 'geo-ranking-tables'); ?></li>
                            <li><?php _e('The plugin will use that IP for location detection instead of the visitor\'s real IP', 'geo-ranking-tables'); ?></li>
                            <li><?php _e('You\'ll see a [TEST MODE] indicator in the location display', 'geo-ranking-tables'); ?></li>
                        </ol>
                        
                        <h4><?php _e('Example URLs:', 'geo-ranking-tables'); ?></h4>
                        <div class="grt-example-urls">
                            <?php $current_page_url = get_permalink(get_the_ID()) ?: home_url(); ?>
                            <p><strong><?php _e('Test from New York:', 'geo-ranking-tables'); ?></strong><br>
                            <code><?php echo esc_url($current_page_url); ?>?ftm-ip=8.8.8.8</code>
                            <button type="button" class="button button-small" onclick="grtTestUrl('<?php echo esc_js($current_page_url); ?>?ftm-ip=8.8.8.8')">
                                <?php _e('Test This URL', 'geo-ranking-tables'); ?>
                            </button></p>
                            
                            <p><strong><?php _e('Test from Los Angeles:', 'geo-ranking-tables'); ?></strong><br>
                            <code><?php echo esc_url($current_page_url); ?>?ftm-ip=1.1.1.1</code>
                            <button type="button" class="button button-small" onclick="grtTestUrl('<?php echo esc_js($current_page_url); ?>?ftm-ip=1.1.1.1')">
                                <?php _e('Test This URL', 'geo-ranking-tables'); ?>
                            </button></p>
                            
                            <p><strong><?php _e('Custom IP Test:', 'geo-ranking-tables'); ?></strong><br>
                            <input type="text" id="grt-custom-test-ip" placeholder="<?php _e('Enter IP address', 'geo-ranking-tables'); ?>" style="width: 200px;" />
                            <button type="button" class="button" onclick="grtTestCustomIP()">
                                <?php _e('Generate Test URL', 'geo-ranking-tables'); ?>
                            </button></p>
                        </div>
                        
                        <div class="notice notice-info inline" style="margin-top: 15px;">
                            <p>
                                <strong><?php _e('Note:', 'geo-ranking-tables'); ?></strong> 
                                <?php _e('Test mode URLs work on any page that contains a Geo Ranking Table block. The IP parameter accepts both IPv4 and IPv6 addresses.', 'geo-ranking-tables'); ?>
                            </p>
                        </div>
                    </div>
                </div>
                
                <!-- API Status -->
                <div class="grt-troubleshoot-section">
                    <h2><?php _e('API Configuration Status', 'geo-ranking-tables'); ?></h2>
                    <div class="grt-api-status">
                        <?php if (empty($api_key)): ?>
                            <div class="notice notice-warning inline">
                                <p>
                                    <strong><?php _e('No API Key Configured', 'geo-ranking-tables'); ?></strong><br>
                                    <?php _e('You are using basic IP detection. For better accuracy, configure an API key in Settings.', 'geo-ranking-tables'); ?>
                                    <a href="<?php echo admin_url('admin.php?page=grt-settings'); ?>" class="button button-secondary">
                                        <?php _e('Configure API Key', 'geo-ranking-tables'); ?>
                                    </a>
                                </p>
                            </div>
                        <?php else: ?>
                            <div class="notice notice-success inline">
                                <p>
                                    <strong><?php _e('API Key Configured', 'geo-ranking-tables'); ?></strong><br>
                                    <?php _e('Using IPGeolocation.io API for enhanced location detection.', 'geo-ranking-tables'); ?>
                                </p>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
                
                <!-- Quick Tests -->
                <div class="grt-troubleshoot-section">
                    <h2><?php _e('Quick Tests', 'geo-ranking-tables'); ?></h2>
                    <div class="grt-quick-tests">
                        <button type="button" class="button" onclick="grtTestIP('8.8.8.8')">
                            <?php _e('Test Google DNS (8.8.8.8)', 'geo-ranking-tables'); ?>
                        </button>
                        <button type="button" class="button" onclick="grtTestIP('1.1.1.1')">
                            <?php _e('Test Cloudflare DNS (1.1.1.1)', 'geo-ranking-tables'); ?>
                        </button>
                        <button type="button" class="button" onclick="grtTestCurrentIP()">
                            <?php _e('Test Your Current IP', 'geo-ranking-tables'); ?>
                        </button>
                    </div>
                </div>
                
                <!-- System Info -->
                <div class="grt-troubleshoot-section">
                    <h2><?php _e('System Information', 'geo-ranking-tables'); ?></h2>
                    <div class="grt-system-info">
                        <table class="widefat striped">
                            <tbody>
                                <tr>
                                    <td><strong><?php _e('Plugin Version', 'geo-ranking-tables'); ?></strong></td>
                                    <td><?php echo esc_html(GRT_VERSION); ?></td>
                                </tr>
                                <tr>
                                    <td><strong><?php _e('WordPress Version', 'geo-ranking-tables'); ?></strong></td>
                                    <td><?php echo esc_html(get_bloginfo('version')); ?></td>
                                </tr>
                                <tr>
                                    <td><strong><?php _e('PHP Version', 'geo-ranking-tables'); ?></strong></td>
                                    <td><?php echo esc_html(PHP_VERSION); ?></td>
                                </tr>
                                <tr>
                                    <td><strong><?php _e('Server IP', 'geo-ranking-tables'); ?></strong></td>
                                    <td><?php echo esc_html($_SERVER['SERVER_ADDR'] ?? 'Unknown'); ?></td>
                                </tr>
                                <tr>
                                    <td><strong><?php _e('User Agent', 'geo-ranking-tables'); ?></strong></td>
                                    <td><?php echo esc_html($_SERVER['HTTP_USER_AGENT'] ?? 'Unknown'); ?></td>
                                </tr>
                                <tr>
                                    <td><strong><?php _e('Remote Address Detection', 'geo-ranking-tables'); ?></strong></td>
                                    <td>
                                        <?php 
                                        $detected_ip = $this->get_current_user_ip();
                                        echo esc_html($detected_ip);
                                        if ($detected_ip === '127.0.0.1' || strpos($detected_ip, '192.168.') === 0 || strpos($detected_ip, '10.') === 0) {
                                            echo ' <span style="color: #d63638;">(Local/Private IP)</span>';
                                        }
                                        ?>
                                    </td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
        
        <style>
        .grt-troubleshoot-container {
            max-width: 1200px;
        }
        
        .grt-troubleshoot-section {
            background: #fff;
            margin: 20px 0;
            padding: 20px;
            border: 1px solid #ddd;
            border-radius: 5px;
            box-shadow: 0 1px 3px rgba(0,0,0,0.1);
        }
        
        .grt-troubleshoot-section h2 {
            margin: 0 0 15px 0;
            color: #23282d;
            border-bottom: 2px solid #0073aa;
            padding-bottom: 10px;
        }
        
        .grt-lookup-form {
            margin-bottom: 20px;
        }
        
        .grt-results-container {
            margin-top: 20px;
            padding: 15px;
            background: #f9f9f9;
            border: 1px solid #ddd;
            border-radius: 4px;
        }
        
        .grt-result-success {
            background: #d4edda;
            border-color: #c3e6cb;
            color: #155724;
        }
        
        .grt-result-error {
            background: #f8d7da;
            border-color: #f5c6cb;
            color: #721c24;
        }
        
        .grt-result-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 15px;
            margin-top: 15px;
        }
        
        .grt-result-item {
            background: #fff;
            padding: 10px;
            border-radius: 4px;
            border: 1px solid #ddd;
        }
        
        .grt-result-item strong {
            color: #0073aa;
        }
        
        .grt-quick-tests {
            display: flex;
            gap: 10px;
            flex-wrap: wrap;
        }
        
        .grt-api-status .notice {
            margin: 0;
        }
        
        .grt-loading {
            display: none;
            align-items: center;
            gap: 10px;
            color: #666;
        }
        
        .grt-loading.active {
            display: flex;
        }
        
        .grt-spinner {
            width: 16px;
            height: 16px;
            border: 2px solid #f3f3f3;
            border-top: 2px solid #0073aa;
            border-radius: 50%;
            animation: grt-spin 1s linear infinite;
        }
        
        @keyframes grt-spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        .grt-test-mode-info {
            background: #f8f9fa;
            padding: 15px;
            border-radius: 5px;
            border: 1px solid #dee2e6;
        }
        
        .grt-test-mode-info h4 {
            margin: 0 0 10px 0;
            color: #0073aa;
        }
        
        .grt-test-mode-info ol {
            margin: 0 0 15px 20px;
        }
        
        .grt-test-mode-info li {
            margin-bottom: 5px;
        }
        
        .grt-example-urls {
            background: #fff;
            padding: 15px;
            border-radius: 4px;
            border: 1px solid #ddd;
        }
        
        .grt-example-urls p {
            margin: 10px 0;
            padding: 10px;
            background: #f9f9f9;
            border-radius: 4px;
        }
        
        .grt-example-urls code {
            background: #e9ecef;
            padding: 4px 8px;
            border-radius: 3px;
            font-family: 'Courier New', monospace;
            word-break: break-all;
        }
        
        .button-small {
            font-size: 11px !important;
            height: auto !important;
            padding: 4px 8px !important;
            margin-left: 10px;
        }
        
        #grt-lookup-btn.loading {
            opacity: 0.6;
            pointer-events: none;
        }
        
        @media (max-width: 768px) {
            .grt-result-grid {
                grid-template-columns: 1fr;
            }
            
            .grt-quick-tests {
                flex-direction: column;
            }
        }
        </style>
        
        <script>
        jQuery(document).ready(function($) {
            $('#grt-ip-lookup-form').on('submit', function(e) {
                e.preventDefault();
                var ip = $('#grt_test_ip').val().trim();
                if (ip) {
                    grtLookupIP(ip);
                } else {
                    alert('<?php _e('Please enter a valid IP address', 'geo-ranking-tables'); ?>');
                }
            });
        });
        
        function grtLookupIP(ip) {
            var $ = jQuery;
            var $btn = $('#grt-lookup-btn');
            var $results = $('#grt-lookup-results');
            
            // Validate IP format
            if (!grtValidateIP(ip)) {
                alert('<?php _e('Please enter a valid IP address', 'geo-ranking-tables'); ?>');
                return;
            }
            
            // Set loading state
            $btn.addClass('loading').text('<?php _e('Looking up...', 'geo-ranking-tables'); ?>');
            $('#grt_test_ip').val(ip);
            
            // Make AJAX request
            $.post(ajaxurl, {
                action: 'grt_lookup_ip',
                ip: ip,
                nonce: '<?php echo wp_create_nonce('grt_lookup_ip'); ?>'
            }, function(response) {
                $btn.removeClass('loading').text('<?php _e('Lookup Location', 'geo-ranking-tables'); ?>');
                
                if (response.success) {
                    grtDisplayResults(response.data, ip);
                } else {
                    grtDisplayError(response.data || '<?php _e('Unknown error occurred', 'geo-ranking-tables'); ?>');
                }
            }).fail(function() {
                $btn.removeClass('loading').text('<?php _e('Lookup Location', 'geo-ranking-tables'); ?>');
                grtDisplayError('<?php _e('Request failed. Please try again.', 'geo-ranking-tables'); ?>');
            });
        }
        
        function grtTestIP(ip) {
            grtLookupIP(ip);
        }
        
        function grtTestCurrentIP() {
            // Try to get user's current IP
            jQuery.getJSON('https://api.ipify.org?format=json', function(data) {
                grtLookupIP(data.ip);
            }).fail(function() {
                alert('<?php _e('Could not detect your current IP address', 'geo-ranking-tables'); ?>');
            });
        }
        
        function grtValidateIP(ip) {
            var ipRegex = /^(?:(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}(?:25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)$/;
            return ipRegex.test(ip);
        }
        
        function grtDisplayResults(data, ip) {
            var $ = jQuery;
            var $results = $('#grt-lookup-results');
            
            var html = '<div class="grt-result-success">';
            html += '<h3><?php _e('Location Results for IP:', 'geo-ranking-tables'); ?> ' + ip + '</h3>';
            html += '<div class="grt-result-grid">';
            
            html += '<div class="grt-result-item">';
            html += '<strong><?php _e('Basic Info', 'geo-ranking-tables'); ?></strong><br>';
            html += '<?php _e('IP Address:', 'geo-ranking-tables'); ?> ' + (data.ip || ip) + '<br>';
            html += '<?php _e('ISP:', 'geo-ranking-tables'); ?> ' + (data.isp || 'Unknown') + '<br>';
            html += '<?php _e('Organization:', 'geo-ranking-tables'); ?> ' + (data.organization || 'Unknown');
            html += '</div>';
            
            html += '<div class="grt-result-item">';
            html += '<strong><?php _e('Location', 'geo-ranking-tables'); ?></strong><br>';
            html += '<?php _e('Country:', 'geo-ranking-tables'); ?> ' + (data.country_name || 'Unknown') + ' (' + (data.country_code2 || 'N/A') + ')<br>';
            html += '<?php _e('State/Region:', 'geo-ranking-tables'); ?> ' + (data.state_prov || 'Unknown') + ' (' + (data.state_code || 'N/A') + ')<br>';
            html += '<?php _e('City:', 'geo-ranking-tables'); ?> ' + (data.city || 'Unknown');
            html += '</div>';
            
            html += '<div class="grt-result-item">';
            html += '<strong><?php _e('Coordinates', 'geo-ranking-tables'); ?></strong><br>';
            html += '<?php _e('Latitude:', 'geo-ranking-tables'); ?> ' + (data.latitude || 'Unknown') + '<br>';
            html += '<?php _e('Longitude:', 'geo-ranking-tables'); ?> ' + (data.longitude || 'Unknown') + '<br>';
            html += '<?php _e('Timezone:', 'geo-ranking-tables'); ?> ' + (data.time_zone ? data.time_zone.name : 'Unknown');
            html += '</div>';
            
            html += '<div class="grt-result-item">';
            html += '<strong><?php _e('Detection Info', 'geo-ranking-tables'); ?></strong><br>';
            html += '<?php _e('API Service:', 'geo-ranking-tables'); ?> ' + (data.service || 'IPGeolocation.io') + '<br>';
            html += '<?php _e('Response Time:', 'geo-ranking-tables'); ?> ' + (data.response_time || 'Unknown') + '<br>';
            html += '<?php _e('Accuracy:', 'geo-ranking-tables'); ?> ' + (data.country_code2 && data.state_code ? '<?php _e('High', 'geo-ranking-tables'); ?>' : '<?php _e('Basic', 'geo-ranking-tables'); ?>');
            html += '</div>';
            
            html += '</div>';
            
            // Show gambling site availability for this location
            if (data.state_code) {
                html += '<div style="margin-top: 15px; padding: 10px; background: #e7f3ff; border-radius: 4px;">';
                html += '<strong><?php _e('Gambling Site Availability for this Location:', 'geo-ranking-tables'); ?></strong><br>';
                html += '<em><?php _e('This would show which of your configured gambling sites accept players from', 'geo-ranking-tables'); ?> ' + data.state_prov + ' (' + data.state_code + ')</em>';
                html += '</div>';
            }
            
            html += '</div>';
            
            $results.html(html).show();
        }
        
        function grtDisplayError(message) {
            var $ = jQuery;
            var $results = $('#grt-lookup-results');
            
            var html = '<div class="grt-result-error">';
            html += '<h3><?php _e('Lookup Failed', 'geo-ranking-tables'); ?></h3>';
            html += '<p>' + message + '</p>';
            html += '<p><em><?php _e('This could be due to: invalid IP address, API key issues, or network connectivity problems.', 'geo-ranking-tables'); ?></em></p>';
            html += '</div>';
            
            $results.html(html).show();
        }
        
        function grtTestUrl(url) {
            if (confirm('<?php _e('Open this test URL in a new tab?', 'geo-ranking-tables'); ?>')) {
                window.open(url, '_blank');
            }
        }
        
        function grtTestCustomIP() {
            var ip = document.getElementById('grt-custom-test-ip').value.trim();
            if (!ip) {
                alert('<?php _e('Please enter an IP address', 'geo-ranking-tables'); ?>');
                return;
            }
            
            if (!grtValidateIP(ip)) {
                alert('<?php _e('Please enter a valid IP address', 'geo-ranking-tables'); ?>');
                return;
            }
            
            var currentUrl = '<?php echo esc_js($current_page_url); ?>';
            var testUrl = currentUrl + (currentUrl.indexOf('?') > -1 ? '&' : '?') + 'ftm-ip=' + encodeURIComponent(ip);
            
            if (confirm('<?php _e('Open test URL with IP', 'geo-ranking-tables'); ?> ' + ip + ' <?php _e('in a new tab?', 'geo-ranking-tables'); ?>')) {
                window.open(testUrl, '_blank');
            }
        }
        </script>
        <?php
    }

    /**
     * Count all active table instances across the site
     * Counts Gutenberg blocks and shortcodes in published content
     *
     * @return int Total count of active table instances
     */
    private function count_active_tables() {
        $cache_key = 'grt_active_tables_count';
        $cached = get_transient($cache_key);

        if ($cached !== false) {
            return (int) $cached;
        }

        global $wpdb;
        $total_count = 0;

        // Get public post types, exclude plugin CPTs
        $post_types = get_post_types(array('public' => true), 'names');
        $post_types = array_unique(array_merge($post_types, array('post', 'page')));
        $post_types = array_diff($post_types, array('grt_site_block', 'grt_managed_table', 'attachment'));

        if (empty($post_types)) {
            set_transient($cache_key, 0, 300);
            return 0;
        }

        $placeholders = implode(',', array_fill(0, count($post_types), '%s'));

        // 1. Count Gutenberg blocks
        $block_pattern = '%<!-- wp:geo-ranking-tables/ranking-table%';
        $sql = $wpdb->prepare(
            "SELECT post_content FROM {$wpdb->posts}
             WHERE post_status = 'publish'
             AND post_type IN ($placeholders)
             AND post_content LIKE %s",
            array_merge($post_types, array($block_pattern))
        );

        $posts_with_blocks = $wpdb->get_col($sql);
        foreach ($posts_with_blocks as $content) {
            $total_count += substr_count($content, '<!-- wp:geo-ranking-tables/ranking-table');
        }

        // 2. Count shortcodes
        $shortcode_pattern = '%[grt_table %';
        $sql = $wpdb->prepare(
            "SELECT post_content FROM {$wpdb->posts}
             WHERE post_status = 'publish'
             AND post_type IN ($placeholders)
             AND post_content LIKE %s",
            array_merge($post_types, array($shortcode_pattern))
        );

        $posts_with_shortcodes = $wpdb->get_col($sql);
        foreach ($posts_with_shortcodes as $content) {
            preg_match_all('/\[grt_table\s+[^\]]*\]/', $content, $matches);
            $total_count += count($matches[0]);
        }

        // Cache for 5 minutes
        set_transient($cache_key, $total_count, 300);
        return $total_count;
    }

    /**
     * Invalidate active tables cache when posts change
     *
     * @param int $post_id Post ID
     */
    public function invalidate_active_tables_cache($post_id) {
        if (wp_is_post_revision($post_id)) {
            return;
        }
        delete_transient('grt_active_tables_count');
    }

    /**
     * Get plugin statistics
     */
    private function get_plugin_stats() {
        $site_blocks_count = wp_count_posts('grt_site_block');
        $published_blocks = isset($site_blocks_count->publish) ? $site_blocks_count->publish : 0;
        
        // Count active tables (blocks and shortcodes across all published content)
        $active_tables = $this->count_active_tables();
        
        // Count states covered
        $states_covered = 0;
        $all_states = array();
        
        $site_blocks = get_posts(array(
            'post_type' => 'grt_site_block',
            'post_status' => 'publish',
            'numberposts' => -1,
        ));
        
        foreach ($site_blocks as $block) {
            $states = get_post_meta($block->ID, '_grt_states_accepted', true);
            if (is_array($states)) {
                $all_states = array_merge($all_states, $states);
            }
        }
        
        $states_covered = count(array_unique($all_states));
        
        return array(
            'site_blocks' => $published_blocks,
            'active_tables' => $active_tables,
            'states_covered' => $states_covered,
        );
    }
    
    /**
     * Settings section callback
     */
    public function general_section_callback() {
        echo '<p>' . __('Configure the plugin settings below.', 'geo-ranking-tables') . '</p>';
    }
    
    /**
     * Geo API key field callback
     */
    public function geo_api_key_callback() {
        $options = get_option('grt_settings', array());
        $api_key = isset($options['geo_api_key']) ? $options['geo_api_key'] : '';
        ?>
        <input type="text" name="grt_settings[geo_api_key]" value="<?php echo esc_attr($api_key); ?>" class="regular-text" />
        <p class="description"><?php _e('Optional: IPGeolocation.io API key for enhanced location detection', 'geo-ranking-tables'); ?></p>
        <?php
    }
    
    /**
     * Fallback enabled field callback
     */
    public function fallback_enabled_callback() {
        $options = get_option('grt_settings', array());
        $fallback_enabled = isset($options['fallback_enabled']) ? $options['fallback_enabled'] : true;
        ?>
        <input type="checkbox" name="grt_settings[fallback_enabled]" value="1" <?php checked($fallback_enabled, 1); ?> />
        <label><?php _e('Show fallback message when no sites are available for user\'s location', 'geo-ranking-tables'); ?></label>
        <?php
    }
    
    /**
     * Cache duration field callback
     */
    public function cache_duration_callback() {
        $options = get_option('grt_settings', array());
        $cache_duration = isset($options['cache_duration']) ? $options['cache_duration'] : 3600;
        ?>
        <input type="number" name="grt_settings[cache_duration]" value="<?php echo esc_attr($cache_duration); ?>" min="300" max="86400" />
        <p class="description"><?php _e('How long to cache location data (300-86400 seconds)', 'geo-ranking-tables'); ?></p>
        <?php
    }

    /**
     * Debug mode field callback
     */
    public function debug_mode_callback() {
        $options = get_option('grt_settings', array());
        $debug_mode = isset($options['debug_mode']) ? $options['debug_mode'] : false;
        ?>
        <input type="checkbox" name="grt_settings[debug_mode]" value="1" <?php checked($debug_mode, 1); ?> />
        <label><?php _e('Enable debug logging in browser console (for development)', 'geo-ranking-tables'); ?></label>
        <?php
    }

    /**
     * Sanitize settings
     */
    public function sanitize_settings($input) {
        $sanitized = array();
        
        if (isset($input['geo_api_key'])) {
            $sanitized['geo_api_key'] = sanitize_text_field($input['geo_api_key']);
        }
        
        if (isset($input['fallback_enabled'])) {
            $sanitized['fallback_enabled'] = (bool)$input['fallback_enabled'];
        } else {
            $sanitized['fallback_enabled'] = false;
        }
        
        if (isset($input['cache_duration'])) {
            $cache_duration = intval($input['cache_duration']);
            $sanitized['cache_duration'] = max(300, min(86400, $cache_duration));
        }

        if (isset($input['debug_mode'])) {
            $sanitized['debug_mode'] = (bool)$input['debug_mode'];
        } else {
            $sanitized['debug_mode'] = false;
        }

        return $sanitized;
    }
    
    /**
     * Enqueue admin scripts
     */
    public function enqueue_admin_scripts($hook) {
        // Match plugin pages: geo-ranking-tables, grt_site_block, grt_managed_table, grt-import-export, grt-settings, grt-troubleshoot, grt-analytics
        if (strpos($hook, 'geo-ranking-tables') !== false ||
            strpos($hook, 'grt_site_block') !== false ||
            strpos($hook, 'grt_managed_table') !== false ||
            strpos($hook, 'grt-import-export') !== false ||
            strpos($hook, 'grt-settings') !== false ||
            strpos($hook, 'grt-troubleshoot') !== false ||
            strpos($hook, 'grt-analytics') !== false) {
            wp_enqueue_style('grt-admin', GRT_PLUGIN_URL . 'assets/css/admin.css', array(), GRT_VERSION);
            wp_enqueue_script('grt-admin', GRT_PLUGIN_URL . 'assets/js/admin.js', array('jquery'), GRT_VERSION, true);
            
            // Localize script for AJAX
            wp_localize_script('grt-admin', 'grtAdminAjax', array(
                'ajaxUrl' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('grt_sample_data_nonce'),
                'strings' => array(
                    'confirmReset' => __('Are you sure you want to delete all sample data and recreate it? This will remove the existing 5 sample gambling sites.', 'geo-ranking-tables'),
                    'creating' => __('Creating sample data...', 'geo-ranking-tables'),
                    'deleting' => __('Deleting sample data...', 'geo-ranking-tables'),
                    'success' => __('Success!', 'geo-ranking-tables'),
                    'error' => __('Error', 'geo-ranking-tables')
                )
            ));
        }
    }
    
    /**
     * AJAX handler for IP lookup
     */
    public function ajax_lookup_ip() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'grt_lookup_ip')) {
            wp_send_json_error('Security check failed');
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        $ip = sanitize_text_field($_POST['ip']);
        
        // Validate IP format
        if (!filter_var($ip, FILTER_VALIDATE_IP)) {
            wp_send_json_error('Invalid IP address format');
        }
        
        // Get location data
        $location_data = $this->lookup_ip_location($ip);
        
        if ($location_data) {
            wp_send_json_success($location_data);
        } else {
            wp_send_json_error('Unable to lookup location for this IP address');
        }
    }
    
    /**
     * Lookup IP location using IPGeolocation.io
     */
    private function lookup_ip_location($ip) {
        $settings = get_option('grt_settings', array());
        $api_key = isset($settings['geo_api_key']) ? trim($settings['geo_api_key']) : '';
        
        $start_time = microtime(true);
        
        if (!empty($api_key)) {
            // Use IPGeolocation.io API
            $url = 'https://api.ipgeolocation.io/ipgeo?apiKey=' . urlencode($api_key) . '&ip=' . urlencode($ip);
            
            $response = wp_remote_get($url, array(
                'timeout' => 10,
                'headers' => array(
                    'User-Agent' => 'WordPress GeoRankingTables Plugin'
                )
            ));
            
            if (is_wp_error($response)) {
                error_log('GRT: IPGeolocation.io API request failed: ' . $response->get_error_message());
                return $this->fallback_ip_lookup($ip, $start_time);
            }
            
            $response_code = wp_remote_retrieve_response_code($response);
            if ($response_code !== 200) {
                error_log('GRT: IPGeolocation.io API returned status: ' . $response_code);
                return $this->fallback_ip_lookup($ip, $start_time);
            }
            
            $body = wp_remote_retrieve_body($response);
            $data = json_decode($body, true);
            
            if (!$data || isset($data['error'])) {
                error_log('GRT: IPGeolocation.io API error: ' . print_r($data, true));
                return $this->fallback_ip_lookup($ip, $start_time);
            }
            
            // Add response time and service info
            $data['response_time'] = round((microtime(true) - $start_time) * 1000) . 'ms';
            $data['service'] = 'ipgeolocation.io';
            
            return $data;
        } else {
            // Fallback to free API
            return $this->fallback_ip_lookup($ip, $start_time);
        }
    }
    
    /**
     * Fallback IP lookup using free API
     */
    private function fallback_ip_lookup($ip, $start_time) {
        // Try ip-api.com (free service)
        $url = 'http://ip-api.com/json/' . urlencode($ip) . '?fields=status,country,countryCode,region,regionName,city,lat,lon,timezone,isp,org,query';
        
        $response = wp_remote_get($url, array(
            'timeout' => 10,
            'headers' => array(
                'User-Agent' => 'WordPress GeoRankingTables Plugin'
            )
        ));
        
        if (is_wp_error($response)) {
            return false;
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        if ($response_code !== 200) {
            return false;
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if (!$data || $data['status'] !== 'success') {
            return false;
        }
        
        // Normalize data to match IPGeolocation.io format
        $normalized_data = array(
            'ip' => $data['query'],
            'country_name' => $data['country'],
            'country_code2' => $data['countryCode'],
            'state_prov' => $data['regionName'],
            'state_code' => $this->convert_state_name_to_code($data['regionName']),
            'city' => $data['city'],
            'latitude' => $data['lat'],
            'longitude' => $data['lon'],
            'isp' => $data['isp'],
            'organization' => $data['org'],
            'time_zone' => array('name' => $data['timezone']),
            'response_time' => round((microtime(true) - $start_time) * 1000) . 'ms',
            'service' => 'ip-api.com (fallback)'
        );
        
        return $normalized_data;
    }
    
    /**
     * Convert state name to 2-letter code
     */
    private function convert_state_name_to_code($state_name) {
        $states = array(
            'Alabama' => 'AL', 'Alaska' => 'AK', 'Arizona' => 'AZ', 'Arkansas' => 'AR', 'California' => 'CA',
            'Colorado' => 'CO', 'Connecticut' => 'CT', 'Delaware' => 'DE', 'Florida' => 'FL', 'Georgia' => 'GA',
            'Hawaii' => 'HI', 'Idaho' => 'ID', 'Illinois' => 'IL', 'Indiana' => 'IN', 'Iowa' => 'IA',
            'Kansas' => 'KS', 'Kentucky' => 'KY', 'Louisiana' => 'LA', 'Maine' => 'ME', 'Maryland' => 'MD',
            'Massachusetts' => 'MA', 'Michigan' => 'MI', 'Minnesota' => 'MN', 'Mississippi' => 'MS', 'Missouri' => 'MO',
            'Montana' => 'MT', 'Nebraska' => 'NE', 'Nevada' => 'NV', 'New Hampshire' => 'NH', 'New Jersey' => 'NJ',
            'New Mexico' => 'NM', 'New York' => 'NY', 'North Carolina' => 'NC', 'North Dakota' => 'ND', 'Ohio' => 'OH',
            'Oklahoma' => 'OK', 'Oregon' => 'OR', 'Pennsylvania' => 'PA', 'Rhode Island' => 'RI', 'South Carolina' => 'SC',
            'South Dakota' => 'SD', 'Tennessee' => 'TN', 'Texas' => 'TX', 'Utah' => 'UT', 'Vermont' => 'VT',
            'Virginia' => 'VA', 'Washington' => 'WA', 'West Virginia' => 'WV', 'Wisconsin' => 'WI', 'Wyoming' => 'WY',
            'District of Columbia' => 'DC'
        );
        
        return isset($states[$state_name]) ? $states[$state_name] : '';
    }
    
    /**
     * Get current user IP for system info
     */
    private function get_current_user_ip() {
        $ip_headers = array(
            'HTTP_CF_CONNECTING_IP',
            'HTTP_CLIENT_IP',
            'HTTP_X_FORWARDED_FOR',
            'HTTP_X_FORWARDED',
            'HTTP_X_CLUSTER_CLIENT_IP',
            'HTTP_FORWARDED_FOR',
            'HTTP_FORWARDED',
            'REMOTE_ADDR'
        );
        
        foreach ($ip_headers as $header) {
            if (!empty($_SERVER[$header])) {
                $ip = $_SERVER[$header];
                if (strpos($ip, ',') !== false) {
                    $ip_list = explode(',', $ip);
                    $ip = trim($ip_list[0]);
                }
                if (filter_var($ip, FILTER_VALIDATE_IP)) {
                    return $ip;
                }
            }
        }
        
        return '127.0.0.1';
    }
    
    /**
     * Check if sample data exists
     */
    private function check_sample_data_exists() {
        // Check if we have the option flag
        if (get_option('grt_sample_data_created')) {
            // Also verify that at least some sample posts still exist
            $sample_posts = get_posts(array(
                'post_type' => 'grt_site_block',
                'meta_key' => '_grt_is_sample_data',
                'meta_value' => '1',
                'posts_per_page' => 1,
                'post_status' => 'publish'
            ));
            
            if (!empty($sample_posts)) {
                return true;
            } else {
                // Sample data was deleted manually, reset the flag
                delete_option('grt_sample_data_created');
                return false;
            }
        }
        
        return false;
    }
    
    /**
     * AJAX handler to create sample data
     */
    public function ajax_create_sample_data() {
        // Check nonce
        check_ajax_referer('grt_sample_data_nonce', 'nonce');
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions', 'geo-ranking-tables')));
        }
        
        // Include sample data file
        require_once GRT_PLUGIN_PATH . 'includes/sample-data.php';
        
        // Create sample data
        $result = grt_setup_sample_data();
        
        if ($result && !empty($result['site_ids'])) {
            // Mark as created
            update_option('grt_sample_data_created', true);
            
            wp_send_json_success(array(
                'message' => sprintf(
                    __('%d sample gambling sites created successfully!', 'geo-ranking-tables'),
                    count($result['site_ids'])
                ),
                'sites_created' => count($result['site_ids']),
                'page_url' => $result['page_url'],
                'manage_url' => admin_url('edit.php?post_type=grt_site_block')
            ));
        } else {
            wp_send_json_error(array('message' => __('Failed to create sample data. Please try again.', 'geo-ranking-tables')));
        }
    }
    
    /**
     * AJAX handler to delete sample data
     */
    public function ajax_delete_sample_data() {
        // Check nonce
        check_ajax_referer('grt_sample_data_nonce', 'nonce');
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions', 'geo-ranking-tables')));
        }
        
        // Include sample data file
        require_once GRT_PLUGIN_PATH . 'includes/sample-data.php';
        
        // Delete sample data
        $deleted_count = grt_delete_sample_data();
        
        if ($deleted_count > 0) {
            // Reset the flag
            delete_option('grt_sample_data_created');
            
            wp_send_json_success(array(
                'message' => sprintf(
                    __('%d sample site(s) deleted successfully!', 'geo-ranking-tables'),
                    $deleted_count
                ),
                'deleted_count' => $deleted_count
            ));
        } else {
            wp_send_json_error(array('message' => __('No sample data found to delete.', 'geo-ranking-tables')));
        }
    }

    /**
     * Get all posts/pages containing tables
     *
     * @return array Array of post data with table info
     */
    private function get_table_locations() {
        global $wpdb;

        $post_types = get_post_types(array('public' => true), 'names');
        $post_types = array_unique(array_merge($post_types, array('post', 'page')));
        $post_types = array_diff($post_types, array('grt_site_block', 'grt_managed_table', 'attachment'));

        if (empty($post_types)) {
            return array();
        }

        $placeholders = implode(',', array_fill(0, count($post_types), '%s'));

        // Find posts with blocks OR shortcodes
        $block_pattern = '%<!-- wp:geo-ranking-tables/ranking-table%';
        $shortcode_pattern = '%[grt_table %';

        $sql = $wpdb->prepare(
            "SELECT ID, post_title, post_type, post_content FROM {$wpdb->posts}
             WHERE post_status = 'publish'
             AND post_type IN ($placeholders)
             AND (post_content LIKE %s OR post_content LIKE %s)
             ORDER BY post_title ASC",
            array_merge($post_types, array($block_pattern, $shortcode_pattern))
        );

        $posts = $wpdb->get_results($sql);
        $locations = array();

        foreach ($posts as $post) {
            // Count tables in this post
            $block_count = substr_count($post->post_content, '<!-- wp:geo-ranking-tables/ranking-table');
            preg_match_all('/\[grt_table\s+[^\]]*\]/', $post->post_content, $matches);
            $shortcode_count = count($matches[0]);

            $locations[] = array(
                'id' => $post->ID,
                'title' => $post->post_title,
                'post_type' => $post->post_type,
                'table_count' => $block_count + $shortcode_count,
                'edit_url' => get_edit_post_link($post->ID, 'raw'),
            );
        }

        return $locations;
    }

    /**
     * Render table usage report page
     */
    public function table_usage_page() {
        $locations = $this->get_table_locations();
        ?>
        <div class="wrap">
            <h1><?php _e('Table Usage Report', 'geo-ranking-tables'); ?></h1>
            <p><?php _e('Published pages and posts containing Geo Ranking Tables.', 'geo-ranking-tables'); ?></p>

            <?php if (empty($locations)) : ?>
                <p><?php _e('No tables found in any published content.', 'geo-ranking-tables'); ?></p>
            <?php else : ?>
                <table class="wp-list-table widefat fixed striped">
                    <thead>
                        <tr>
                            <th><?php _e('Page/Post Title', 'geo-ranking-tables'); ?></th>
                            <th><?php _e('Type', 'geo-ranking-tables'); ?></th>
                            <th><?php _e('Tables', 'geo-ranking-tables'); ?></th>
                            <th><?php _e('Action', 'geo-ranking-tables'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($locations as $loc) : ?>
                            <tr>
                                <td><strong><?php echo esc_html($loc['title']); ?></strong></td>
                                <td><?php echo esc_html(ucfirst($loc['post_type'])); ?></td>
                                <td><?php echo esc_html($loc['table_count']); ?></td>
                                <td><a href="<?php echo esc_url($loc['edit_url']); ?>" class="button button-small"><?php _e('Edit', 'geo-ranking-tables'); ?></a></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
                <p class="description" style="margin-top: 10px;">
                    <?php printf(
                        __('Total: %d tables across %d pages', 'geo-ranking-tables'),
                        array_sum(array_column($locations, 'table_count')),
                        count($locations)
                    ); ?>
                </p>
            <?php endif; ?>
        </div>
        <?php
    }

    /**
     * Render click analytics page
     */
    public function analytics_page() {
        // Handle date range
        $end_date = isset($_GET['end_date']) ? sanitize_text_field($_GET['end_date']) : date('Y-m-d');
        $start_date = isset($_GET['start_date']) ? sanitize_text_field($_GET['start_date']) : date('Y-m-d', strtotime('-30 days'));

        // Validate dates
        if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $start_date)) {
            $start_date = date('Y-m-d', strtotime('-30 days'));
        }
        if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $end_date)) {
            $end_date = date('Y-m-d');
        }

        $reports = new GRT_ClickReports();
        $summary = $reports->get_summary_stats($start_date, $end_date);
        $by_site = $reports->get_clicks_by_site($start_date, $end_date, 10);
        $by_state = $reports->get_clicks_by_state($start_date, $end_date);
        $by_device = $reports->get_clicks_by_device($start_date, $end_date);
        $by_rank = $reports->get_clicks_by_rank($start_date, $end_date);
        $total_records = $reports->get_record_count();
        ?>
        <div class="wrap">
            <h1><?php _e('Click Analytics', 'geo-ranking-tables'); ?></h1>

            <!-- Date Range Filter -->
            <div class="grt-analytics-filters" style="background: #fff; padding: 15px; margin: 20px 0; border: 1px solid #ccd0d4; border-radius: 4px;">
                <form method="get" style="display: flex; align-items: center; gap: 15px; flex-wrap: wrap;">
                    <input type="hidden" name="page" value="grt-analytics">
                    <label>
                        <?php _e('Start Date:', 'geo-ranking-tables'); ?>
                        <input type="date" name="start_date" value="<?php echo esc_attr($start_date); ?>" style="margin-left: 5px;">
                    </label>
                    <label>
                        <?php _e('End Date:', 'geo-ranking-tables'); ?>
                        <input type="date" name="end_date" value="<?php echo esc_attr($end_date); ?>" style="margin-left: 5px;">
                    </label>
                    <button type="submit" class="button button-primary"><?php _e('Filter', 'geo-ranking-tables'); ?></button>
                    <a href="<?php echo admin_url('admin.php?page=grt-analytics'); ?>" class="button"><?php _e('Reset', 'geo-ranking-tables'); ?></a>
                </form>
            </div>

            <!-- Summary Cards -->
            <div class="grt-analytics-summary" style="display: grid; grid-template-columns: repeat(auto-fit, minmax(180px, 1fr)); gap: 15px; margin-bottom: 30px;">
                <div class="grt-stat-card" style="background: #fff; padding: 20px; border: 1px solid #ccd0d4; border-radius: 4px; text-align: center;">
                    <h3 style="margin: 0 0 10px; color: #1d2327; font-size: 14px;"><?php _e('Total Clicks', 'geo-ranking-tables'); ?></h3>
                    <div style="font-size: 32px; font-weight: 600; color: #2271b1;"><?php echo number_format($summary['total_clicks']); ?></div>
                </div>
                <div class="grt-stat-card" style="background: #fff; padding: 20px; border: 1px solid #ccd0d4; border-radius: 4px; text-align: center;">
                    <h3 style="margin: 0 0 10px; color: #1d2327; font-size: 14px;"><?php _e('Visit Clicks', 'geo-ranking-tables'); ?></h3>
                    <div style="font-size: 32px; font-weight: 600; color: #00a32a;"><?php echo number_format($summary['visit_clicks']); ?></div>
                </div>
                <div class="grt-stat-card" style="background: #fff; padding: 20px; border: 1px solid #ccd0d4; border-radius: 4px; text-align: center;">
                    <h3 style="margin: 0 0 10px; color: #1d2327; font-size: 14px;"><?php _e('Review Clicks', 'geo-ranking-tables'); ?></h3>
                    <div style="font-size: 32px; font-weight: 600; color: #dba617;"><?php echo number_format($summary['review_clicks']); ?></div>
                </div>
                <div class="grt-stat-card" style="background: #fff; padding: 20px; border: 1px solid #ccd0d4; border-radius: 4px; text-align: center;">
                    <h3 style="margin: 0 0 10px; color: #1d2327; font-size: 14px;"><?php _e('Unique Sessions', 'geo-ranking-tables'); ?></h3>
                    <div style="font-size: 32px; font-weight: 600; color: #8c8f94;"><?php echo number_format($summary['unique_sessions']); ?></div>
                </div>
            </div>

            <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px;">
                <!-- Top Sites -->
                <div style="background: #fff; padding: 20px; border: 1px solid #ccd0d4; border-radius: 4px;">
                    <h2 style="margin-top: 0;"><?php _e('Top Sites by Clicks', 'geo-ranking-tables'); ?></h2>
                    <?php if (empty($by_site)) : ?>
                        <p><?php _e('No click data available for this period.', 'geo-ranking-tables'); ?></p>
                    <?php else : ?>
                        <table class="wp-list-table widefat fixed striped">
                            <thead>
                                <tr>
                                    <th><?php _e('Site', 'geo-ranking-tables'); ?></th>
                                    <th style="width: 80px; text-align: right;"><?php _e('Total', 'geo-ranking-tables'); ?></th>
                                    <th style="width: 80px; text-align: right;"><?php _e('Visit', 'geo-ranking-tables'); ?></th>
                                    <th style="width: 80px; text-align: right;"><?php _e('Review', 'geo-ranking-tables'); ?></th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($by_site as $site) : ?>
                                    <tr>
                                        <td><?php echo esc_html($site['site_name'] ?: 'Unknown Site #' . $site['site_id']); ?></td>
                                        <td style="text-align: right;"><strong><?php echo number_format($site['total_clicks']); ?></strong></td>
                                        <td style="text-align: right;"><?php echo number_format($site['visit_clicks']); ?></td>
                                        <td style="text-align: right;"><?php echo number_format($site['review_clicks']); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php endif; ?>
                </div>

                <!-- Clicks by State -->
                <div style="background: #fff; padding: 20px; border: 1px solid #ccd0d4; border-radius: 4px;">
                    <h2 style="margin-top: 0;"><?php _e('Clicks by State', 'geo-ranking-tables'); ?></h2>
                    <?php if (empty($by_state)) : ?>
                        <p><?php _e('No geographic data available for this period.', 'geo-ranking-tables'); ?></p>
                    <?php else : ?>
                        <table class="wp-list-table widefat fixed striped">
                            <thead>
                                <tr>
                                    <th><?php _e('State', 'geo-ranking-tables'); ?></th>
                                    <th style="width: 80px; text-align: right;"><?php _e('Clicks', 'geo-ranking-tables'); ?></th>
                                    <th style="width: 80px; text-align: right;"><?php _e('Sessions', 'geo-ranking-tables'); ?></th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach (array_slice($by_state, 0, 10) as $state) : ?>
                                    <tr>
                                        <td><?php echo esc_html($state['user_state']); ?></td>
                                        <td style="text-align: right;"><strong><?php echo number_format($state['total_clicks']); ?></strong></td>
                                        <td style="text-align: right;"><?php echo number_format($state['unique_sessions']); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php endif; ?>
                </div>

                <!-- Device Breakdown -->
                <div style="background: #fff; padding: 20px; border: 1px solid #ccd0d4; border-radius: 4px;">
                    <h2 style="margin-top: 0;"><?php _e('Clicks by Device', 'geo-ranking-tables'); ?></h2>
                    <?php if (empty($by_device)) : ?>
                        <p><?php _e('No device data available for this period.', 'geo-ranking-tables'); ?></p>
                    <?php else : ?>
                        <table class="wp-list-table widefat fixed striped">
                            <thead>
                                <tr>
                                    <th><?php _e('Device', 'geo-ranking-tables'); ?></th>
                                    <th style="width: 100px; text-align: right;"><?php _e('Clicks', 'geo-ranking-tables'); ?></th>
                                    <th style="width: 80px; text-align: right;"><?php _e('%', 'geo-ranking-tables'); ?></th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($by_device as $device) : ?>
                                    <tr>
                                        <td><?php echo esc_html(ucfirst($device['device_type'])); ?></td>
                                        <td style="text-align: right;"><strong><?php echo number_format($device['total_clicks']); ?></strong></td>
                                        <td style="text-align: right;"><?php echo esc_html($device['percentage']); ?>%</td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php endif; ?>
                </div>

                <!-- Clicks by Rank Position -->
                <div style="background: #fff; padding: 20px; border: 1px solid #ccd0d4; border-radius: 4px;">
                    <h2 style="margin-top: 0;"><?php _e('Clicks by Rank Position', 'geo-ranking-tables'); ?></h2>
                    <?php if (empty($by_rank)) : ?>
                        <p><?php _e('No rank data available for this period.', 'geo-ranking-tables'); ?></p>
                    <?php else : ?>
                        <table class="wp-list-table widefat fixed striped">
                            <thead>
                                <tr>
                                    <th><?php _e('Position', 'geo-ranking-tables'); ?></th>
                                    <th style="width: 80px; text-align: right;"><?php _e('Total', 'geo-ranking-tables'); ?></th>
                                    <th style="width: 80px; text-align: right;"><?php _e('Visit', 'geo-ranking-tables'); ?></th>
                                    <th style="width: 80px; text-align: right;"><?php _e('Review', 'geo-ranking-tables'); ?></th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($by_rank as $rank) : ?>
                                    <tr>
                                        <td>#<?php echo esc_html($rank['rank_position']); ?></td>
                                        <td style="text-align: right;"><strong><?php echo number_format($rank['total_clicks']); ?></strong></td>
                                        <td style="text-align: right;"><?php echo number_format($rank['visit_clicks']); ?></td>
                                        <td style="text-align: right;"><?php echo number_format($rank['review_clicks']); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Export & Data Management -->
            <div style="background: #fff; padding: 20px; margin-top: 20px; border: 1px solid #ccd0d4; border-radius: 4px;">
                <h2 style="margin-top: 0;"><?php _e('Data Management', 'geo-ranking-tables'); ?></h2>
                <div style="display: flex; gap: 20px; flex-wrap: wrap; align-items: flex-end;">
                    <!-- CSV Export -->
                    <div>
                        <form method="post" action="<?php echo admin_url('admin-ajax.php'); ?>" target="_blank">
                            <input type="hidden" name="action" value="grt_export_clicks">
                            <?php wp_nonce_field('grt_export_clicks', 'nonce'); ?>
                            <input type="hidden" name="start_date" value="<?php echo esc_attr($start_date); ?>">
                            <input type="hidden" name="end_date" value="<?php echo esc_attr($end_date); ?>">
                            <button type="submit" class="button button-secondary">
                                <?php _e('Export CSV', 'geo-ranking-tables'); ?>
                            </button>
                        </form>
                    </div>

                    <!-- Data Purge -->
                    <div style="border-left: 1px solid #ccd0d4; padding-left: 20px;">
                        <p style="margin: 0 0 10px; color: #646970;">
                            <?php printf(__('Total records in database: %s', 'geo-ranking-tables'), '<strong>' . number_format($total_records) . '</strong>'); ?>
                        </p>
                        <form method="post" id="grt-purge-form" style="display: flex; gap: 10px; align-items: center;">
                            <?php wp_nonce_field('grt_purge_click_data', 'purge_nonce'); ?>
                            <label>
                                <?php _e('Delete records older than', 'geo-ranking-tables'); ?>
                                <input type="number" name="purge_days" id="grt-purge-days" value="90" min="1" max="365" style="width: 70px; margin: 0 5px;">
                                <?php _e('days', 'geo-ranking-tables'); ?>
                            </label>
                            <button type="submit" class="button button-secondary" id="grt-purge-btn">
                                <?php _e('Purge Old Data', 'geo-ranking-tables'); ?>
                            </button>
                        </form>
                    </div>
                </div>
            </div>
        </div>

        <script>
        jQuery(document).ready(function($) {
            $('#grt-purge-form').on('submit', function(e) {
                e.preventDefault();
                var days = $('#grt-purge-days').val();
                if (!confirm('<?php echo esc_js(__('Are you sure you want to delete all click records older than', 'geo-ranking-tables')); ?> ' + days + ' <?php echo esc_js(__('days? This cannot be undone.', 'geo-ranking-tables')); ?>')) {
                    return;
                }

                var $btn = $('#grt-purge-btn');
                $btn.prop('disabled', true).text('<?php echo esc_js(__('Purging...', 'geo-ranking-tables')); ?>');

                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'grt_purge_click_data',
                        nonce: $('#purge_nonce').val(),
                        days: days
                    },
                    success: function(response) {
                        if (response.success) {
                            alert(response.data.message);
                            location.reload();
                        } else {
                            alert('Error: ' + (response.data || 'Unknown error'));
                        }
                    },
                    error: function() {
                        alert('<?php echo esc_js(__('An error occurred while purging data.', 'geo-ranking-tables')); ?>');
                    },
                    complete: function() {
                        $btn.prop('disabled', false).text('<?php echo esc_js(__('Purge Old Data', 'geo-ranking-tables')); ?>');
                    }
                });
            });
        });
        </script>
        <?php
    }

    /**
     * AJAX handler for purging old click data
     */
    public function ajax_purge_click_data() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'grt_purge_click_data')) {
            wp_send_json_error(__('Security check failed', 'geo-ranking-tables'));
        }

        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('Insufficient permissions', 'geo-ranking-tables'));
        }

        $days = isset($_POST['days']) ? intval($_POST['days']) : 90;
        if ($days < 1 || $days > 365) {
            wp_send_json_error(__('Invalid number of days. Must be between 1 and 365.', 'geo-ranking-tables'));
        }

        $reports = new GRT_ClickReports();
        $deleted = $reports->purge_old_data($days);

        wp_send_json_success(array(
            'message' => sprintf(
                __('Successfully deleted %d click records older than %d days.', 'geo-ranking-tables'),
                $deleted,
                $days
            ),
            'deleted' => $deleted
        ));
    }

    /**
     * AJAX handler for exporting clicks to CSV
     */
    public function ajax_export_clicks() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'grt_export_clicks')) {
            wp_die(__('Security check failed', 'geo-ranking-tables'));
        }

        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_die(__('Insufficient permissions', 'geo-ranking-tables'));
        }

        $start_date = isset($_POST['start_date']) ? sanitize_text_field($_POST['start_date']) : date('Y-m-d', strtotime('-30 days'));
        $end_date = isset($_POST['end_date']) ? sanitize_text_field($_POST['end_date']) : date('Y-m-d');

        // Validate dates
        if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $start_date)) {
            $start_date = date('Y-m-d', strtotime('-30 days'));
        }
        if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $end_date)) {
            $end_date = date('Y-m-d');
        }

        $reports = new GRT_ClickReports();
        $data = $reports->export_csv($start_date, $end_date);

        // Set headers for CSV download
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename=grt-clicks-' . $start_date . '-to-' . $end_date . '.csv');
        header('Pragma: no-cache');
        header('Expires: 0');

        $output = fopen('php://output', 'w');

        // Add UTF-8 BOM for Excel compatibility
        fprintf($output, chr(0xEF).chr(0xBB).chr(0xBF));

        // Header row
        fputcsv($output, array(
            __('Date/Time', 'geo-ranking-tables'),
            __('Site Name', 'geo-ranking-tables'),
            __('Click Type', 'geo-ranking-tables'),
            __('Rank Position', 'geo-ranking-tables'),
            __('State', 'geo-ranking-tables'),
            __('Device', 'geo-ranking-tables'),
            __('Theme', 'geo-ranking-tables'),
            __('Page URL', 'geo-ranking-tables')
        ));

        // Data rows
        foreach ($data as $row) {
            fputcsv($output, array(
                $row['created_at'],
                $row['site_name'] ?: 'Unknown',
                $row['click_type'],
                $row['rank_position'] ?: '-',
                $row['user_state'] ?: '-',
                $row['device_type'],
                $row['theme'],
                $row['referrer_url'] ?: '-'
            ));
        }

        fclose($output);
        exit;
    }
}