<?php
/**
 * Importer Class
 * Handles importing site blocks and managed tables from JSON
 */

if (!defined('ABSPATH')) {
    exit;
}

class GRT_Importer {

    private $import_data = null;
    private $id_mapping = array(); // old_id => new_id
    private $duplicate_handling = 'skip';
    private $results = array(
        'sites_created' => 0,
        'sites_updated' => 0,
        'sites_skipped' => 0,
        'tables_created' => 0,
        'tables_updated' => 0,
        'tables_skipped' => 0,
        'images_imported' => 0,
        'images_failed' => 0,
        'errors' => array(),
        'warnings' => array(),
    );

    /**
     * Parse JSON string and validate structure
     */
    public function parse_json($json_string) {
        $data = json_decode($json_string, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            return new WP_Error('invalid_json', __('Invalid JSON format: ', 'geo-ranking-tables') . json_last_error_msg());
        }

        // Validate required fields
        if (!isset($data['plugin_version'])) {
            return new WP_Error('missing_version', __('Export file is missing plugin version information.', 'geo-ranking-tables'));
        }

        if (!isset($data['site_blocks']) && !isset($data['managed_tables'])) {
            return new WP_Error('no_data', __('Export file contains no site blocks or managed tables.', 'geo-ranking-tables'));
        }

        // Initialize arrays if not set
        if (!isset($data['site_blocks'])) {
            $data['site_blocks'] = array();
        }
        if (!isset($data['managed_tables'])) {
            $data['managed_tables'] = array();
        }

        $this->import_data = $data;
        return $data;
    }

    /**
     * Generate preview of what will be imported
     */
    public function generate_preview($data) {
        $preview = array(
            'export_info' => array(
                'plugin_version' => isset($data['plugin_version']) ? $data['plugin_version'] : 'Unknown',
                'export_date' => isset($data['export_date']) ? $data['export_date'] : 'Unknown',
                'source_site' => isset($data['site_url']) ? $data['site_url'] : 'Unknown',
                'export_type' => isset($data['export_type']) ? $data['export_type'] : 'Unknown',
            ),
            'site_blocks' => array(
                'total' => count($data['site_blocks']),
                'new' => 0,
                'duplicates' => 0,
                'items' => array(),
            ),
            'managed_tables' => array(
                'total' => count($data['managed_tables']),
                'new' => 0,
                'duplicates' => 0,
                'items' => array(),
            ),
        );

        // Check site blocks for duplicates
        foreach ($data['site_blocks'] as $site) {
            $title = isset($site['post_data']['post_title']) ? $site['post_data']['post_title'] : '';
            $slug = isset($site['post_data']['post_name']) ? $site['post_data']['post_name'] : '';

            $existing = $this->check_duplicate($slug, $title, 'grt_site_block');
            $is_duplicate = $existing !== false;

            if ($is_duplicate) {
                $preview['site_blocks']['duplicates']++;
            } else {
                $preview['site_blocks']['new']++;
            }

            $preview['site_blocks']['items'][] = array(
                'export_id' => isset($site['export_id']) ? $site['export_id'] : 0,
                'title' => $title,
                'slug' => $slug,
                'is_duplicate' => $is_duplicate,
                'existing_id' => $existing,
                'has_image' => !empty($site['featured_image']['url']),
                'has_badge' => !empty($site['certified_badge']['url']),
            );
        }

        // Check managed tables for duplicates
        foreach ($data['managed_tables'] as $table) {
            $title = isset($table['post_data']['post_title']) ? $table['post_data']['post_title'] : '';
            $slug = isset($table['post_data']['post_name']) ? $table['post_data']['post_name'] : '';

            $existing = $this->check_duplicate($slug, $title, 'grt_managed_table');
            $is_duplicate = $existing !== false;

            if ($is_duplicate) {
                $preview['managed_tables']['duplicates']++;
            } else {
                $preview['managed_tables']['new']++;
            }

            $sites_count = 0;
            if (isset($table['meta']['_grt_managed_sites']) && is_array($table['meta']['_grt_managed_sites'])) {
                $sites_count = count($table['meta']['_grt_managed_sites']);
            }

            $preview['managed_tables']['items'][] = array(
                'export_id' => isset($table['export_id']) ? $table['export_id'] : 0,
                'title' => $title,
                'slug' => $slug,
                'is_duplicate' => $is_duplicate,
                'existing_id' => $existing,
                'sites_count' => $sites_count,
            );
        }

        return $preview;
    }

    /**
     * Execute the import
     */
    public function execute_import($data, $options = array()) {
        $this->duplicate_handling = isset($options['duplicate_handling']) ? $options['duplicate_handling'] : 'skip';
        $this->id_mapping = array();
        $this->results = array(
            'sites_created' => 0,
            'sites_updated' => 0,
            'sites_skipped' => 0,
            'tables_created' => 0,
            'tables_updated' => 0,
            'tables_skipped' => 0,
            'images_imported' => 0,
            'images_failed' => 0,
            'errors' => array(),
            'warnings' => array(),
        );

        // Import site blocks first to build ID mapping
        if (!empty($data['site_blocks'])) {
            foreach ($data['site_blocks'] as $site) {
                $this->import_site_block($site);
            }
        }

        // Import managed tables using the ID mapping
        if (!empty($data['managed_tables'])) {
            foreach ($data['managed_tables'] as $table) {
                $this->import_managed_table($table);
            }
        }

        return $this->results;
    }

    /**
     * Import a single site block
     */
    private function import_site_block($site_data) {
        $title = isset($site_data['post_data']['post_title']) ? sanitize_text_field($site_data['post_data']['post_title']) : '';
        $slug = isset($site_data['post_data']['post_name']) ? sanitize_title($site_data['post_data']['post_name']) : '';
        $export_id = isset($site_data['export_id']) ? intval($site_data['export_id']) : 0;

        if (empty($title)) {
            $this->results['errors'][] = sprintf(__('Site block with export ID %d has no title.', 'geo-ranking-tables'), $export_id);
            return false;
        }

        // Check for duplicate
        $existing_id = $this->check_duplicate($slug, $title, 'grt_site_block');

        if ($existing_id !== false) {
            switch ($this->duplicate_handling) {
                case 'skip':
                    $this->results['sites_skipped']++;
                    $this->id_mapping[$export_id] = $existing_id;
                    return $existing_id;

                case 'overwrite':
                    $post_id = $this->update_site_block($existing_id, $site_data);
                    if ($post_id) {
                        $this->results['sites_updated']++;
                        $this->id_mapping[$export_id] = $post_id;
                    }
                    return $post_id;

                case 'create_new':
                    // Will create with modified slug below
                    $slug = $this->get_unique_slug($slug, 'grt_site_block');
                    break;
            }
        }

        // Create new site block
        $post_id = $this->create_site_block($site_data, $slug);
        if ($post_id) {
            $this->results['sites_created']++;
            $this->id_mapping[$export_id] = $post_id;
        }

        return $post_id;
    }

    /**
     * Create a new site block
     */
    private function create_site_block($site_data, $slug = '') {
        $post_data = isset($site_data['post_data']) ? $site_data['post_data'] : array();
        $meta = isset($site_data['meta']) ? $site_data['meta'] : array();

        $post_arr = array(
            'post_type' => 'grt_site_block',
            'post_title' => isset($post_data['post_title']) ? sanitize_text_field($post_data['post_title']) : '',
            'post_name' => !empty($slug) ? $slug : (isset($post_data['post_name']) ? sanitize_title($post_data['post_name']) : ''),
            'post_content' => isset($post_data['post_content']) ? wp_kses_post($post_data['post_content']) : '',
            'post_status' => 'publish',
        );

        $post_id = wp_insert_post($post_arr, true);

        if (is_wp_error($post_id)) {
            $this->results['errors'][] = sprintf(
                __('Failed to create site block "%s": %s', 'geo-ranking-tables'),
                $post_arr['post_title'],
                $post_id->get_error_message()
            );
            return false;
        }

        // Save meta fields
        $this->save_site_block_meta($post_id, $meta);

        // Handle featured image
        if (!empty($site_data['featured_image']['url'])) {
            $this->handle_featured_image($site_data['featured_image'], $post_id);
        }

        // Handle certified badge
        if (!empty($site_data['certified_badge']['url'])) {
            $this->handle_certified_badge($site_data['certified_badge'], $post_id);
        }

        return $post_id;
    }

    /**
     * Update an existing site block
     */
    private function update_site_block($post_id, $site_data) {
        $post_data = isset($site_data['post_data']) ? $site_data['post_data'] : array();
        $meta = isset($site_data['meta']) ? $site_data['meta'] : array();

        $post_arr = array(
            'ID' => $post_id,
            'post_title' => isset($post_data['post_title']) ? sanitize_text_field($post_data['post_title']) : '',
            'post_content' => isset($post_data['post_content']) ? wp_kses_post($post_data['post_content']) : '',
        );

        $result = wp_update_post($post_arr, true);

        if (is_wp_error($result)) {
            $this->results['errors'][] = sprintf(
                __('Failed to update site block "%s": %s', 'geo-ranking-tables'),
                $post_arr['post_title'],
                $result->get_error_message()
            );
            return false;
        }

        // Save meta fields
        $this->save_site_block_meta($post_id, $meta);

        // Handle featured image
        if (!empty($site_data['featured_image']['url'])) {
            $this->handle_featured_image($site_data['featured_image'], $post_id);
        }

        // Handle certified badge
        if (!empty($site_data['certified_badge']['url'])) {
            $this->handle_certified_badge($site_data['certified_badge'], $post_id);
        }

        return $post_id;
    }

    /**
     * Save site block meta fields
     */
    private function save_site_block_meta($post_id, $meta) {
        $meta_fields = array(
            '_grt_site_name',
            '_grt_site_description',
            '_grt_star_rating',
            '_grt_affiliate_link',
            '_grt_bonus_info',
            '_grt_license_info',
            '_grt_states_accepted',
            // New fields
            '_grt_pros',
            '_grt_cons',
            '_grt_payout_speed',
            '_grt_compatibility',
            '_grt_payout_percentage',
            '_grt_review_link',
            '_grt_disclaimers',
        );

        foreach ($meta_fields as $field) {
            if (isset($meta[$field])) {
                $value = $meta[$field];

                // Sanitize based on field type
                if ($field === '_grt_states_accepted') {
                    if (!is_array($value)) {
                        $value = array();
                    }
                    $value = array_map('sanitize_text_field', $value);
                } elseif ($field === '_grt_star_rating') {
                    $value = floatval($value);
                    $value = max(0, min(5, $value));
                } elseif ($field === '_grt_affiliate_link' || $field === '_grt_review_link') {
                    $value = esc_url_raw($value);
                } elseif (in_array($field, array('_grt_site_description', '_grt_bonus_info'))) {
                    $value = sanitize_textarea_field($value);
                } elseif (in_array($field, array('_grt_pros', '_grt_cons'))) {
                    // Pros and cons are arrays
                    if (!is_array($value)) {
                        $value = array();
                    }
                    $value = array_map('sanitize_text_field', $value);
                    $value = array_filter($value); // Remove empty entries
                } elseif ($field === '_grt_compatibility') {
                    // Compatibility is an array with allowed values
                    if (!is_array($value)) {
                        $value = array();
                    }
                    $allowed = array('ios', 'android', 'windows', 'mac');
                    $value = array_intersect($value, $allowed);
                } elseif ($field === '_grt_disclaimers') {
                    // Disclaimers are arrays of objects with id, text, and states
                    if (!is_array($value)) {
                        $value = array();
                    }
                    $sanitized_disclaimers = array();
                    $valid_states = array(
                        'AL', 'AK', 'AZ', 'AR', 'CA', 'CO', 'CT', 'DE', 'FL', 'GA',
                        'HI', 'ID', 'IL', 'IN', 'IA', 'KS', 'KY', 'LA', 'ME', 'MD',
                        'MA', 'MI', 'MN', 'MS', 'MO', 'MT', 'NE', 'NV', 'NH', 'NJ',
                        'NM', 'NY', 'NC', 'ND', 'OH', 'OK', 'OR', 'PA', 'RI', 'SC',
                        'SD', 'TN', 'TX', 'UT', 'VT', 'VA', 'WA', 'WV', 'WI', 'WY', 'DC',
                        'AB', 'BC', 'MB', 'NB', 'NL', 'NS', 'NT', 'NU', 'ON', 'PE', 'QC', 'SK', 'YT'
                    );
                    foreach ($value as $disc) {
                        if (empty($disc['text']) || empty($disc['states'])) {
                            continue;
                        }
                        $sanitized_disclaimers[] = array(
                            'id' => !empty($disc['id']) ? sanitize_text_field($disc['id']) : 'disc_' . uniqid(),
                            'text' => wp_kses_post($disc['text']),
                            'states' => array_values(array_intersect(
                                array_map('sanitize_text_field', (array)$disc['states']),
                                $valid_states
                            )),
                        );
                    }
                    $value = $sanitized_disclaimers;
                } else {
                    $value = sanitize_text_field($value);
                }

                update_post_meta($post_id, $field, $value);
            }
        }
    }

    /**
     * Import a single managed table
     */
    private function import_managed_table($table_data) {
        $title = isset($table_data['post_data']['post_title']) ? sanitize_text_field($table_data['post_data']['post_title']) : '';
        $slug = isset($table_data['post_data']['post_name']) ? sanitize_title($table_data['post_data']['post_name']) : '';
        $export_id = isset($table_data['export_id']) ? intval($table_data['export_id']) : 0;

        if (empty($title)) {
            $this->results['errors'][] = sprintf(__('Managed table with export ID %d has no title.', 'geo-ranking-tables'), $export_id);
            return false;
        }

        // Check for duplicate
        $existing_id = $this->check_duplicate($slug, $title, 'grt_managed_table');

        if ($existing_id !== false) {
            switch ($this->duplicate_handling) {
                case 'skip':
                    $this->results['tables_skipped']++;
                    return $existing_id;

                case 'overwrite':
                    $post_id = $this->update_managed_table($existing_id, $table_data);
                    if ($post_id) {
                        $this->results['tables_updated']++;
                    }
                    return $post_id;

                case 'create_new':
                    $slug = $this->get_unique_slug($slug, 'grt_managed_table');
                    break;
            }
        }

        // Create new managed table
        $post_id = $this->create_managed_table($table_data, $slug);
        if ($post_id) {
            $this->results['tables_created']++;
        }

        return $post_id;
    }

    /**
     * Create a new managed table
     */
    private function create_managed_table($table_data, $slug = '') {
        $post_data = isset($table_data['post_data']) ? $table_data['post_data'] : array();
        $meta = isset($table_data['meta']) ? $table_data['meta'] : array();

        $post_arr = array(
            'post_type' => 'grt_managed_table',
            'post_title' => isset($post_data['post_title']) ? sanitize_text_field($post_data['post_title']) : '',
            'post_name' => !empty($slug) ? $slug : (isset($post_data['post_name']) ? sanitize_title($post_data['post_name']) : ''),
            'post_status' => 'publish',
        );

        $post_id = wp_insert_post($post_arr, true);

        if (is_wp_error($post_id)) {
            $this->results['errors'][] = sprintf(
                __('Failed to create managed table "%s": %s', 'geo-ranking-tables'),
                $post_arr['post_title'],
                $post_id->get_error_message()
            );
            return false;
        }

        // Save meta fields with ID remapping
        $this->save_managed_table_meta($post_id, $meta);

        return $post_id;
    }

    /**
     * Update an existing managed table
     */
    private function update_managed_table($post_id, $table_data) {
        $post_data = isset($table_data['post_data']) ? $table_data['post_data'] : array();
        $meta = isset($table_data['meta']) ? $table_data['meta'] : array();

        $post_arr = array(
            'ID' => $post_id,
            'post_title' => isset($post_data['post_title']) ? sanitize_text_field($post_data['post_title']) : '',
        );

        $result = wp_update_post($post_arr, true);

        if (is_wp_error($result)) {
            $this->results['errors'][] = sprintf(
                __('Failed to update managed table "%s": %s', 'geo-ranking-tables'),
                $post_arr['post_title'],
                $result->get_error_message()
            );
            return false;
        }

        // Save meta fields with ID remapping
        $this->save_managed_table_meta($post_id, $meta);

        return $post_id;
    }

    /**
     * Save managed table meta fields with ID remapping
     */
    private function save_managed_table_meta($post_id, $meta) {
        // Remap site IDs
        if (isset($meta['_grt_managed_sites']) && is_array($meta['_grt_managed_sites'])) {
            $remapped_sites = array();
            foreach ($meta['_grt_managed_sites'] as $old_id) {
                if (isset($this->id_mapping[$old_id])) {
                    $remapped_sites[] = $this->id_mapping[$old_id];
                } else {
                    $this->results['warnings'][] = sprintf(
                        __('Site block with original ID %d not found in import. Skipping reference in table.', 'geo-ranking-tables'),
                        $old_id
                    );
                }
            }
            update_post_meta($post_id, '_grt_managed_sites', $remapped_sites);
        }

        // Save other meta fields
        $simple_meta = array(
            '_grt_managed_max_display' => 'intval',
            '_grt_managed_table_name' => 'sanitize_text_field',
            '_grt_managed_show_ratings' => 'sanitize_text_field',
            '_grt_managed_show_bonuses' => 'sanitize_text_field',
            '_grt_managed_show_licenses' => 'sanitize_text_field',
            '_grt_managed_show_debug' => 'sanitize_text_field',
            '_grt_managed_slug' => 'sanitize_title',
            '_grt_managed_theme' => 'sanitize_text_field',
        );

        foreach ($simple_meta as $field => $sanitize_func) {
            if (isset($meta[$field])) {
                $value = call_user_func($sanitize_func, $meta[$field]);
                update_post_meta($post_id, $field, $value);
            }
        }
    }

    /**
     * Check for duplicate post by slug or title
     */
    private function check_duplicate($slug, $title, $post_type) {
        global $wpdb;

        // First try to find by slug
        if (!empty($slug)) {
            $existing = $wpdb->get_var($wpdb->prepare(
                "SELECT ID FROM {$wpdb->posts} WHERE post_name = %s AND post_type = %s AND post_status = 'publish' LIMIT 1",
                $slug,
                $post_type
            ));

            if ($existing) {
                return intval($existing);
            }
        }

        // Then try by title
        if (!empty($title)) {
            $existing = $wpdb->get_var($wpdb->prepare(
                "SELECT ID FROM {$wpdb->posts} WHERE post_title = %s AND post_type = %s AND post_status = 'publish' LIMIT 1",
                $title,
                $post_type
            ));

            if ($existing) {
                return intval($existing);
            }
        }

        return false;
    }

    /**
     * Get unique slug for new post
     */
    private function get_unique_slug($slug, $post_type) {
        global $wpdb;

        $original_slug = $slug;
        $counter = 1;

        while (true) {
            $existing = $wpdb->get_var($wpdb->prepare(
                "SELECT ID FROM {$wpdb->posts} WHERE post_name = %s AND post_type = %s LIMIT 1",
                $slug,
                $post_type
            ));

            if (!$existing) {
                return $slug;
            }

            $counter++;
            $slug = $original_slug . '-' . $counter;
        }
    }

    /**
     * Handle featured image import
     */
    private function handle_featured_image($image_data, $post_id) {
        if (empty($image_data['url'])) {
            return false;
        }

        $url = esc_url_raw($image_data['url']);
        $filename = isset($image_data['filename']) ? sanitize_file_name($image_data['filename']) : basename($url);
        $alt = isset($image_data['alt']) ? sanitize_text_field($image_data['alt']) : '';
        $title = isset($image_data['title']) ? sanitize_text_field($image_data['title']) : '';

        // Check if image already exists in media library by URL
        $existing_id = $this->find_attachment_by_url($url);
        if ($existing_id) {
            set_post_thumbnail($post_id, $existing_id);
            return $existing_id;
        }

        // Download and sideload the image
        require_once(ABSPATH . 'wp-admin/includes/file.php');
        require_once(ABSPATH . 'wp-admin/includes/media.php');
        require_once(ABSPATH . 'wp-admin/includes/image.php');

        // Download file to temp location
        $tmp_file = download_url($url, 30);

        if (is_wp_error($tmp_file)) {
            $this->results['images_failed']++;
            $this->results['warnings'][] = sprintf(
                __('Failed to download image for "%s": %s', 'geo-ranking-tables'),
                get_the_title($post_id),
                $tmp_file->get_error_message()
            );
            return false;
        }

        $file_array = array(
            'name' => $filename,
            'tmp_name' => $tmp_file,
        );

        // Sideload the file
        $attachment_id = media_handle_sideload($file_array, $post_id, $title);

        // Clean up temp file
        if (file_exists($tmp_file)) {
            @unlink($tmp_file);
        }

        if (is_wp_error($attachment_id)) {
            $this->results['images_failed']++;
            $this->results['warnings'][] = sprintf(
                __('Failed to import image for "%s": %s', 'geo-ranking-tables'),
                get_the_title($post_id),
                $attachment_id->get_error_message()
            );
            return false;
        }

        // Set alt text
        if (!empty($alt)) {
            update_post_meta($attachment_id, '_wp_attachment_image_alt', $alt);
        }

        // Set as featured image
        set_post_thumbnail($post_id, $attachment_id);

        $this->results['images_imported']++;
        return $attachment_id;
    }

    /**
     * Handle certified badge import
     */
    private function handle_certified_badge($image_data, $post_id) {
        if (empty($image_data['url'])) {
            return false;
        }

        $url = esc_url_raw($image_data['url']);
        $filename = isset($image_data['filename']) ? sanitize_file_name($image_data['filename']) : basename($url);
        $alt = isset($image_data['alt']) ? sanitize_text_field($image_data['alt']) : '';
        $title = isset($image_data['title']) ? sanitize_text_field($image_data['title']) : '';

        // Check if image already exists in media library by URL
        $existing_id = $this->find_attachment_by_url($url);
        if ($existing_id) {
            update_post_meta($post_id, '_grt_certified_badge', $existing_id);
            return $existing_id;
        }

        // Download and sideload the image
        require_once(ABSPATH . 'wp-admin/includes/file.php');
        require_once(ABSPATH . 'wp-admin/includes/media.php');
        require_once(ABSPATH . 'wp-admin/includes/image.php');

        // Download file to temp location
        $tmp_file = download_url($url, 30);

        if (is_wp_error($tmp_file)) {
            $this->results['images_failed']++;
            $this->results['warnings'][] = sprintf(
                __('Failed to download certified badge for "%s": %s', 'geo-ranking-tables'),
                get_the_title($post_id),
                $tmp_file->get_error_message()
            );
            return false;
        }

        $file_array = array(
            'name' => $filename,
            'tmp_name' => $tmp_file,
        );

        // Sideload the file
        $attachment_id = media_handle_sideload($file_array, $post_id, $title);

        // Clean up temp file
        if (file_exists($tmp_file)) {
            @unlink($tmp_file);
        }

        if (is_wp_error($attachment_id)) {
            $this->results['images_failed']++;
            $this->results['warnings'][] = sprintf(
                __('Failed to import certified badge for "%s": %s', 'geo-ranking-tables'),
                get_the_title($post_id),
                $attachment_id->get_error_message()
            );
            return false;
        }

        // Set alt text
        if (!empty($alt)) {
            update_post_meta($attachment_id, '_wp_attachment_image_alt', $alt);
        }

        // Save as certified badge meta
        update_post_meta($post_id, '_grt_certified_badge', $attachment_id);

        $this->results['images_imported']++;
        return $attachment_id;
    }

    /**
     * Find existing attachment by URL
     */
    private function find_attachment_by_url($url) {
        global $wpdb;

        $attachment_id = $wpdb->get_var($wpdb->prepare(
            "SELECT ID FROM {$wpdb->posts} WHERE guid = %s AND post_type = 'attachment' LIMIT 1",
            $url
        ));

        return $attachment_id ? intval($attachment_id) : false;
    }
}
