<?php
/**
 * Site Blocks Management Class
 * Handles the custom post type and meta fields for gambling sites
 */

if (!defined('ABSPATH')) {
    exit;
}

class GRT_SiteBlocks {
    
    public function __construct() {
        add_action('add_meta_boxes', array($this, 'add_meta_boxes'));
        add_action('save_post_grt_site_block', array($this, 'save_meta_boxes'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        add_filter('post_updated_messages', array($this, 'updated_messages'));
        add_filter('manage_grt_site_block_posts_columns', array($this, 'add_custom_columns'));
        add_action('manage_grt_site_block_posts_custom_column', array($this, 'custom_column_content'), 10, 2);
    }
    
    /**
     * Add meta boxes for site data
     */
    public function add_meta_boxes() {
        add_meta_box(
            'grt_site_details',
            __('Site Details', 'geo-ranking-tables'),
            array($this, 'site_details_callback'),
            'grt_site_block',
            'normal',
            'high'
        );
        
        add_meta_box(
            'grt_geo_settings',
            __('Geographic Settings', 'geo-ranking-tables'),
            array($this, 'geo_settings_callback'),
            'grt_site_block',
            'normal',
            'high'
        );

        add_meta_box(
            'grt_additional_info',
            __('Additional Site Information', 'geo-ranking-tables'),
            array($this, 'additional_info_callback'),
            'grt_site_block',
            'normal',
            'default'
        );

        add_meta_box(
            'grt_disclaimers',
            __('Location-Based Disclaimers', 'geo-ranking-tables'),
            array($this, 'disclaimers_callback'),
            'grt_site_block',
            'normal',
            'default'
        );
    }
    
    /**
     * Site details meta box callback
     */
    public function site_details_callback($post) {
        wp_nonce_field('grt_save_meta_boxes', 'grt_meta_nonce');
        
        $site_name = get_post_meta($post->ID, '_grt_site_name', true);
        $site_description = get_post_meta($post->ID, '_grt_site_description', true);
        $star_rating = get_post_meta($post->ID, '_grt_star_rating', true);
        $affiliate_link = get_post_meta($post->ID, '_grt_affiliate_link', true);
        $bonus_info = get_post_meta($post->ID, '_grt_bonus_info', true);
        $license_info = get_post_meta($post->ID, '_grt_license_info', true);
        
        ?>
        <table class="form-table">
            <tr>
                <th scope="row">
                    <label for="grt_site_name"><?php _e('Site Name', 'geo-ranking-tables'); ?></label>
                </th>
                <td>
                    <input type="text" id="grt_site_name" name="grt_site_name" value="<?php echo esc_attr($site_name); ?>" class="regular-text" />
                    <p class="description"><?php _e('The name of the gambling site', 'geo-ranking-tables'); ?></p>
                </td>
            </tr>
            <tr>
                <th scope="row">
                    <label for="grt_site_description"><?php _e('Site Description', 'geo-ranking-tables'); ?></label>
                </th>
                <td>
                    <textarea id="grt_site_description" name="grt_site_description" rows="4" class="large-text"><?php echo esc_textarea($site_description); ?></textarea>
                    <p class="description"><?php _e('Brief description of the gambling site', 'geo-ranking-tables'); ?></p>
                </td>
            </tr>
            <tr>
                <th scope="row">
                    <label for="grt_star_rating"><?php _e('Star Rating', 'geo-ranking-tables'); ?></label>
                </th>
                <td>
                    <select id="grt_star_rating" name="grt_star_rating">
                        <?php for ($i = 0; $i <= 50; $i += 5): ?>
                            <option value="<?php echo $i/10; ?>" <?php selected($star_rating, $i/10); ?>><?php echo $i/10; ?> Stars</option>
                        <?php endfor; ?>
                    </select>
                    <p class="description"><?php _e('Rating out of 5 stars', 'geo-ranking-tables'); ?></p>
                </td>
            </tr>
            <tr>
                <th scope="row">
                    <label for="grt_affiliate_link"><?php _e('Affiliate Link', 'geo-ranking-tables'); ?></label>
                </th>
                <td>
                    <input type="url" id="grt_affiliate_link" name="grt_affiliate_link" value="<?php echo esc_url($affiliate_link); ?>" class="regular-text" />
                    <p class="description"><?php _e('Affiliate/referral link for this site', 'geo-ranking-tables'); ?></p>
                </td>
            </tr>
            <tr>
                <th scope="row">
                    <label for="grt_bonus_info"><?php _e('Bonus Information', 'geo-ranking-tables'); ?></label>
                </th>
                <td>
                    <textarea id="grt_bonus_info" name="grt_bonus_info" rows="3" class="large-text"><?php echo esc_textarea($bonus_info); ?></textarea>
                    <p class="description"><?php _e('Welcome bonus or promotional information', 'geo-ranking-tables'); ?></p>
                </td>
            </tr>
            <tr>
                <th scope="row">
                    <label for="grt_license_info"><?php _e('License Information', 'geo-ranking-tables'); ?></label>
                </th>
                <td>
                    <input type="text" id="grt_license_info" name="grt_license_info" value="<?php echo esc_attr($license_info); ?>" class="regular-text" />
                    <p class="description"><?php _e('Licensing authority and license number', 'geo-ranking-tables'); ?></p>
                </td>
            </tr>
        </table>
        <?php
    }
    
    /**
     * Geographic settings meta box callback
     */
    public function geo_settings_callback($post) {
        $states_accepted = get_post_meta($post->ID, '_grt_states_accepted', true);
        if (!is_array($states_accepted)) {
            $states_accepted = array();
        }

        $locations = $this->get_all_locations();

        ?>
        <div class="grt-states-container">
            <h4><?php _e('Locations Where This Site Accepts Players', 'geo-ranking-tables'); ?></h4>
            <p class="description"><?php _e('Select all US states and Canadian provinces where this gambling site legally accepts players', 'geo-ranking-tables'); ?></p>

            <div class="grt-select-all-controls" style="margin: 10px 0;">
                <button type="button" id="grt-select-all-states" class="button"><?php _e('Select All', 'geo-ranking-tables'); ?></button>
                <button type="button" id="grt-deselect-all-states" class="button"><?php _e('Deselect All', 'geo-ranking-tables'); ?></button>
            </div>

            <div class="grt-states-grid" style="display: grid; grid-template-columns: repeat(auto-fill, minmax(200px, 1fr)); gap: 10px; max-height: 300px; overflow-y: auto; border: 1px solid #ddd; padding: 15px;">
                <?php foreach ($locations as $code => $name): ?>
                    <label style="display: flex; align-items: center;">
                        <input type="checkbox" name="grt_states_accepted[]" value="<?php echo esc_attr($code); ?>" <?php checked(in_array($code, $states_accepted)); ?> />
                        <span style="margin-left: 5px;"><?php echo esc_html($name . ' (' . $code . ')'); ?></span>
                    </label>
                <?php endforeach; ?>
            </div>
        </div>
        
        <script>
        jQuery(document).ready(function($) {
            $('#grt-select-all-states').click(function() {
                $('.grt-states-grid input[type="checkbox"]').prop('checked', true);
            });

            $('#grt-deselect-all-states').click(function() {
                $('.grt-states-grid input[type="checkbox"]').prop('checked', false);
            });
        });
        </script>
        <?php
    }

    /**
     * Additional site information meta box callback
     */
    public function additional_info_callback($post) {
        $pros = get_post_meta($post->ID, '_grt_pros', true);
        $cons = get_post_meta($post->ID, '_grt_cons', true);
        $payout_speed = get_post_meta($post->ID, '_grt_payout_speed', true);
        $compatibility = get_post_meta($post->ID, '_grt_compatibility', true);
        $payout_percentage = get_post_meta($post->ID, '_grt_payout_percentage', true);
        $review_link = get_post_meta($post->ID, '_grt_review_link', true);
        $certified_badge = get_post_meta($post->ID, '_grt_certified_badge', true);

        // Convert arrays to newline-separated text for textareas
        $pros_text = is_array($pros) ? implode("\n", $pros) : '';
        $cons_text = is_array($cons) ? implode("\n", $cons) : '';

        if (!is_array($compatibility)) {
            $compatibility = array();
        }

        $badge_url = $certified_badge ? wp_get_attachment_url($certified_badge) : '';
        ?>
        <table class="form-table">
            <tr>
                <th scope="row">
                    <label for="grt_pros"><?php _e('Pros', 'geo-ranking-tables'); ?></label>
                </th>
                <td>
                    <textarea id="grt_pros" name="grt_pros" rows="5" class="large-text"><?php echo esc_textarea($pros_text); ?></textarea>
                    <p class="description"><?php _e('Enter one advantage per line (e.g., "Fast withdrawals", "Great customer support")', 'geo-ranking-tables'); ?></p>
                </td>
            </tr>
            <tr>
                <th scope="row">
                    <label for="grt_cons"><?php _e('Cons', 'geo-ranking-tables'); ?></label>
                </th>
                <td>
                    <textarea id="grt_cons" name="grt_cons" rows="5" class="large-text"><?php echo esc_textarea($cons_text); ?></textarea>
                    <p class="description"><?php _e('Enter one disadvantage per line (e.g., "Limited game selection", "Slow verification")', 'geo-ranking-tables'); ?></p>
                </td>
            </tr>
            <tr>
                <th scope="row">
                    <label for="grt_payout_speed"><?php _e('Payout Speed', 'geo-ranking-tables'); ?></label>
                </th>
                <td>
                    <input type="text" id="grt_payout_speed" name="grt_payout_speed" value="<?php echo esc_attr($payout_speed); ?>" class="regular-text" placeholder="e.g., 2-5 Days" />
                    <p class="description"><?php _e('How long withdrawals typically take', 'geo-ranking-tables'); ?></p>
                </td>
            </tr>
            <tr>
                <th scope="row">
                    <?php _e('Compatibility', 'geo-ranking-tables'); ?>
                </th>
                <td>
                    <fieldset>
                        <label style="margin-right: 15px;">
                            <input type="checkbox" name="grt_compatibility[]" value="ios" <?php checked(in_array('ios', $compatibility)); ?> />
                            <?php _e('iOS', 'geo-ranking-tables'); ?>
                        </label>
                        <label style="margin-right: 15px;">
                            <input type="checkbox" name="grt_compatibility[]" value="android" <?php checked(in_array('android', $compatibility)); ?> />
                            <?php _e('Android', 'geo-ranking-tables'); ?>
                        </label>
                        <label style="margin-right: 15px;">
                            <input type="checkbox" name="grt_compatibility[]" value="windows" <?php checked(in_array('windows', $compatibility)); ?> />
                            <?php _e('Windows', 'geo-ranking-tables'); ?>
                        </label>
                        <label>
                            <input type="checkbox" name="grt_compatibility[]" value="mac" <?php checked(in_array('mac', $compatibility)); ?> />
                            <?php _e('Mac', 'geo-ranking-tables'); ?>
                        </label>
                    </fieldset>
                    <p class="description"><?php _e('Select all platforms this site supports', 'geo-ranking-tables'); ?></p>
                </td>
            </tr>
            <tr>
                <th scope="row">
                    <label for="grt_payout_percentage"><?php _e('Payout Percentage', 'geo-ranking-tables'); ?></label>
                </th>
                <td>
                    <input type="text" id="grt_payout_percentage" name="grt_payout_percentage" value="<?php echo esc_attr($payout_percentage); ?>" class="regular-text" placeholder="e.g., 97.31%" />
                    <p class="description"><?php _e('Average payout/RTP percentage', 'geo-ranking-tables'); ?></p>
                </td>
            </tr>
            <tr>
                <th scope="row">
                    <label for="grt_review_link"><?php _e('Review Link', 'geo-ranking-tables'); ?></label>
                </th>
                <td>
                    <input type="url" id="grt_review_link" name="grt_review_link" value="<?php echo esc_url($review_link); ?>" class="regular-text" placeholder="https://example.com/review" />
                    <p class="description"><?php _e('Full URL to the review page for this site', 'geo-ranking-tables'); ?></p>
                </td>
            </tr>
            <tr>
                <th scope="row">
                    <label for="grt_certified_badge"><?php _e('Certified Badge', 'geo-ranking-tables'); ?></label>
                </th>
                <td>
                    <input type="hidden" id="grt_certified_badge" name="grt_certified_badge" value="<?php echo esc_attr($certified_badge); ?>" />
                    <div id="grt-badge-preview" style="margin-bottom: 10px;">
                        <?php if ($badge_url): ?>
                            <img src="<?php echo esc_url($badge_url); ?>" style="max-width: 150px; max-height: 80px;" />
                        <?php endif; ?>
                    </div>
                    <button type="button" id="grt-upload-badge" class="button"><?php _e('Upload Badge', 'geo-ranking-tables'); ?></button>
                    <button type="button" id="grt-remove-badge" class="button" <?php echo $certified_badge ? '' : 'style="display:none;"'; ?>><?php _e('Remove Badge', 'geo-ranking-tables'); ?></button>
                    <p class="description"><?php _e('Upload a custom certification badge image (e.g., licensed badge, SSL verified)', 'geo-ranking-tables'); ?></p>
                </td>
            </tr>
        </table>
        <?php
    }

    /**
     * Location-based disclaimers meta box callback
     */
    public function disclaimers_callback($post) {
        $disclaimers = get_post_meta($post->ID, '_grt_disclaimers', true);
        if (!is_array($disclaimers)) {
            $disclaimers = array();
        }

        $locations = $this->get_all_locations();
        ?>
        <div id="grt-disclaimers-container">
            <p class="description"><?php _e('Add disclaimer text that will display for users in specific states. Each disclaimer can target multiple states. Only the first matching disclaimer will be shown.', 'geo-ranking-tables'); ?></p>

            <div id="grt-disclaimers-list">
                <?php if (!empty($disclaimers)): ?>
                    <?php foreach ($disclaimers as $index => $disclaimer): ?>
                        <div class="grt-disclaimer-item" data-index="<?php echo esc_attr($index); ?>">
                            <div class="grt-disclaimer-header">
                                <h4><?php _e('Disclaimer', 'geo-ranking-tables'); ?> <span class="disclaimer-number"><?php echo ($index + 1); ?></span></h4>
                                <button type="button" class="button grt-remove-disclaimer"><?php _e('Remove', 'geo-ranking-tables'); ?></button>
                            </div>

                            <div class="grt-disclaimer-content">
                                <div class="grt-disclaimer-text-field">
                                    <label><?php _e('Disclaimer Text', 'geo-ranking-tables'); ?></label>
                                    <textarea name="grt_disclaimers[<?php echo $index; ?>][text]" rows="4" class="large-text"><?php echo esc_textarea($disclaimer['text']); ?></textarea>
                                    <p class="description"><?php _e('HTML allowed: links, bold, italic, etc.', 'geo-ranking-tables'); ?></p>
                                </div>

                                <div class="grt-disclaimer-states-field">
                                    <label><?php _e('Show for States', 'geo-ranking-tables'); ?></label>
                                    <div class="grt-disclaimer-states-controls">
                                        <button type="button" class="button button-small grt-disclaimer-select-all"><?php _e('Select All', 'geo-ranking-tables'); ?></button>
                                        <button type="button" class="button button-small grt-disclaimer-deselect-all"><?php _e('Deselect All', 'geo-ranking-tables'); ?></button>
                                    </div>
                                    <div class="grt-disclaimer-states-grid">
                                        <?php
                                        $selected_states = isset($disclaimer['states']) && is_array($disclaimer['states']) ? $disclaimer['states'] : array();
                                        foreach ($locations as $code => $name):
                                        ?>
                                            <label style="display: flex; align-items: center;">
                                                <input type="checkbox" name="grt_disclaimers[<?php echo $index; ?>][states][]" value="<?php echo esc_attr($code); ?>" <?php checked(in_array($code, $selected_states)); ?> />
                                                <span style="margin-left: 5px;"><?php echo esc_html($name . ' (' . $code . ')'); ?></span>
                                            </label>
                                        <?php endforeach; ?>
                                    </div>
                                </div>
                            </div>

                            <input type="hidden" name="grt_disclaimers[<?php echo $index; ?>][id]" value="<?php echo esc_attr(isset($disclaimer['id']) ? $disclaimer['id'] : 'disc_' . uniqid()); ?>" />
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>

            <button type="button" id="grt-add-disclaimer" class="button button-secondary"><?php _e('Add Disclaimer', 'geo-ranking-tables'); ?></button>
        </div>

        <!-- Template for new disclaimer (hidden) -->
        <script type="text/template" id="grt-disclaimer-template">
            <div class="grt-disclaimer-item" data-index="{{INDEX}}">
                <div class="grt-disclaimer-header">
                    <h4><?php _e('Disclaimer', 'geo-ranking-tables'); ?> <span class="disclaimer-number">{{NUMBER}}</span></h4>
                    <button type="button" class="button grt-remove-disclaimer"><?php _e('Remove', 'geo-ranking-tables'); ?></button>
                </div>

                <div class="grt-disclaimer-content">
                    <div class="grt-disclaimer-text-field">
                        <label><?php _e('Disclaimer Text', 'geo-ranking-tables'); ?></label>
                        <textarea name="grt_disclaimers[{{INDEX}}][text]" rows="4" class="large-text"></textarea>
                        <p class="description"><?php _e('HTML allowed: links, bold, italic, etc.', 'geo-ranking-tables'); ?></p>
                    </div>

                    <div class="grt-disclaimer-states-field">
                        <label><?php _e('Show for States', 'geo-ranking-tables'); ?></label>
                        <div class="grt-disclaimer-states-controls">
                            <button type="button" class="button button-small grt-disclaimer-select-all"><?php _e('Select All', 'geo-ranking-tables'); ?></button>
                            <button type="button" class="button button-small grt-disclaimer-deselect-all"><?php _e('Deselect All', 'geo-ranking-tables'); ?></button>
                        </div>
                        <div class="grt-disclaimer-states-grid">
                            <?php foreach ($locations as $code => $name): ?>
                                <label style="display: flex; align-items: center;">
                                    <input type="checkbox" name="grt_disclaimers[{{INDEX}}][states][]" value="<?php echo esc_attr($code); ?>" />
                                    <span style="margin-left: 5px;"><?php echo esc_html($name . ' (' . $code . ')'); ?></span>
                                </label>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>

                <input type="hidden" name="grt_disclaimers[{{INDEX}}][id]" value="disc_{{UNIQUE_ID}}" />
            </div>
        </script>

        <style>
            #grt-disclaimers-container { margin-top: 10px; }
            .grt-disclaimer-item {
                background: #f9f9f9;
                border: 1px solid #ddd;
                border-radius: 4px;
                margin-bottom: 15px;
                padding: 15px;
            }
            .grt-disclaimer-header {
                display: flex;
                justify-content: space-between;
                align-items: center;
                margin-bottom: 15px;
                border-bottom: 1px solid #ddd;
                padding-bottom: 10px;
            }
            .grt-disclaimer-header h4 { margin: 0; }
            .grt-disclaimer-text-field { margin-bottom: 15px; }
            .grt-disclaimer-text-field label,
            .grt-disclaimer-states-field label { font-weight: 600; display: block; margin-bottom: 5px; }
            .grt-disclaimer-states-controls { margin-bottom: 10px; }
            .grt-disclaimer-states-controls button { margin-right: 5px; }
            .grt-disclaimer-states-grid {
                display: grid;
                grid-template-columns: repeat(auto-fill, minmax(200px, 1fr));
                gap: 10px;
                max-height: 300px;
                overflow-y: auto;
                border: 1px solid #ddd;
                padding: 15px;
                background: #fff;
            }
            .grt-disclaimer-states-grid label {
                display: flex;
                align-items: center;
                font-weight: normal;
            }
            .grt-disclaimer-states-grid input { margin-right: 3px; }
            #grt-add-disclaimer { margin-top: 10px; }
        </style>
        <?php
    }

    /**
     * Save meta box data
     */
    public function save_meta_boxes($post_id) {
        // Check if nonce is set
        if (!isset($_POST['grt_meta_nonce'])) {
            return;
        }
        
        // Verify nonce
        if (!wp_verify_nonce($_POST['grt_meta_nonce'], 'grt_save_meta_boxes')) {
            return;
        }
        
        // Check if user has permission to edit
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }
        
        // Don't save on autosave
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }
        
        // Save meta fields
        $meta_fields = array(
            'grt_site_name',
            'grt_site_description',
            'grt_star_rating',
            'grt_affiliate_link',
            'grt_bonus_info',
            'grt_license_info'
        );
        
        foreach ($meta_fields as $field) {
            if (isset($_POST[$field])) {
                update_post_meta($post_id, '_' . $field, sanitize_text_field($_POST[$field]));
            }
        }
        
        // Save states accepted
        if (isset($_POST['grt_states_accepted']) && is_array($_POST['grt_states_accepted'])) {
            $states = array_map('sanitize_text_field', $_POST['grt_states_accepted']);
            update_post_meta($post_id, '_grt_states_accepted', $states);
        } else {
            update_post_meta($post_id, '_grt_states_accepted', array());
        }

        // Save pros (textarea to array)
        if (isset($_POST['grt_pros'])) {
            $pros_text = sanitize_textarea_field($_POST['grt_pros']);
            $pros_array = array_filter(array_map('trim', explode("\n", $pros_text)));
            update_post_meta($post_id, '_grt_pros', $pros_array);
        }

        // Save cons (textarea to array)
        if (isset($_POST['grt_cons'])) {
            $cons_text = sanitize_textarea_field($_POST['grt_cons']);
            $cons_array = array_filter(array_map('trim', explode("\n", $cons_text)));
            update_post_meta($post_id, '_grt_cons', $cons_array);
        }

        // Save payout speed
        if (isset($_POST['grt_payout_speed'])) {
            update_post_meta($post_id, '_grt_payout_speed', sanitize_text_field($_POST['grt_payout_speed']));
        }

        // Save compatibility (checkbox array)
        if (isset($_POST['grt_compatibility']) && is_array($_POST['grt_compatibility'])) {
            $valid_platforms = array('ios', 'android', 'windows', 'mac');
            $compatibility = array_intersect(
                array_map('sanitize_text_field', $_POST['grt_compatibility']),
                $valid_platforms
            );
            update_post_meta($post_id, '_grt_compatibility', array_values($compatibility));
        } else {
            update_post_meta($post_id, '_grt_compatibility', array());
        }

        // Save payout percentage
        if (isset($_POST['grt_payout_percentage'])) {
            update_post_meta($post_id, '_grt_payout_percentage', sanitize_text_field($_POST['grt_payout_percentage']));
        }

        // Save review link
        if (isset($_POST['grt_review_link'])) {
            update_post_meta($post_id, '_grt_review_link', esc_url_raw($_POST['grt_review_link']));
        }

        // Save certified badge (attachment ID)
        if (isset($_POST['grt_certified_badge'])) {
            $badge_id = intval($_POST['grt_certified_badge']);
            if ($badge_id > 0 && get_post_type($badge_id) === 'attachment') {
                update_post_meta($post_id, '_grt_certified_badge', $badge_id);
            } else {
                delete_post_meta($post_id, '_grt_certified_badge');
            }
        }

        // Save disclaimers
        if (isset($_POST['grt_disclaimers']) && is_array($_POST['grt_disclaimers'])) {
            $disclaimers = array();
            $valid_states = array(
                // US States
                'AL', 'AK', 'AZ', 'AR', 'CA', 'CO', 'CT', 'DE', 'FL', 'GA',
                'HI', 'ID', 'IL', 'IN', 'IA', 'KS', 'KY', 'LA', 'ME', 'MD',
                'MA', 'MI', 'MN', 'MS', 'MO', 'MT', 'NE', 'NV', 'NH', 'NJ',
                'NM', 'NY', 'NC', 'ND', 'OH', 'OK', 'OR', 'PA', 'RI', 'SC',
                'SD', 'TN', 'TX', 'UT', 'VT', 'VA', 'WA', 'WV', 'WI', 'WY', 'DC',
                // Canadian Provinces/Territories
                'AB', 'BC', 'MB', 'NB', 'NL', 'NS', 'NT', 'NU', 'ON', 'PE', 'QC', 'SK', 'YT'
            );

            foreach ($_POST['grt_disclaimers'] as $disc) {
                // Skip empty disclaimers
                if (empty($disc['text']) || !isset($disc['states']) || empty($disc['states'])) {
                    continue;
                }

                // Sanitize text with wp_kses_post to allow limited HTML
                $text = wp_kses_post($disc['text']);

                // Sanitize and validate states
                $states = array_values(array_intersect(
                    array_map('sanitize_text_field', (array)$disc['states']),
                    $valid_states
                ));

                // Sanitize ID or generate new one
                $id = !empty($disc['id']) ? sanitize_text_field($disc['id']) : 'disc_' . uniqid();

                if (!empty($text) && !empty($states)) {
                    $disclaimers[] = array(
                        'id' => $id,
                        'text' => $text,
                        'states' => $states,
                    );
                }
            }

            update_post_meta($post_id, '_grt_disclaimers', $disclaimers);
        } else {
            update_post_meta($post_id, '_grt_disclaimers', array());
        }
    }
    
    /**
     * Enqueue admin scripts
     */
    public function enqueue_admin_scripts($hook) {
        if ('post.php' == $hook || 'post-new.php' == $hook) {
            global $post_type;
            if ('grt_site_block' == $post_type) {
                wp_enqueue_script('jquery');
                wp_enqueue_media();

                wp_enqueue_script(
                    'grt-site-blocks-admin',
                    GRT_PLUGIN_URL . 'assets/js/site-blocks-admin.js',
                    array('jquery'),
                    GRT_VERSION,
                    true
                );
            }
        }
    }
    
    /**
     * Get all locations (US states + Canadian provinces/territories)
     * Shared data source for geo settings and disclaimers
     */
    private function get_all_locations() {
        return array(
            // US States
            'AL' => 'Alabama', 'AK' => 'Alaska', 'AZ' => 'Arizona', 'AR' => 'Arkansas', 'CA' => 'California',
            'CO' => 'Colorado', 'CT' => 'Connecticut', 'DE' => 'Delaware', 'FL' => 'Florida', 'GA' => 'Georgia',
            'HI' => 'Hawaii', 'ID' => 'Idaho', 'IL' => 'Illinois', 'IN' => 'Indiana', 'IA' => 'Iowa',
            'KS' => 'Kansas', 'KY' => 'Kentucky', 'LA' => 'Louisiana', 'ME' => 'Maine', 'MD' => 'Maryland',
            'MA' => 'Massachusetts', 'MI' => 'Michigan', 'MN' => 'Minnesota', 'MS' => 'Mississippi', 'MO' => 'Missouri',
            'MT' => 'Montana', 'NE' => 'Nebraska', 'NV' => 'Nevada', 'NH' => 'New Hampshire', 'NJ' => 'New Jersey',
            'NM' => 'New Mexico', 'NY' => 'New York', 'NC' => 'North Carolina', 'ND' => 'North Dakota', 'OH' => 'Ohio',
            'OK' => 'Oklahoma', 'OR' => 'Oregon', 'PA' => 'Pennsylvania', 'RI' => 'Rhode Island', 'SC' => 'South Carolina',
            'SD' => 'South Dakota', 'TN' => 'Tennessee', 'TX' => 'Texas', 'UT' => 'Utah', 'VT' => 'Vermont',
            'VA' => 'Virginia', 'WA' => 'Washington', 'WV' => 'West Virginia', 'WI' => 'Wisconsin', 'WY' => 'Wyoming',
            'DC' => 'District of Columbia',
            // Canadian Provinces/Territories
            'AB' => 'Alberta', 'BC' => 'British Columbia', 'MB' => 'Manitoba',
            'NB' => 'New Brunswick', 'NL' => 'Newfoundland and Labrador',
            'NS' => 'Nova Scotia', 'NT' => 'Northwest Territories', 'NU' => 'Nunavut',
            'ON' => 'Ontario', 'PE' => 'Prince Edward Island', 'QC' => 'Quebec',
            'SK' => 'Saskatchewan', 'YT' => 'Yukon'
        );
    }

    /**
     * Custom updated messages for site blocks
     */
    public function updated_messages($messages) {
        $post = get_post();
        
        $messages['grt_site_block'] = array(
            0  => '', // Unused. Messages start at index 1.
            1  => __('Site block updated.', 'geo-ranking-tables'),
            2  => __('Custom field updated.', 'geo-ranking-tables'),
            3  => __('Custom field deleted.', 'geo-ranking-tables'),
            4  => __('Site block updated.', 'geo-ranking-tables'),
            5  => isset($_GET['revision']) ? sprintf(__('Site block restored to revision from %s', 'geo-ranking-tables'), wp_post_revision_title((int) $_GET['revision'], false)) : false,
            6  => __('Site block published.', 'geo-ranking-tables'),
            7  => __('Site block saved.', 'geo-ranking-tables'),
            8  => __('Site block submitted.', 'geo-ranking-tables'),
            9  => sprintf(__('Site block scheduled for: <strong>%1$s</strong>.', 'geo-ranking-tables'), date_i18n(__('M j, Y @ G:i'), strtotime($post->post_date))),
            10 => __('Site block draft updated.', 'geo-ranking-tables')
        );
        
        return $messages;
    }
    
    /**
     * Add custom columns to site blocks list
     */
    public function add_custom_columns($columns) {
        $new_columns = array();
        $new_columns['cb'] = $columns['cb'];
        $new_columns['title'] = $columns['title'];
        $new_columns['site_name'] = __('Site Name', 'geo-ranking-tables');
        $new_columns['rating'] = __('Rating', 'geo-ranking-tables');
        $new_columns['states_count'] = __('States', 'geo-ranking-tables');
        $new_columns['date'] = $columns['date'];
        
        return $new_columns;
    }
    
    /**
     * Custom column content
     */
    public function custom_column_content($column, $post_id) {
        switch ($column) {
            case 'site_name':
                $site_name = get_post_meta($post_id, '_grt_site_name', true);
                echo esc_html($site_name ? $site_name : '-');
                break;
                
            case 'rating':
                $rating = get_post_meta($post_id, '_grt_star_rating', true);
                if ($rating) {
                    echo esc_html($rating . '/5 ⭐');
                } else {
                    echo '-';
                }
                break;
                
            case 'states_count':
                $states = get_post_meta($post_id, '_grt_states_accepted', true);
                if (is_array($states)) {
                    echo esc_html(count($states) . ' states');
                } else {
                    echo '0 states';
                }
                break;
        }
    }
}