
(function(){
    if (typeof window === 'undefined' || typeof document === 'undefined') return;

    function drawDonut(canvas, cfg) {
        var ctx = canvas.getContext('2d');
        var w = canvas.width, h = canvas.height;
        var cx = w/2, cy = h/2;
        var outerR = Math.min(w, h) * 0.48;
        var innerR = outerR - (cfg.ringWidth || 36);

        function clear(){ ctx.clearRect(0,0,w,h); }
        function ringSegment(start, end, color){
            ctx.beginPath(); ctx.arc(cx, cy, outerR, start, end, false);
            ctx.arc(cx, cy, innerR, end, start, true); ctx.closePath();
            ctx.fillStyle = color; ctx.fill();
        }
        function divider(){
            if (!cfg.showDivider) return;
            ctx.save(); ctx.strokeStyle = 'rgba(0,0,0,0.15)';
            ctx.lineWidth = Math.max(1, cfg.ringWidth*0.04);
            ctx.beginPath(); ctx.moveTo(cx, cy - innerR + 2); ctx.lineTo(cx, cy + innerR - 2);
            ctx.stroke(); ctx.restore();
        }
        function centerShadow(){
            ctx.save();
            var grd = ctx.createRadialGradient(cx, cy, innerR*0.7, cx, cy, innerR);
            grd.addColorStop(0,'rgba(255,255,255,0.9)'); grd.addColorStop(1,'rgba(255,255,255,0.4)');
            ctx.fillStyle = grd; ctx.beginPath(); ctx.arc(cx, cy, innerR, 0, Math.PI*2); ctx.fill(); ctx.restore();
        }

        var start = -Math.PI/2;
        var aAngle = (cfg.winA/100) * Math.PI*2;
        var bAngle = (cfg.winB/100) * Math.PI*2;

        var steps = cfg.animate ? 24 : 1, i = 0;
        function paintFrame(){
            var t = (i+1)/steps; var aCur = aAngle*t; var bCur = bAngle*t;
            clear();
            ctx.save(); ctx.shadowColor = 'rgba(0,0,0,0.12)'; ctx.shadowBlur = 10; ctx.shadowOffsetY = 3;
            ringSegment(start, start + aCur, cfg.colorA);
            ringSegment(start + aCur, start + aCur + bCur, cfg.colorB);
            ctx.restore();
            centerShadow(); divider();
            ctx.save(); ctx.textAlign='center'; ctx.textBaseline='middle'; ctx.font='600 22px system-ui, -apple-system, Segoe UI, Roboto, sans-serif'; ctx.fillStyle='#0f172a';
            ctx.fillText(Math.round(cfg.winA)+'% / '+Math.round(cfg.winB)+'%', cx, cy); ctx.restore();
            if (++i < steps) requestAnimationFrame(paintFrame);
        }
        paintFrame();

        // tooltip
        var tip = document.createElement('div');
        tip.className='gsm-mp-tooltip'; tip.style.position='absolute'; tip.style.pointerEvents='none'; tip.style.zIndex='3';
        tip.style.padding='6px 8px'; tip.style.borderRadius='6px'; tip.style.background='rgba(20,20,20,0.92)'; tip.style.color='#fff';
        tip.style.font='500 12px system-ui, -apple-system, Segoe UI, Roboto, sans-serif'; tip.style.display='none';
        canvas.parentNode.style.position='relative'; canvas.parentNode.appendChild(tip);

        function getMousePos(evt){ var r = canvas.getBoundingClientRect(); return { x: evt.clientX-r.left, y: evt.clientY-r.top }; }
        function isInsideDonut(mx,my){ var dx=mx-cx, dy=my-cy, R=Math.sqrt(dx*dx+dy*dy); return R<=outerR && R>=innerR; }
        function whichSeg(mx,my){ var ang=Math.atan2(my-cy,mx-cx); if (ang<start) ang+=Math.PI*2; var rel=ang-start; if (rel>=0 && rel<=aAngle) return 'A'; if (rel>aAngle && rel<=aAngle+bAngle) return 'B'; return null; }

        canvas.addEventListener('mousemove', function(evt){
            var p=getMousePos(evt); if (!isInsideDonut(p.x,p.y)){ tip.style.display='none'; canvas.style.cursor='default'; return; }
            var seg=whichSeg(p.x,p.y); if (!seg){ tip.style.display='none'; canvas.style.cursor='default'; return; }
            canvas.style.cursor='pointer'; var pct=Math.round(seg==='A'?cfg.winA:cfg.winB); var team=seg==='A'?cfg.teamA:cfg.teamB;
            tip.textContent=team+': '+pct+'%'; tip.style.display='block'; tip.style.left=(p.x+10)+'px'; tip.style.top=(p.y-10)+'px';
        });
        canvas.addEventListener('mouseleave', function(){ tip.style.display='none'; canvas.style.cursor='default'; });
    }

    function initAll(){
        document.querySelectorAll('.gsm-mp-canvas').forEach(function(cv){
            var cfg = {
                teamA: cv.dataset.teama || 'Team A',
                teamB: cv.dataset.teamb || 'Team B',
                winA: parseFloat(cv.dataset.wina || '50'),
                winB: parseFloat(cv.dataset.winb || '50'),
                colorA: cv.dataset.colora || '#2e7dff',
                colorB: cv.dataset.colorb || '#f5b700',
                ringWidth: parseFloat(cv.dataset.ringw || '36'),
                showDivider: cv.dataset.divider === '1',
                animate: cv.dataset.animate === '1'
            };
            if (!isFinite(cfg.winB)) cfg.winB = 100 - cfg.winA;
            drawDonut(cv, cfg);
        });
    }
    if (document.readyState === 'loading') document.addEventListener('DOMContentLoaded', initAll);
    else initAll();
})();
