<?php
/**
 * Plugin Name: GamblingSite Matchup Predictor (Canvas)
 * Description: Gutenberg block to enter two teams, logos, win %, and odds; renders a polished donut breakdown with team colors, animation, and hover tooltips.
 * Version: 1.1.1
 * Author: FT Media + ChatGPT
 * License: GPLv2 or later
 */

if ( ! defined( 'ABSPATH' ) ) { exit; }

define( 'GSM_MP_VERSION', '1.1.1' );

add_action( 'init', 'gsm_mp_register_block' );
function gsm_mp_register_block() {
    wp_register_script(
        'gsm-mp-editor',
        plugins_url( 'editor.js', __FILE__ ),
        array( 'wp-blocks', 'wp-element', 'wp-components', 'wp-block-editor', 'wp-i18n' ),
        GSM_MP_VERSION,
        true
    );
    wp_register_script(
        'gsm-mp-frontend',
        plugins_url( 'frontend.js', __FILE__ ),
        array(),
        GSM_MP_VERSION,
        true
    );
    wp_register_style(
        'gsm-mp-style',
        plugins_url( 'style.css', __FILE__ ),
        array(),
        GSM_MP_VERSION
    );

    register_block_type( 'gsm/matchup-predictor', array(
        'editor_script'   => 'gsm-mp-editor',
        'style'           => 'gsm-mp-style',
        'render_callback' => 'gsm_mp_render_block',
        'attributes'      => array(
            'teamAName'    => array( 'type' => 'string', 'default' => 'Team A' ),
            'teamBName'    => array( 'type' => 'string', 'default' => 'Team B' ),
            'teamALogoUrl' => array( 'type' => 'string', 'default' => '' ),
            'teamBLogoUrl' => array( 'type' => 'string', 'default' => '' ),
            'winA'         => array( 'type' => 'number', 'default' => 55 ),
            'winB'         => array( 'type' => 'number', 'default' => 45 ),
            'mlA'          => array( 'type' => 'string', 'default' => '' ),
            'mlB'          => array( 'type' => 'string', 'default' => '' ),
            'spreadA'      => array( 'type' => 'string', 'default' => '' ),
            'spreadB'      => array( 'type' => 'string', 'default' => '' ),
            'total'        => array( 'type' => 'string', 'default' => '' ),
            'book'         => array( 'type' => 'string', 'default' => '' ),
            'kickoff'      => array( 'type' => 'string', 'default' => '' ),
            'location'     => array( 'type' => 'string', 'default' => '' ),
            'note'         => array( 'type' => 'string', 'default' => '' ),
            'teamAColor'   => array( 'type' => 'string', 'default' => '#2e7dff' ),
            'teamBColor'   => array( 'type' => 'string', 'default' => '#f5b700' ),
            'ringWidth'    => array( 'type' => 'number', 'default' => 36 ),
            'showDivider'  => array( 'type' => 'boolean', 'default' => true ),
            'animateDraw'  => array( 'type' => 'boolean', 'default' => true ),
            'showHeader'   => array( 'type' => 'boolean', 'default' => true ),
            'headerText'   => array( 'type' => 'string', 'default' => 'Prediction Breakdown' ),
            'league'       => array( 'type' => 'string', 'default' => 'MLB' ),
            'teamAKey'     => array( 'type' => 'string', 'default' => '' ),
            'teamBKey'     => array( 'type' => 'string', 'default' => '' ),
        )
    ) );
}

function gsm_mp_safe_text( $v ) { return is_scalar($v) ? $v : ''; }

function gsm_mp_render_block( $attributes ) {
    if ( defined('REST_REQUEST') && REST_REQUEST && ! is_admin() ) { return ''; }
    if ( ! is_admin() ) { wp_enqueue_script( 'gsm-mp-frontend' ); wp_enqueue_style( 'gsm-mp-style' ); }

    $def = wp_parse_args( is_array($attributes) ? $attributes : array(), array(
        'teamAName'    => 'Team A',
        'teamBName'    => 'Team B',
        'teamALogoUrl' => '',
        'teamBLogoUrl' => '',
        'winA'         => 55,
        'winB'         => 45,
        'mlA'          => '',
        'mlB'          => '',
        'spreadA'      => '',
        'spreadB'      => '',
        'total'        => '',
        'book'         => '',
        'kickoff'      => '',
        'location'     => '',
        'note'         => '',
        'teamAColor'   => '#2e7dff',
        'teamBColor'   => '#f5b700',
        'ringWidth'    => 36,
        'showDivider'  => true,
        'animateDraw'  => true,
        'showHeader'   => true,
        'headerText'   => 'Prediction Breakdown',
        'league'       => 'MLB',
        'teamAKey'     => '',
        'teamBKey'     => '',
    ) );

    $def['teamAName']    = sanitize_text_field( gsm_mp_safe_text( $def['teamAName'] ) );
    $def['teamBName']    = sanitize_text_field( gsm_mp_safe_text( $def['teamBName'] ) );
    $def['teamALogoUrl'] = esc_url_raw( gsm_mp_safe_text( $def['teamALogoUrl'] ) );
    $def['teamBLogoUrl'] = esc_url_raw( gsm_mp_safe_text( $def['teamBLogoUrl'] ) );
    $def['winA'] = floatval( $def['winA'] );
    $def['winB'] = floatval( $def['winB'] );
    if ( is_numeric( $def['winA'] ) && ( ! is_numeric( $def['winB'] ) || $def['winB'] === '' ) ) { $def['winB'] = max(0,min(100,100-$def['winA'])); }
    elseif ( is_numeric( $def['winB'] ) && ( ! is_numeric( $def['winA'] ) || $def['winA'] === '' ) ) { $def['winA'] = max(0,min(100,100-$def['winB'])); }

    $aColor = preg_match('/^#([0-9a-f]{3}|[0-9a-f]{6})$/i', $def['teamAColor']) ? $def['teamAColor'] : '#2e7dff';
    $bColor = preg_match('/^#([0-9a-f]{3}|[0-9a-f]{6})$/i', $def['teamBColor']) ? $def['teamBColor'] : '#f5b700';
    $ringW  = max(16, min(60, intval($def['ringWidth']) ));

    ob_start(); ?>
    <div class="gsm-mp-card gsm-mp-elevated" role="group" aria-label="Matchup Predictor">
        <?php if ( ! empty($def['showHeader']) ) : ?>
            <div class="gsm-mp-header"><?php echo esc_html( $def['headerText'] ); ?></div>
        <?php endif; ?>
        <div class="gsm-mp-toprow">
            <div class="gsm-mp-team gsm-mp-left">
                <?php if ( ! empty( $def['teamALogoUrl'] ) ) : ?>
                    <img src="<?php echo esc_url( $def['teamALogoUrl'] ); ?>" alt="<?php echo esc_attr( $def['teamAName'] ); ?> logo" class="gsm-mp-logo" />
                <?php endif; ?>
                <h3 class="gsm-mp-teamname"><?php echo esc_html( $def['teamAName'] ); ?></h3>
                <div class="gsm-mp-pill">Win: <strong><?php echo esc_html( round( $def['winA'], 1 ) ); ?>%</strong></div>
                <?php if ( $def['mlA'] !== '' ) : ?><div class="gsm-mp-pill">ML: <strong><?php echo esc_html( $def['mlA'] ); ?></strong></div><?php endif; ?>
                <?php if ( $def['spreadA'] !== '' ) : ?><div class="gsm-mp-pill">Spread: <strong><?php echo esc_html( $def['spreadA'] ); ?></strong></div><?php endif; ?>
            </div>

            <div class="gsm-mp-center">
                <canvas class="gsm-mp-canvas"
                    width="360" height="360"
                    data-teama="<?php echo esc_attr( $def['teamAName'] ); ?>"
                    data-teamb="<?php echo esc_attr( $def['teamBName'] ); ?>"
                    data-wina="<?php echo esc_attr( $def['winA'] ); ?>"
                    data-winb="<?php echo esc_attr( $def['winB'] ); ?>"
                    data-colora="<?php echo esc_attr( $aColor ); ?>"
                    data-colorb="<?php echo esc_attr( $bColor ); ?>"
                    data-ringw="<?php echo esc_attr( $ringW ); ?>"
                    data-divider="<?php echo esc_attr( !empty($def['showDivider']) ? '1':'0' ); ?>"
                    data-animate="<?php echo esc_attr( !empty($def['animateDraw']) ? '1':'0' ); ?>"
                    aria-label="Win probability donut chart" role="img"></canvas>
                <div class="gsm-mp-vs">VS</div>
            </div>

            <div class="gsm-mp-team gsm-mp-right">
                <?php if ( ! empty( $def['teamBLogoUrl'] ) ) : ?>
                    <img src="<?php echo esc_url( $def['teamBLogoUrl'] ); ?>" alt="<?php echo esc_attr( $def['teamBName'] ); ?> logo" class="gsm-mp-logo" />
                <?php endif; ?>
                <h3 class="gsm-mp-teamname"><?php echo esc_html( $def['teamBName'] ); ?></h3>
                <div class="gsm-mp-pill">Win: <strong><?php echo esc_html( round( $def['winB'], 1 ) ); ?>%</strong></div>
                <?php if ( $def['mlB'] !== '' ) : ?><div class="gsm-mp-pill">ML: <strong><?php echo esc_html( $def['mlB'] ); ?></strong></div><?php endif; ?>
                <?php if ( $def['spreadB'] !== '' ) : ?><div class="gsm-mp-pill">Spread: <strong><?php echo esc_html( $def['spreadB'] ); ?></strong></div><?php endif; ?>
            </div>
        </div>

        <div class="gsm-mp-bottomrow">
            <?php if ( $def['total'] !== '' ) : ?><div class="gsm-mp-pill">Total: <strong><?php echo esc_html( $def['total'] ); ?></strong></div><?php endif; ?>
            <?php if ( $def['book'] !== '' ) : ?><div class="gsm-mp-pill">Book: <strong><?php echo esc_html( $def['book'] ); ?></strong></div><?php endif; ?>
            <?php if ( $def['kickoff'] !== '' ) : ?><div class="gsm-mp-pill">Start: <strong><?php echo esc_html( $def['kickoff'] ); ?></strong></div><?php endif; ?>
            <?php if ( $def['location'] !== '' ) : ?><div class="gsm-mp-pill">Venue: <strong><?php echo esc_html( $def['location'] ); ?></strong></div><?php endif; ?>
            <?php if ( $def['note'] !== '' ) : ?><div class="gsm-mp-note"><?php echo esc_html( $def['note'] ); ?></div><?php endif; ?>
        </div>
    </div>
    <?php
    return ob_get_clean();
}
