<?php

if (!defined('ABSPATH')) {
    exit;
}

$pick_manager = new HON_Pick_Manager();

// Handle admin actions
if (isset($_POST['action']) && wp_verify_nonce($_POST['_wpnonce'], 'hon_admin_action')) {
    switch ($_POST['action']) {
        case 'auto_generate':
            $result = $pick_manager->auto_generate_picks();
            if ($result['success']) {
                echo '<div class="notice notice-success"><p>' . esc_html($result['message']) . '</p></div>';
            } else {
                echo '<div class="notice notice-error"><p>' . esc_html($result['message']) . '</p></div>';
            }
            break;
            
        case 'cleanup':
            $result = $pick_manager->cleanup_test_data();
            $total_deleted = $result['test_picks_deleted'] + $result['votes_deleted'] + $result['duplicate_picks_deleted'];
            echo '<div class="notice notice-success"><p>Cleanup completed! Removed ' . $total_deleted . ' items.</p></div>';
            break;
            
        case 'toggle_status':
            if (isset($_POST['pick_id'])) {
                $pick_manager->toggle_pick_status(sanitize_text_field($_POST['pick_id']));
                echo '<div class="notice notice-success"><p>Pick status updated!</p></div>';
            }
            break;
            
        case 'delete_pick':
            if (isset($_POST['pick_id'])) {
                $pick_manager->delete_pick(sanitize_text_field($_POST['pick_id']));
                echo '<div class="notice notice-success"><p>Pick deleted!</p></div>';
            }
            break;
    }
}

$picks = $pick_manager->get_picks(50);
$analytics = $pick_manager->get_analytics();

?>
<div class="wrap">
    <h1><?php _e('Hot or Not - Daily Betting Picks', 'hot-or-not-betting-picks'); ?></h1>
    
    <!-- Quick Stats -->
    <div class="hon-admin-stats">
        <div class="hon-stat-card">
            <h3><?php echo esc_html($analytics['total_picks']); ?></h3>
            <p><?php _e('Total Picks', 'hot-or-not-betting-picks'); ?></p>
        </div>
        <div class="hon-stat-card">
            <h3><?php echo esc_html($analytics['active_picks']); ?></h3>
            <p><?php _e('Active Picks', 'hot-or-not-betting-picks'); ?></p>
        </div>
        <div class="hon-stat-card">
            <h3><?php echo esc_html($analytics['total_votes']); ?></h3>
            <p><?php _e('Total Votes', 'hot-or-not-betting-picks'); ?></p>
        </div>
    </div>
    
    <!-- Admin Actions -->
    <div class="hon-admin-actions">
        <form method="post" style="display: inline-block;">
            <?php wp_nonce_field('hon_admin_action'); ?>
            <input type="hidden" name="action" value="auto_generate">
            <button type="submit" class="button button-primary">
                🎲 <?php _e('Auto-Generate Picks from Real Games', 'hot-or-not-betting-picks'); ?>
            </button>
        </form>
        
        <form method="post" style="display: inline-block;">
            <?php wp_nonce_field('hon_admin_action'); ?>
            <input type="hidden" name="action" value="cleanup">
            <button type="submit" class="button button-secondary">
                🧹 <?php _e('Clean Up Test Data', 'hot-or-not-betting-picks'); ?>
            </button>
        </form>
    </div>
    
    <!-- Picks Table -->
    <h2><?php _e('All Picks', 'hot-or-not-betting-picks'); ?></h2>
    
    <?php if (empty($picks)): ?>
        <div class="hon-empty-state">
            <h3><?php _e('No picks available yet', 'hot-or-not-betting-picks'); ?></h3>
            <p><?php _e('Generate picks from today\'s real games to get started!', 'hot-or-not-betting-picks'); ?></p>
        </div>
    <?php else: ?>
        <table class="wp-list-table widefat fixed striped">
            <thead>
                <tr>
                    <th><?php _e('Matchup', 'hot-or-not-betting-picks'); ?></th>
                    <th><?php _e('Pick', 'hot-or-not-betting-picks'); ?></th>
                    <th><?php _e('Sport', 'hot-or-not-betting-picks'); ?></th>
                    <th><?php _e('Game Time', 'hot-or-not-betting-picks'); ?></th>
                    <th><?php _e('Votes', 'hot-or-not-betting-picks'); ?></th>
                    <th><?php _e('Status', 'hot-or-not-betting-picks'); ?></th>
                    <th><?php _e('Actions', 'hot-or-not-betting-picks'); ?></th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($picks as $pick): 
                    $total_votes = $pick['hot_votes'] + $pick['not_votes'];
                    $hot_percentage = $total_votes > 0 ? round(($pick['hot_votes'] / $total_votes) * 100, 1) : 0;
                    $game_time = get_date_from_gmt($pick['game_time'], 'M j, Y g:i A');
                ?>
                    <tr>
                        <td>
                            <strong><?php echo esc_html($pick['matchup']); ?></strong>
                            <div class="hon-insight"><?php echo esc_html($pick['insight']); ?></div>
                        </td>
                        <td>
                            <strong><?php echo esc_html($pick['pick_text']); ?></strong>
                            <?php if ($pick['odds']): ?>
                                <br><span class="hon-odds"><?php echo esc_html($pick['odds']); ?></span>
                            <?php endif; ?>
                        </td>
                        <td><span class="hon-sport-badge"><?php echo esc_html($pick['sport']); ?></span></td>
                        <td><?php echo esc_html($game_time); ?></td>
                        <td>
                            🔥 <?php echo esc_html($pick['hot_votes']); ?> (<?php echo esc_html($hot_percentage); ?>%)<br>
                            ❌ <?php echo esc_html($pick['not_votes']); ?>
                        </td>
                        <td>
                            <span class="hon-status-<?php echo $pick['is_active'] ? 'active' : 'inactive'; ?>">
                                <?php echo $pick['is_active'] ? __('Active', 'hot-or-not-betting-picks') : __('Inactive', 'hot-or-not-betting-picks'); ?>
                            </span>
                        </td>
                        <td>
                            <form method="post" style="display: inline-block;">
                                <?php wp_nonce_field('hon_admin_action'); ?>
                                <input type="hidden" name="action" value="toggle_status">
                                <input type="hidden" name="pick_id" value="<?php echo esc_attr($pick['pick_id']); ?>">
                                <button type="submit" class="button button-small">
                                    <?php echo $pick['is_active'] ? __('Deactivate', 'hot-or-not-betting-picks') : __('Activate', 'hot-or-not-betting-picks'); ?>
                                </button>
                            </form>
                            
                            <form method="post" style="display: inline-block;" onsubmit="return confirm('Are you sure you want to delete this pick?');">
                                <?php wp_nonce_field('hon_admin_action'); ?>
                                <input type="hidden" name="action" value="delete_pick">
                                <input type="hidden" name="pick_id" value="<?php echo esc_attr($pick['pick_id']); ?>">
                                <button type="submit" class="button button-small button-link-delete">
                                    <?php _e('Delete', 'hot-or-not-betting-picks'); ?>
                                </button>
                            </form>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    <?php endif; ?>
    
    <!-- Shortcode Information -->
    <div class="hon-shortcode-info">
        <h3><?php _e('Display Picks on Your Site', 'hot-or-not-betting-picks'); ?></h3>
        <p><?php _e('Use this shortcode to display the betting picks on any page or post:', 'hot-or-not-betting-picks'); ?></p>
        <code>[hot_or_not_picks]</code>
        
        <h4><?php _e('Shortcode Options:', 'hot-or-not-betting-picks'); ?></h4>
        <ul>
            <li><code>[hot_or_not_picks limit="6"]</code> - <?php _e('Limit number of picks shown', 'hot-or-not-betting-picks'); ?></li>
            <li><code>[hot_or_not_picks sport="NBA"]</code> - <?php _e('Show only specific sport', 'hot-or-not-betting-picks'); ?></li>
            <li><code>[hot_or_not_picks show_voting="false"]</code> - <?php _e('Hide voting buttons', 'hot-or-not-betting-picks'); ?></li>
        </ul>
    </div>
</div>

<style>
.hon-admin-stats {
    display: flex;
    gap: 20px;
    margin: 20px 0;
}

.hon-stat-card {
    background: #fff;
    border: 1px solid #ccd0d4;
    border-radius: 4px;
    padding: 20px;
    text-align: center;
    min-width: 120px;
}

.hon-stat-card h3 {
    font-size: 2em;
    margin: 0 0 10px 0;
    color: #0073aa;
}

.hon-stat-card p {
    margin: 0;
    color: #666;
}

.hon-admin-actions {
    margin: 20px 0;
}

.hon-admin-actions .button {
    margin-right: 10px;
}

.hon-insight {
    font-style: italic;
    color: #666;
    font-size: 0.9em;
    margin-top: 5px;
}

.hon-odds {
    color: #0073aa;
    font-weight: bold;
}

.hon-sport-badge {
    background: #0073aa;
    color: white;
    padding: 2px 8px;
    border-radius: 3px;
    font-size: 0.8em;
    text-transform: uppercase;
}

.hon-status-active {
    color: #46b450;
    font-weight: bold;
}

.hon-status-inactive {
    color: #dc3232;
}

.hon-empty-state {
    text-align: center;
    padding: 40px;
    background: #f9f9f9;
    border: 1px solid #e5e5e5;
    border-radius: 4px;
}

.hon-shortcode-info {
    background: #f9f9f9;
    border: 1px solid #e5e5e5;
    border-radius: 4px;
    padding: 20px;
    margin-top: 30px;
}

.hon-shortcode-info code {
    background: #fff;
    border: 1px solid #ccd0d4;
    padding: 2px 6px;
    border-radius: 3px;
}
</style>