<?php

if (!defined('ABSPATH')) {
    exit;
}

// Handle settings save
if (isset($_POST['save_settings']) && wp_verify_nonce($_POST['_wpnonce'], 'hon_settings')) {
    $settings = array(
        'odds_api_key' => sanitize_text_field($_POST['odds_api_key']),
        'openai_api_key' => sanitize_text_field($_POST['openai_api_key']),
        'auto_generate_enabled' => isset($_POST['auto_generate_enabled']) ? 1 : 0,
        'auto_generate_frequency' => sanitize_text_field($_POST['auto_generate_frequency']),
        'max_picks_per_generation' => intval($_POST['max_picks_per_generation']),
        'enable_voting' => isset($_POST['enable_voting']) ? 1 : 0,
        'show_insights' => isset($_POST['show_insights']) ? 1 : 0,
        'show_game_times' => isset($_POST['show_game_times']) ? 1 : 0,
        'enabled_sports' => isset($_POST['enabled_sports']) ? array_map('sanitize_text_field', $_POST['enabled_sports']) : array()
    );
    
    update_option('hon_settings', $settings);
    
    // Update cron schedule if changed
    $old_settings = get_option('hon_settings', array());
    if ($settings['auto_generate_frequency'] !== ($old_settings['auto_generate_frequency'] ?? 'twicedaily')) {
        wp_clear_scheduled_hook('hon_auto_generate_picks');
        if ($settings['auto_generate_enabled']) {
            wp_schedule_event(time(), $settings['auto_generate_frequency'], 'hon_auto_generate_picks');
        }
    }
    
    echo '<div class="notice notice-success"><p>' . __('Settings saved successfully!', 'hot-or-not-betting-picks') . '</p></div>';
}

$settings = get_option('hon_settings', array());
$defaults = array(
    'odds_api_key' => '',
    'openai_api_key' => '',
    'auto_generate_enabled' => 1,
    'auto_generate_frequency' => 'twicedaily',
    'max_picks_per_generation' => 6,
    'enable_voting' => 1,
    'show_insights' => 1,
    'show_game_times' => 1,
    'enabled_sports' => array('NBA', 'NFL', 'MLB', 'NHL', 'MLS', 'Tennis')
);

$settings = wp_parse_args($settings, $defaults);

// Test API connections
$odds_api = new HON_Odds_API();
$openai_api = new HON_OpenAI_API();
$odds_status = $odds_api->test_connection();
$openai_status = $openai_api->test_connection();

?>
<div class="wrap">
    <h1><?php _e('Hot or Not - Settings', 'hot-or-not-betting-picks'); ?></h1>
    
    <form method="post">
        <?php wp_nonce_field('hon_settings'); ?>
        
        <table class="form-table">
            <tr>
                <th colspan="2">
                    <h2><?php _e('API Configuration', 'hot-or-not-betting-picks'); ?></h2>
                </th>
            </tr>
            
            <tr>
                <th scope="row">
                    <label for="odds_api_key"><?php _e('The Odds API Key', 'hot-or-not-betting-picks'); ?></label>
                </th>
                <td>
                    <input type="text" id="odds_api_key" name="odds_api_key" value="<?php echo esc_attr($settings['odds_api_key']); ?>" class="regular-text" />
                    <p class="description">
                        <?php _e('Get your API key from', 'hot-or-not-betting-picks'); ?> <a href="https://the-odds-api.com/" target="_blank">the-odds-api.com</a><br>
                        <?php _e('Status:', 'hot-or-not-betting-picks'); ?> 
                        <span class="hon-status-<?php echo $odds_status['status']; ?>">
                            <?php echo $odds_status['status'] === 'success' ? '✅ Connected' : '❌ ' . $odds_status['message']; ?>
                        </span>
                        <?php if ($odds_status['status'] === 'success'): ?>
                            <br><?php printf(__('Found %d sports available', 'hot-or-not-betting-picks'), $odds_status['sports_count']); ?>
                        <?php endif; ?>
                    </p>
                </td>
            </tr>
            
            <tr>
                <th scope="row">
                    <label for="openai_api_key"><?php _e('OpenAI API Key', 'hot-or-not-betting-picks'); ?></label>
                </th>
                <td>
                    <input type="password" id="openai_api_key" name="openai_api_key" value="<?php echo esc_attr($settings['openai_api_key']); ?>" class="regular-text" />
                    <p class="description">
                        <?php _e('Get your API key from', 'hot-or-not-betting-picks'); ?> <a href="https://platform.openai.com/api-keys" target="_blank">platform.openai.com</a><br>
                        <?php _e('Status:', 'hot-or-not-betting-picks'); ?> 
                        <span class="hon-status-<?php echo $openai_status['status']; ?>">
                            <?php echo $openai_status['status'] === 'success' ? '✅ Connected' : '❌ ' . $openai_status['message']; ?>
                        </span>
                        <?php if ($openai_status['status'] === 'success'): ?>
                            <br><?php _e('Sample insight:', 'hot-or-not-betting-picks'); ?> "<?php echo esc_html($openai_status['sample_insight']); ?>"
                        <?php endif; ?>
                    </p>
                </td>
            </tr>
            
            <tr>
                <th colspan="2">
                    <h2><?php _e('Auto-Generation Settings', 'hot-or-not-betting-picks'); ?></h2>
                </th>
            </tr>
            
            <tr>
                <th scope="row"><?php _e('Enable Auto-Generation', 'hot-or-not-betting-picks'); ?></th>
                <td>
                    <label>
                        <input type="checkbox" name="auto_generate_enabled" value="1" <?php checked($settings['auto_generate_enabled']); ?> />
                        <?php _e('Automatically generate picks from real games', 'hot-or-not-betting-picks'); ?>
                    </label>
                    <p class="description"><?php _e('When enabled, picks will be automatically generated twice daily from real games.', 'hot-or-not-betting-picks'); ?></p>
                </td>
            </tr>
            
            <tr>
                <th scope="row">
                    <label for="auto_generate_frequency"><?php _e('Generation Frequency', 'hot-or-not-betting-picks'); ?></label>
                </th>
                <td>
                    <select id="auto_generate_frequency" name="auto_generate_frequency">
                        <option value="hourly" <?php selected($settings['auto_generate_frequency'], 'hourly'); ?>><?php _e('Hourly', 'hot-or-not-betting-picks'); ?></option>
                        <option value="twicedaily" <?php selected($settings['auto_generate_frequency'], 'twicedaily'); ?>><?php _e('Twice Daily', 'hot-or-not-betting-picks'); ?></option>
                        <option value="daily" <?php selected($settings['auto_generate_frequency'], 'daily'); ?>><?php _e('Daily', 'hot-or-not-betting-picks'); ?></option>
                    </select>
                    <p class="description"><?php _e('How often to automatically generate new picks.', 'hot-or-not-betting-picks'); ?></p>
                </td>
            </tr>
            
            <tr>
                <th scope="row">
                    <label for="max_picks_per_generation"><?php _e('Max Picks Per Generation', 'hot-or-not-betting-picks'); ?></label>
                </th>
                <td>
                    <input type="number" id="max_picks_per_generation" name="max_picks_per_generation" value="<?php echo esc_attr($settings['max_picks_per_generation']); ?>" min="1" max="20" />
                    <p class="description"><?php _e('Maximum number of picks to generate each time.', 'hot-or-not-betting-picks'); ?></p>
                </td>
            </tr>
            
            <tr>
                <th colspan="2">
                    <h2><?php _e('Sports Selection', 'hot-or-not-betting-picks'); ?></h2>
                </th>
            </tr>
            
            <tr>
                <th scope="row"><?php _e('Enabled Sports', 'hot-or-not-betting-picks'); ?></th>
                <td>
                    <?php
                    $available_sports = array('NBA', 'NFL', 'MLB', 'NHL', 'MLS', 'Tennis');
                    foreach ($available_sports as $sport):
                    ?>
                        <label style="display: block; margin-bottom: 5px;">
                            <input type="checkbox" name="enabled_sports[]" value="<?php echo esc_attr($sport); ?>" <?php checked(in_array($sport, $settings['enabled_sports'])); ?> />
                            <?php echo esc_html($sport); ?>
                        </label>
                    <?php endforeach; ?>
                    <p class="description"><?php _e('Select which sports to include when generating picks.', 'hot-or-not-betting-picks'); ?></p>
                </td>
            </tr>
            
            <tr>
                <th colspan="2">
                    <h2><?php _e('Display Settings', 'hot-or-not-betting-picks'); ?></h2>
                </th>
            </tr>
            
            <tr>
                <th scope="row"><?php _e('Enable Voting', 'hot-or-not-betting-picks'); ?></th>
                <td>
                    <label>
                        <input type="checkbox" name="enable_voting" value="1" <?php checked($settings['enable_voting']); ?> />
                        <?php _e('Allow users to vote Hot or Not on picks', 'hot-or-not-betting-picks'); ?>
                    </label>
                </td>
            </tr>
            
            <tr>
                <th scope="row"><?php _e('Show AI Insights', 'hot-or-not-betting-picks'); ?></th>
                <td>
                    <label>
                        <input type="checkbox" name="show_insights" value="1" <?php checked($settings['show_insights']); ?> />
                        <?php _e('Display AI-generated insights for each pick', 'hot-or-not-betting-picks'); ?>
                    </label>
                </td>
            </tr>
            
            <tr>
                <th scope="row"><?php _e('Show Game Times', 'hot-or-not-betting-picks'); ?></th>
                <td>
                    <label>
                        <input type="checkbox" name="show_game_times" value="1" <?php checked($settings['show_game_times']); ?> />
                        <?php _e('Display game times for each pick', 'hot-or-not-betting-picks'); ?>
                    </label>
                </td>
            </tr>
        </table>
        
        <p class="submit">
            <input type="submit" name="save_settings" class="button-primary" value="<?php _e('Save Settings', 'hot-or-not-betting-picks'); ?>" />
        </p>
    </form>
</div>

<style>
.hon-status-success {
    color: #46b450;
    font-weight: bold;
}

.hon-status-error {
    color: #dc3232;
    font-weight: bold;
}
</style>