<?php
/**
 * Plugin Name: Hot or Not - Daily Betting Picks
 * Plugin URI: https://your-website.com/
 * Description: Display real sports betting picks with AI-generated insights and community voting. Automatically updates with live games from The Odds API and OpenAI GPT-4.
 * Version: 1.0.0
 * Author: Your Name
 * Author URI: https://your-website.com/
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: hot-or-not-betting-picks
 * Domain Path: /languages
 * Requires at least: 5.0
 * Tested up to: 6.4
 * Requires PHP: 7.4
 * Network: false
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('HON_PLUGIN_VERSION', '1.0.0');
define('HON_PLUGIN_URL', plugin_dir_url(__FILE__));
define('HON_PLUGIN_PATH', plugin_dir_path(__FILE__));
define('HON_PLUGIN_FILE', __FILE__);

// Main plugin class
class HotOrNotBettingPicks {
    
    private static $instance = null;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        $this->init();
    }
    
    private function init() {
        // Hook into WordPress
        add_action('init', array($this, 'init_plugin'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_frontend_scripts'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('wp_ajax_hon_vote', array($this, 'handle_ajax_vote'));
        add_action('wp_ajax_nopriv_hon_vote', array($this, 'handle_ajax_vote'));
        add_action('wp_ajax_hon_auto_generate', array($this, 'handle_ajax_auto_generate'));
        add_action('wp_ajax_hon_cleanup', array($this, 'handle_ajax_cleanup'));
        
        // Register activation/deactivation hooks
        register_activation_hook(__FILE__, array($this, 'activate_plugin'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate_plugin'));
        
        // Add shortcode
        add_shortcode('hot_or_not_picks', array($this, 'display_picks_shortcode'));
        
        // Schedule automatic pick generation
        if (!wp_next_scheduled('hon_auto_generate_picks')) {
            wp_schedule_event(time(), 'twicedaily', 'hon_auto_generate_picks');
        }
        add_action('hon_auto_generate_picks', array($this, 'auto_generate_picks_cron'));
        
        // Include required files
        $this->include_files();
    }
    
    private function include_files() {
        require_once HON_PLUGIN_PATH . 'includes/class-database.php';
        require_once HON_PLUGIN_PATH . 'includes/class-odds-api.php';
        require_once HON_PLUGIN_PATH . 'includes/class-openai-api.php';
        require_once HON_PLUGIN_PATH . 'includes/class-pick-manager.php';
        require_once HON_PLUGIN_PATH . 'includes/admin-functions.php';
        require_once HON_PLUGIN_PATH . 'includes/frontend-functions.php';
    }
    
    public function init_plugin() {
        // Initialize database tables
        $database = new HON_Database();
        $database->create_tables();
        
        // Load text domain for translations
        load_plugin_textdomain('hot-or-not-betting-picks', false, dirname(plugin_basename(__FILE__)) . '/languages/');
    }
    
    public function enqueue_frontend_scripts() {
        wp_enqueue_style(
            'hon-frontend-style',
            HON_PLUGIN_URL . 'assets/css/frontend.css',
            array(),
            HON_PLUGIN_VERSION
        );
        
        wp_enqueue_script(
            'hon-frontend-script',
            HON_PLUGIN_URL . 'assets/js/frontend.js',
            array('jquery'),
            HON_PLUGIN_VERSION,
            true
        );
        
        // Localize script for AJAX
        wp_localize_script('hon-frontend-script', 'hon_ajax', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('hon_nonce'),
        ));
    }
    
    public function enqueue_admin_scripts($hook) {
        // Only load on our admin pages
        if (strpos($hook, 'hot-or-not') === false) {
            return;
        }
        
        wp_enqueue_style(
            'hon-admin-style',
            HON_PLUGIN_URL . 'assets/css/admin.css',
            array(),
            HON_PLUGIN_VERSION
        );
        
        wp_enqueue_script(
            'hon-admin-script',
            HON_PLUGIN_URL . 'assets/js/admin.js',
            array('jquery'),
            HON_PLUGIN_VERSION,
            true
        );
        
        wp_localize_script('hon-admin-script', 'hon_admin_ajax', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('hon_admin_nonce'),
        ));
    }
    
    public function add_admin_menu() {
        add_menu_page(
            __('Hot or Not Picks', 'hot-or-not-betting-picks'),
            __('Hot or Not Picks', 'hot-or-not-betting-picks'),
            'manage_options',
            'hot-or-not-picks',
            array($this, 'admin_dashboard_page'),
            'dashicons-chart-line',
            30
        );
        
        add_submenu_page(
            'hot-or-not-picks',
            __('All Picks', 'hot-or-not-betting-picks'),
            __('All Picks', 'hot-or-not-betting-picks'),
            'manage_options',
            'hot-or-not-picks',
            array($this, 'admin_dashboard_page')
        );
        
        add_submenu_page(
            'hot-or-not-picks',
            __('Settings', 'hot-or-not-betting-picks'),
            __('Settings', 'hot-or-not-betting-picks'),
            'manage_options',
            'hot-or-not-picks-settings',
            array($this, 'admin_settings_page')
        );
        
        add_submenu_page(
            'hot-or-not-picks',
            __('Analytics', 'hot-or-not-betting-picks'),
            __('Analytics', 'hot-or-not-betting-picks'),
            'manage_options',
            'hot-or-not-picks-analytics',
            array($this, 'admin_analytics_page')
        );
    }
    
    public function admin_dashboard_page() {
        include HON_PLUGIN_PATH . 'admin/dashboard.php';
    }
    
    public function admin_settings_page() {
        include HON_PLUGIN_PATH . 'admin/settings.php';
    }
    
    public function admin_analytics_page() {
        include HON_PLUGIN_PATH . 'admin/analytics.php';
    }
    
    public function display_picks_shortcode($atts) {
        $atts = shortcode_atts(array(
            'limit' => 6,
            'sport' => '',
            'show_time' => 'true',
            'show_voting' => 'true',
            'auto_refresh' => 'false'
        ), $atts);
        
        ob_start();
        include HON_PLUGIN_PATH . 'templates/picks-display.php';
        return ob_get_clean();
    }
    
    public function handle_ajax_vote() {
        check_ajax_referer('hon_nonce', 'nonce');
        
        $pick_id = intval($_POST['pick_id']);
        $vote_type = sanitize_text_field($_POST['vote_type']);
        
        if (!in_array($vote_type, array('hot', 'not'))) {
            wp_die('Invalid vote type');
        }
        
        $pick_manager = new HON_Pick_Manager();
        $result = $pick_manager->add_vote($pick_id, $vote_type);
        
        wp_send_json_success($result);
    }
    
    public function handle_ajax_auto_generate() {
        check_ajax_referer('hon_admin_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Insufficient permissions');
        }
        
        $pick_manager = new HON_Pick_Manager();
        $result = $pick_manager->auto_generate_picks();
        
        wp_send_json_success($result);
    }
    
    public function handle_ajax_cleanup() {
        check_ajax_referer('hon_admin_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Insufficient permissions');
        }
        
        $pick_manager = new HON_Pick_Manager();
        $result = $pick_manager->cleanup_test_data();
        
        wp_send_json_success($result);
    }
    
    public function auto_generate_picks_cron() {
        $pick_manager = new HON_Pick_Manager();
        $pick_manager->auto_generate_picks();
    }
    
    public function activate_plugin() {
        // Create database tables
        $database = new HON_Database();
        $database->create_tables();
        
        // Schedule cron job
        if (!wp_next_scheduled('hon_auto_generate_picks')) {
            wp_schedule_event(time(), 'twicedaily', 'hon_auto_generate_picks');
        }
        
        // Flush rewrite rules
        flush_rewrite_rules();
    }
    
    public function deactivate_plugin() {
        // Clear scheduled cron job
        wp_clear_scheduled_hook('hon_auto_generate_picks');
        
        // Flush rewrite rules
        flush_rewrite_rules();
    }
}

// Initialize the plugin
function hon_init() {
    return HotOrNotBettingPicks::get_instance();
}

// Start the plugin
hon_init();

// Uninstall hook
if (function_exists('register_uninstall_hook')) {
    register_uninstall_hook(__FILE__, 'hon_uninstall_plugin');
}

function hon_uninstall_plugin() {
    // Remove database tables and options
    if (class_exists('HON_Database')) {
        $database = new HON_Database();
        $database->drop_tables();
    }
    
    // Remove plugin options
    delete_option('hon_settings');
    delete_option('hon_version');
}