<?php

if (!defined('ABSPATH')) {
    exit;
}

class HON_Database {
    
    private $picks_table;
    private $votes_table;
    
    public function __construct() {
        global $wpdb;
        $this->picks_table = $wpdb->prefix . 'hon_picks';
        $this->votes_table = $wpdb->prefix . 'hon_votes';
    }
    
    public function create_tables() {
        global $wpdb;
        
        $charset_collate = $wpdb->get_charset_collate();
        
        // Picks table
        $picks_sql = "CREATE TABLE IF NOT EXISTS {$this->picks_table} (
            id int(11) NOT NULL AUTO_INCREMENT,
            pick_id varchar(50) NOT NULL UNIQUE,
            matchup varchar(255) NOT NULL,
            team_1 varchar(100) NOT NULL,
            team_2 varchar(100) NOT NULL,
            pick_text varchar(255) NOT NULL,
            pick_type enum('moneyline', 'spread', 'total') NOT NULL,
            odds varchar(20) DEFAULT NULL,
            sport varchar(50) NOT NULL,
            game_time datetime NOT NULL,
            insight text DEFAULT NULL,
            hot_votes int(11) DEFAULT 0,
            not_votes int(11) DEFAULT 0,
            is_active tinyint(1) DEFAULT 1,
            result enum('win', 'loss', 'push') DEFAULT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY idx_sport (sport),
            KEY idx_game_time (game_time),
            KEY idx_is_active (is_active),
            KEY idx_created_at (created_at)
        ) $charset_collate;";
        
        // Votes table
        $votes_sql = "CREATE TABLE IF NOT EXISTS {$this->votes_table} (
            id int(11) NOT NULL AUTO_INCREMENT,
            pick_id varchar(50) NOT NULL,
            vote_type enum('hot', 'not') NOT NULL,
            user_id int(11) DEFAULT NULL,
            user_ip varchar(45) NOT NULL,
            user_agent varchar(255) DEFAULT NULL,
            voted_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY idx_pick_id (pick_id),
            KEY idx_user_ip (user_ip),
            KEY idx_voted_at (voted_at),
            UNIQUE KEY unique_vote (pick_id, user_ip)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($picks_sql);
        dbDelta($votes_sql);
    }
    
    public function drop_tables() {
        global $wpdb;
        $wpdb->query("DROP TABLE IF EXISTS {$this->votes_table}");
        $wpdb->query("DROP TABLE IF EXISTS {$this->picks_table}");
    }
    
    public function get_picks($limit = 20, $sport = '', $active_only = true) {
        global $wpdb;
        
        $where_clauses = array();
        $values = array();
        
        if ($active_only) {
            $where_clauses[] = "is_active = %d";
            $values[] = 1;
        }
        
        if (!empty($sport)) {
            $where_clauses[] = "sport = %s";
            $values[] = $sport;
        }
        
        $where_sql = !empty($where_clauses) ? 'WHERE ' . implode(' AND ', $where_clauses) : '';
        
        $sql = $wpdb->prepare(
            "SELECT * FROM {$this->picks_table} 
             {$where_sql}
             ORDER BY game_time ASC, created_at DESC
             LIMIT %d",
            array_merge($values, array($limit))
        );
        
        return $wpdb->get_results($sql, ARRAY_A);
    }
    
    public function get_pick($pick_id) {
        global $wpdb;
        
        return $wpdb->get_row(
            $wpdb->prepare(
                "SELECT * FROM {$this->picks_table} WHERE pick_id = %s",
                $pick_id
            ),
            ARRAY_A
        );
    }
    
    public function get_pick_by_matchup($matchup, $sport) {
        global $wpdb;
        
        return $wpdb->get_row(
            $wpdb->prepare(
                "SELECT * FROM {$this->picks_table} WHERE matchup = %s AND sport = %s",
                $matchup,
                $sport
            ),
            ARRAY_A
        );
    }
    
    public function insert_pick($data) {
        global $wpdb;
        
        // Generate unique pick ID
        $data['pick_id'] = $this->generate_pick_id();
        
        $result = $wpdb->insert(
            $this->picks_table,
            $data,
            array(
                '%s', // pick_id
                '%s', // matchup
                '%s', // team_1
                '%s', // team_2
                '%s', // pick_text
                '%s', // pick_type
                '%s', // odds
                '%s', // sport
                '%s', // game_time
                '%s', // insight
                '%d', // hot_votes
                '%d', // not_votes
                '%d', // is_active
                '%s'  // result
            )
        );
        
        if ($result !== false) {
            return $data['pick_id'];
        }
        
        return false;
    }
    
    public function update_pick($pick_id, $data) {
        global $wpdb;
        
        return $wpdb->update(
            $this->picks_table,
            $data,
            array('pick_id' => $pick_id),
            null,
            array('%s')
        );
    }
    
    public function delete_pick($pick_id) {
        global $wpdb;
        
        // Delete votes first
        $wpdb->delete(
            $this->votes_table,
            array('pick_id' => $pick_id),
            array('%s')
        );
        
        // Delete pick
        return $wpdb->delete(
            $this->picks_table,
            array('pick_id' => $pick_id),
            array('%s')
        );
    }
    
    public function add_vote($pick_id, $vote_type, $user_ip) {
        global $wpdb;
        
        // Check if user already voted
        $existing_vote = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT id FROM {$this->votes_table} WHERE pick_id = %s AND user_ip = %s",
                $pick_id,
                $user_ip
            )
        );
        
        if ($existing_vote) {
            return array(
                'success' => false,
                'message' => 'You have already voted on this pick'
            );
        }
        
        // Add the vote
        $vote_result = $wpdb->insert(
            $this->votes_table,
            array(
                'pick_id' => $pick_id,
                'vote_type' => $vote_type,
                'user_id' => get_current_user_id(),
                'user_ip' => $user_ip,
                'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? ''
            ),
            array('%s', '%s', '%d', '%s', '%s')
        );
        
        if ($vote_result === false) {
            return array(
                'success' => false,
                'message' => 'Failed to record vote'
            );
        }
        
        // Update pick vote counts
        $column = $vote_type === 'hot' ? 'hot_votes' : 'not_votes';
        $wpdb->query(
            $wpdb->prepare(
                "UPDATE {$this->picks_table} SET {$column} = {$column} + 1 WHERE pick_id = %s",
                $pick_id
            )
        );
        
        // Get updated pick data
        $updated_pick = $this->get_pick($pick_id);
        
        return array(
            'success' => true,
            'message' => 'Vote recorded successfully',
            'pick' => $updated_pick
        );
    }
    
    public function get_vote_stats($pick_id) {
        global $wpdb;
        
        $pick = $this->get_pick($pick_id);
        
        if (!$pick) {
            return false;
        }
        
        $total_votes = $pick['hot_votes'] + $pick['not_votes'];
        $hot_percentage = $total_votes > 0 ? round(($pick['hot_votes'] / $total_votes) * 100, 1) : 0;
        
        return array(
            'pick_id' => $pick_id,
            'hot_votes' => (int)$pick['hot_votes'],
            'not_votes' => (int)$pick['not_votes'],
            'total_votes' => $total_votes,
            'hot_percentage' => $hot_percentage
        );
    }
    
    public function cleanup_test_data() {
        global $wpdb;
        
        // Remove test picks
        $test_deleted = $wpdb->query(
            "DELETE FROM {$this->picks_table} 
             WHERE matchup LIKE '%Test Team%' 
             OR matchup LIKE '%Lakers vs Warriors%'
             OR matchup LIKE '%Celtics vs Bucks%'"
        );
        
        // Remove orphaned votes
        $votes_deleted = $wpdb->query(
            "DELETE v FROM {$this->votes_table} v
             LEFT JOIN {$this->picks_table} p ON v.pick_id = p.pick_id
             WHERE p.pick_id IS NULL"
        );
        
        // Remove duplicate picks
        $duplicates_deleted = $wpdb->query(
            "DELETE p1 FROM {$this->picks_table} p1
             INNER JOIN {$this->picks_table} p2 
             WHERE p1.id > p2.id 
             AND p1.matchup = p2.matchup 
             AND p1.pick_text = p2.pick_text 
             AND p1.sport = p2.sport"
        );
        
        return array(
            'test_picks_deleted' => $test_deleted,
            'votes_deleted' => $votes_deleted,
            'duplicate_picks_deleted' => $duplicates_deleted
        );
    }
    
    public function get_analytics() {
        global $wpdb;
        
        $total_picks = $wpdb->get_var("SELECT COUNT(*) FROM {$this->picks_table}");
        $active_picks = $wpdb->get_var("SELECT COUNT(*) FROM {$this->picks_table} WHERE is_active = 1");
        $total_votes = $wpdb->get_var("SELECT COUNT(*) FROM {$this->votes_table}");
        
        $top_picks = $wpdb->get_results(
            "SELECT * FROM {$this->picks_table} 
             ORDER BY (hot_votes + not_votes) DESC 
             LIMIT 10",
            ARRAY_A
        );
        
        return array(
            'total_picks' => (int)$total_picks,
            'active_picks' => (int)$active_picks,
            'total_votes' => (int)$total_votes,
            'top_picks' => $top_picks
        );
    }
    
    private function generate_pick_id() {
        return 'pick_' . wp_generate_uuid4();
    }
    
    public function get_picks_table() {
        return $this->picks_table;
    }
    
    public function get_votes_table() {
        return $this->votes_table;
    }
}