<?php

if (!defined('ABSPATH')) {
    exit;
}

class HON_Odds_API {
    
    private $api_key;
    private $base_url = 'https://api.the-odds-api.com/v4';
    private $sports_mapping;
    
    public function __construct() {
        $settings = get_option('hon_settings', array());
        $this->api_key = $settings['odds_api_key'] ?? '';
        
        $this->sports_mapping = array(
            'NBA' => 'basketball_nba',
            'NFL' => 'americanfootball_nfl',
            'MLB' => 'baseball_mlb',
            'NHL' => 'icehockey_nhl',
            'MLS' => 'soccer_usa_mls',
            'Tennis' => 'tennis_atp_french_open'
        );
    }
    
    public function test_connection() {
        if (empty($this->api_key)) {
            return array(
                'status' => 'error',
                'message' => 'No API key configured'
            );
        }
        
        $sports = $this->get_sports();
        
        if (is_wp_error($sports)) {
            return array(
                'status' => 'error',
                'message' => $sports->get_error_message()
            );
        }
        
        return array(
            'status' => 'success',
            'sports_count' => count($sports),
            'sample_sports' => array_slice($sports, 0, 5)
        );
    }
    
    public function get_sports() {
        $url = $this->base_url . '/sports';
        $args = array(
            'timeout' => 30,
            'headers' => array(
                'User-Agent' => 'WordPress/HotOrNotPlugin'
            )
        );
        
        $url = add_query_arg('apiKey', $this->api_key, $url);
        
        $response = wp_remote_get($url, $args);
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            return new WP_Error('json_error', 'Invalid JSON response');
        }
        
        return $data;
    }
    
    public function get_odds($sport_key, $markets = 'h2h,spreads,totals') {
        $url = $this->base_url . '/sports/' . $sport_key . '/odds';
        $args = array(
            'timeout' => 30,
            'headers' => array(
                'User-Agent' => 'WordPress/HotOrNotPlugin'
            )
        );
        
        $query_args = array(
            'apiKey' => $this->api_key,
            'regions' => 'us',
            'markets' => $markets,
            'oddsFormat' => 'american'
        );
        
        $url = add_query_arg($query_args, $url);
        
        $response = wp_remote_get($url, $args);
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            return new WP_Error('json_error', 'Invalid JSON response');
        }
        
        return $data;
    }
    
    public function get_todays_games() {
        $all_games = array();
        
        foreach ($this->sports_mapping as $sport_display => $sport_key) {
            $games = $this->get_odds($sport_key);
            
            if (is_wp_error($games)) {
                error_log('HON Plugin: Error fetching games for ' . $sport_display . ': ' . $games->get_error_message());
                continue;
            }
            
            foreach ($games as $game) {
                $game_time = strtotime($game['commence_time']);
                $now = current_time('timestamp');
                $time_diff = $game_time - $now;
                
                // Include games happening within the next 7 days and not more than 2 hours past
                if ($time_diff >= -7200 && $time_diff <= 604800) { // -2 hours to +7 days
                    $processed_game = $this->process_game_data($game, $sport_display);
                    if ($processed_game) {
                        $all_games[] = $processed_game;
                    }
                }
            }
        }
        
        // Sort by game time
        usort($all_games, function($a, $b) {
            return strtotime($a['commence_time']) - strtotime($b['commence_time']);
        });
        
        return $all_games;
    }
    
    private function process_game_data($game, $sport) {
        $best_odds = $this->extract_best_odds($game['bookmakers'] ?? array());
        
        return array(
            'id' => $game['id'],
            'sport' => $sport,
            'home_team' => $game['home_team'],
            'away_team' => $game['away_team'],
            'matchup' => $game['away_team'] . ' vs ' . $game['home_team'],
            'commence_time' => $game['commence_time'],
            'odds' => $best_odds
        );
    }
    
    private function extract_best_odds($bookmakers) {
        if (empty($bookmakers)) {
            return array();
        }
        
        // Prefer DraftKings, FanDuel, or first available
        $preferred_books = array('draftkings', 'fanduel', 'williamhill_us');
        $bookmaker = null;
        
        foreach ($preferred_books as $preferred) {
            foreach ($bookmakers as $book) {
                if ($book['key'] === $preferred) {
                    $bookmaker = $book;
                    break 2;
                }
            }
        }
        
        if (!$bookmaker) {
            $bookmaker = $bookmakers[0];
        }
        
        $markets = $bookmaker['markets'] ?? array();
        $odds_data = array();
        
        // Extract moneyline odds
        foreach ($markets as $market) {
            if ($market['key'] === 'h2h' && !empty($market['outcomes'])) {
                $outcomes = $market['outcomes'];
                if (count($outcomes) >= 2) {
                    $odds_data['moneyline'] = array();
                    foreach ($outcomes as $outcome) {
                        $odds_data['moneyline'][$outcome['name']] = $outcome['price'];
                    }
                }
            }
            
            // Extract spread odds
            if ($market['key'] === 'spreads' && !empty($market['outcomes'])) {
                $outcomes = $market['outcomes'];
                if (count($outcomes) >= 1) {
                    $outcome = $outcomes[0];
                    $odds_data['spread'] = array(
                        'line' => $outcome['point'] ?? 0,
                        'odds' => $outcome['price']
                    );
                }
            }
            
            // Extract totals odds
            if ($market['key'] === 'totals' && !empty($market['outcomes'])) {
                $outcomes = $market['outcomes'];
                $over_outcome = null;
                $under_outcome = null;
                
                foreach ($outcomes as $outcome) {
                    if ($outcome['name'] === 'Over') {
                        $over_outcome = $outcome;
                    } elseif ($outcome['name'] === 'Under') {
                        $under_outcome = $outcome;
                    }
                }
                
                if ($over_outcome) {
                    $odds_data['total'] = array(
                        'line' => $over_outcome['point'] ?? 0,
                        'over_odds' => $over_outcome['price'],
                        'under_odds' => $under_outcome['price'] ?? null
                    );
                }
            }
        }
        
        return $odds_data;
    }
    
    public function generate_pick_from_odds($game, $pick_type) {
        $odds = $game['odds'] ?? array();
        
        if ($pick_type === 'spread' && isset($odds['spread'])) {
            $line = $odds['spread']['line'];
            $team = $line > 0 ? $game['away_team'] : $game['home_team'];
            $spread_line = abs($line);
            $pick_text = $team . ' ' . ($line > 0 ? '+' : '-') . $spread_line;
            $odds_text = ($odds['spread']['odds'] > 0 ? '+' : '') . $odds['spread']['odds'];
            
        } elseif ($pick_type === 'total' && isset($odds['total'])) {
            $line = $odds['total']['line'];
            $is_over = wp_rand(0, 1); // Randomly choose over or under
            $pick_text = ($is_over ? 'Over ' : 'Under ') . $line;
            $odds_value = $is_over ? $odds['total']['over_odds'] : $odds['total']['under_odds'];
            $odds_text = ($odds_value > 0 ? '+' : '') . $odds_value;
            
        } elseif ($pick_type === 'moneyline' && isset($odds['moneyline'])) {
            // Choose the underdog for more interesting picks
            $ml_odds = $odds['moneyline'];
            $underdog = '';
            $highest_odds = -1000;
            
            foreach ($ml_odds as $team => $price) {
                if ($price > $highest_odds) {
                    $highest_odds = $price;
                    $underdog = $team;
                }
            }
            
            $pick_text = $underdog . ' ML';
            $odds_text = ($highest_odds > 0 ? '+' : '') . $highest_odds;
            
        } else {
            // Fallback
            $pick_text = $game['away_team'] . ' to cover';
            $odds_text = 'Even';
        }
        
        return array($pick_text, $odds_text);
    }
    
    public function get_sports_mapping() {
        return $this->sports_mapping;
    }
}