<?php

if (!defined('ABSPATH')) {
    exit;
}

class HON_OpenAI_API {
    
    private $api_key;
    private $base_url = 'https://api.openai.com/v1';
    
    public function __construct() {
        $settings = get_option('hon_settings', array());
        $this->api_key = $settings['openai_api_key'] ?? '';
    }
    
    public function test_connection() {
        if (empty($this->api_key)) {
            return array(
                'status' => 'error',
                'message' => 'No API key configured'
            );
        }
        
        $insight = $this->generate_insight('Lakers vs Warriors', 'Lakers +3.5', 'NBA');
        
        if (is_wp_error($insight)) {
            return array(
                'status' => 'error',
                'message' => $insight->get_error_message()
            );
        }
        
        return array(
            'status' => 'success',
            'sample_insight' => $insight
        );
    }
    
    public function generate_insight($matchup, $pick_text, $sport) {
        if (empty($this->api_key)) {
            return 'Sharp money trending this way. Don\'t miss the action!';
        }
        
        $prompt = $this->build_prompt($matchup, $pick_text, $sport);
        
        $response = $this->make_api_request($prompt);
        
        if (is_wp_error($response)) {
            // Return a fallback insight if API fails
            return $this->get_fallback_insight($sport);
        }
        
        return trim($response);
    }
    
    private function build_prompt($matchup, $pick_text, $sport) {
        return sprintf(
            'Write a compelling 1-2 sentence betting insight for: %s in %s (%s). Include relevant trends, team performance, or betting angles. Keep it under 120 characters and make it engaging for sports bettors. Do not use quotation marks.',
            $pick_text,
            $matchup,
            $sport
        );
    }
    
    private function make_api_request($prompt) {
        $url = $this->base_url . '/chat/completions';
        
        $body = array(
            'model' => 'gpt-4',
            'messages' => array(
                array(
                    'role' => 'system',
                    'content' => 'You are an expert sports betting analyst. Generate concise, engaging betting insights that highlight key trends, stats, or angles. Keep responses under 120 characters.'
                ),
                array(
                    'role' => 'user',
                    'content' => $prompt
                )
            ),
            'max_tokens' => 100,
            'temperature' => 0.7
        );
        
        $args = array(
            'timeout' => 30,
            'headers' => array(
                'Authorization' => 'Bearer ' . $this->api_key,
                'Content-Type' => 'application/json',
                'User-Agent' => 'WordPress/HotOrNotPlugin'
            ),
            'body' => wp_json_encode($body)
        );
        
        $response = wp_remote_post($url, $args);
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code !== 200) {
            return new WP_Error(
                'api_error',
                'OpenAI API returned status code: ' . $status_code
            );
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            return new WP_Error('json_error', 'Invalid JSON response from OpenAI');
        }
        
        if (!isset($data['choices'][0]['message']['content'])) {
            return new WP_Error('no_content', 'No content in OpenAI response');
        }
        
        return $data['choices'][0]['message']['content'];
    }
    
    private function get_fallback_insight($sport) {
        $fallbacks = array(
            'NBA' => array(
                'Strong offensive matchup favors this play tonight.',
                'Road team has covered 4 of last 5 in this spot.',
                'Key player trends point to value here.',
                'Historical data supports this betting angle.',
                'Sharp money movement indicates solid value.'
            ),
            'NFL' => array(
                'Weather conditions favor the under in this matchup.',
                'Home field advantage critical in this spot.',
                'Divisional rivalry games often go under the total.',
                'Key injuries create value on this line.',
                'Public betting heavily one way, fade the crowd.'
            ),
            'MLB' => array(
                'Starting pitcher matchup heavily favors this side.',
                'Wind blowing out at 15mph, take the over.',
                'Bullpen fatigue creates late-inning opportunities.',
                'Day games after night games offer value.',
                'Team hitting .320 vs lefties last 10 games.'
            ),
            'NHL' => array(
                'Back-to-back games favor the fresh goalie.',
                'Power play efficiency edge in this matchup.',
                'Road team desperate for points in standings.',
                'Goalie matchup heavily favors one side.',
                'Special teams advantage creates value here.'
            ),
            'MLS' => array(
                'International break affects team chemistry.',
                'Home field advantage crucial in MLS play.',
                'Key players returning from injury tonight.',
                'Weather conditions favor possession play.',
                'Playoff positioning creates extra motivation.'
            ),
            'Tennis' => array(
                'Surface advantage goes to the better player.',
                'Recent form heavily favors this selection.',
                'Head-to-head record supports this pick.',
                'Physical condition edge after long season.',
                'Playing style matchup creates clear value.'
            )
        );
        
        $sport_fallbacks = $fallbacks[$sport] ?? $fallbacks['NBA'];
        return $sport_fallbacks[array_rand($sport_fallbacks)];
    }
    
    public function generate_bulk_insights($picks) {
        $insights = array();
        
        foreach ($picks as $pick) {
            $insight = $this->generate_insight(
                $pick['matchup'],
                $pick['pick_text'],
                $pick['sport']
            );
            
            $insights[$pick['pick_id']] = $insight;
            
            // Add a small delay to avoid rate limiting
            usleep(100000); // 0.1 second delay
        }
        
        return $insights;
    }
}