<?php

if (!defined('ABSPATH')) {
    exit;
}

class HON_Pick_Manager {
    
    private $database;
    private $odds_api;
    private $openai_api;
    
    public function __construct() {
        $this->database = new HON_Database();
        $this->odds_api = new HON_Odds_API();
        $this->openai_api = new HON_OpenAI_API();
    }
    
    public function auto_generate_picks() {
        // Clean up old data first
        $this->cleanup_test_data();
        
        // Get real games from The Odds API
        $games = $this->odds_api->get_todays_games();
        
        if (empty($games)) {
            return array(
                'success' => false,
                'message' => 'No real games available today',
                'picks_created' => 0
            );
        }
        
        $picks_created = 0;
        $used_matchups = array();
        
        // Create picks for the best real games
        foreach ($games as $game) {
            if ($picks_created >= 6) { // Limit to 6 picks
                break;
            }
            
            // Skip if we already have a pick for this matchup
            $matchup_key = $game['sport'] . '_' . $game['matchup'];
            if (in_array($matchup_key, $used_matchups)) {
                continue;
            }
            
            // Check if we already have a pick for this exact matchup
            $existing_pick = $this->database->get_pick_by_matchup(
                $game['matchup'],
                $game['sport']
            );
            
            if ($existing_pick) {
                continue;
            }
            
            // Determine best pick type based on available odds
            $pick_types = $this->get_available_pick_types($game);
            
            if (empty($pick_types)) {
                continue;
            }
            
            // Choose the most interesting pick type
            $pick_type = $pick_types[0];
            
            list($pick_text, $odds_text) = $this->odds_api->generate_pick_from_odds($game, $pick_type);
            
            // Generate AI insight
            $insight = $this->openai_api->generate_insight(
                $game['matchup'],
                $pick_text,
                $game['sport']
            );
            
            // Convert UTC time to WordPress timezone
            $game_time = get_date_from_gmt($game['commence_time'], 'Y-m-d H:i:s');
            
            $pick_data = array(
                'matchup' => $game['matchup'],
                'team_1' => $game['away_team'],
                'team_2' => $game['home_team'],
                'pick_text' => $pick_text,
                'pick_type' => $pick_type,
                'odds' => $odds_text,
                'sport' => $game['sport'],
                'game_time' => $game_time,
                'insight' => $insight,
                'hot_votes' => 0,
                'not_votes' => 0,
                'is_active' => 1
            );
            
            $pick_id = $this->database->insert_pick($pick_data);
            
            if ($pick_id) {
                $used_matchups[] = $matchup_key;
                $picks_created++;
            }
        }
        
        return array(
            'success' => true,
            'message' => "Created {$picks_created} picks from real games",
            'picks_created' => $picks_created
        );
    }
    
    private function get_available_pick_types($game) {
        $pick_types = array();
        $odds = $game['odds'] ?? array();
        
        if (isset($odds['spread'])) {
            $pick_types[] = 'spread';
        }
        if (isset($odds['total'])) {
            $pick_types[] = 'total';
        }
        if (isset($odds['moneyline'])) {
            $pick_types[] = 'moneyline';
        }
        
        return $pick_types;
    }
    
    public function add_vote($pick_id, $vote_type) {
        $user_ip = $this->get_user_ip();
        return $this->database->add_vote($pick_id, $vote_type, $user_ip);
    }
    
    public function cleanup_test_data() {
        return $this->database->cleanup_test_data();
    }
    
    public function get_picks($limit = 20, $sport = '') {
        return $this->database->get_picks($limit, $sport);
    }
    
    public function get_pick($pick_id) {
        return $this->database->get_pick($pick_id);
    }
    
    public function get_vote_stats($pick_id) {
        return $this->database->get_vote_stats($pick_id);
    }
    
    public function get_analytics() {
        return $this->database->get_analytics();
    }
    
    public function create_manual_pick($data) {
        // Generate AI insight
        $insight = $this->openai_api->generate_insight(
            $data['matchup'],
            $data['pick_text'],
            $data['sport']
        );
        
        $data['insight'] = $insight;
        $data['hot_votes'] = 0;
        $data['not_votes'] = 0;
        $data['is_active'] = 1;
        
        return $this->database->insert_pick($data);
    }
    
    public function update_pick($pick_id, $data) {
        return $this->database->update_pick($pick_id, $data);
    }
    
    public function delete_pick($pick_id) {
        return $this->database->delete_pick($pick_id);
    }
    
    public function toggle_pick_status($pick_id) {
        $pick = $this->database->get_pick($pick_id);
        
        if (!$pick) {
            return false;
        }
        
        $new_status = $pick['is_active'] ? 0 : 1;
        
        return $this->database->update_pick($pick_id, array(
            'is_active' => $new_status
        ));
    }
    
    public function update_pick_result($pick_id, $result) {
        if (!in_array($result, array('win', 'loss', 'push'))) {
            return false;
        }
        
        return $this->database->update_pick($pick_id, array(
            'result' => $result
        ));
    }
    
    public function get_picks_by_sport($sport, $limit = 20) {
        return $this->database->get_picks($limit, $sport);
    }
    
    public function get_recent_votes($limit = 50) {
        global $wpdb;
        $votes_table = $this->database->get_votes_table();
        $picks_table = $this->database->get_picks_table();
        
        return $wpdb->get_results(
            $wpdb->prepare(
                "SELECT v.*, p.matchup, p.sport, p.pick_text 
                 FROM {$votes_table} v
                 JOIN {$picks_table} p ON v.pick_id = p.pick_id
                 ORDER BY v.voted_at DESC
                 LIMIT %d",
                $limit
            ),
            ARRAY_A
        );
    }
    
    public function get_sports_stats() {
        global $wpdb;
        $picks_table = $this->database->get_picks_table();
        
        return $wpdb->get_results(
            "SELECT sport, 
                    COUNT(*) as total_picks,
                    SUM(hot_votes + not_votes) as total_votes,
                    AVG(hot_votes / NULLIF(hot_votes + not_votes, 0)) * 100 as avg_hot_percentage
             FROM {$picks_table}
             WHERE is_active = 1
             GROUP BY sport
             ORDER BY total_picks DESC",
            ARRAY_A
        );
    }
    
    private function get_user_ip() {
        // Get user IP address
        $ip = '';
        
        if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
            $ip = $_SERVER['HTTP_CLIENT_IP'];
        } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            $ip = $_SERVER['HTTP_X_FORWARDED_FOR'];
        } else {
            $ip = $_SERVER['REMOTE_ADDR'] ?? '';
        }
        
        // Validate IP
        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
            return $ip;
        }
        
        return $_SERVER['REMOTE_ADDR'] ?? 'unknown';
    }
    
    public function regenerate_insight($pick_id) {
        $pick = $this->database->get_pick($pick_id);
        
        if (!$pick) {
            return false;
        }
        
        $new_insight = $this->openai_api->generate_insight(
            $pick['matchup'],
            $pick['pick_text'],
            $pick['sport']
        );
        
        return $this->database->update_pick($pick_id, array(
            'insight' => $new_insight
        ));
    }
}