<?php

if (!defined('ABSPATH')) {
    exit;
}

// Frontend utility functions

function hon_get_user_vote($pick_id, $user_ip = null) {
    global $wpdb;
    
    if (!$user_ip) {
        $user_ip = hon_get_client_ip();
    }
    
    $database = new HON_Database();
    $votes_table = $database->get_votes_table();
    
    return $wpdb->get_var(
        $wpdb->prepare(
            "SELECT vote_type FROM {$votes_table} WHERE pick_id = %s AND user_ip = %s",
            $pick_id,
            $user_ip
        )
    );
}

function hon_get_client_ip() {
    $ip = '';
    
    if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
        $ip = $_SERVER['HTTP_CLIENT_IP'];
    } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
        $ip = $_SERVER['HTTP_X_FORWARDED_FOR'];
    } else {
        $ip = $_SERVER['REMOTE_ADDR'] ?? '';
    }
    
    // Validate IP
    if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
        return $ip;
    }
    
    return $_SERVER['REMOTE_ADDR'] ?? 'unknown';
}

function hon_format_odds($odds) {
    if (empty($odds)) {
        return '';
    }
    
    $odds = (int) $odds;
    return $odds > 0 ? '+' . $odds : (string) $odds;
}

function hon_get_sport_emoji($sport) {
    $emojis = array(
        'NBA' => '🏀',
        'NFL' => '🏈',
        'MLB' => '⚾',
        'NHL' => '🏒',
        'MLS' => '⚽',
        'Tennis' => '🎾'
    );
    
    return isset($emojis[$sport]) ? $emojis[$sport] : '🎯';
}

function hon_time_until_game($game_time) {
    $game_timestamp = strtotime($game_time);
    $now = current_time('timestamp');
    $diff = $game_timestamp - $now;
    
    if ($diff < 0) {
        return __('Game started', 'hot-or-not-betting-picks');
    }
    
    if ($diff < 3600) { // Less than 1 hour
        $minutes = floor($diff / 60);
        return sprintf(_n('%d minute', '%d minutes', $minutes, 'hot-or-not-betting-picks'), $minutes);
    }
    
    if ($diff < 86400) { // Less than 1 day
        $hours = floor($diff / 3600);
        return sprintf(_n('%d hour', '%d hours', $hours, 'hot-or-not-betting-picks'), $hours);
    }
    
    $days = floor($diff / 86400);
    return sprintf(_n('%d day', '%d days', $days, 'hot-or-not-betting-picks'), $days);
}

function hon_get_pick_status_class($pick) {
    $game_timestamp = strtotime($pick['game_time']);
    $now = current_time('timestamp');
    
    if ($game_timestamp < $now) {
        return 'hon-pick-past';
    }
    
    if ($game_timestamp - $now < 3600) { // Less than 1 hour
        return 'hon-pick-soon';
    }
    
    return 'hon-pick-upcoming';
}

function hon_get_vote_percentage_class($percentage) {
    if ($percentage >= 70) {
        return 'hon-very-hot';
    } elseif ($percentage >= 60) {
        return 'hon-hot';
    } elseif ($percentage >= 40) {
        return 'hon-neutral';
    } else {
        return 'hon-cold';
    }
}

function hon_truncate_text($text, $length = 100) {
    if (strlen($text) <= $length) {
        return $text;
    }
    
    return substr($text, 0, $length) . '...';
}

function hon_get_social_share_url($platform, $pick) {
    $base_urls = array(
        'twitter' => 'https://twitter.com/intent/tweet',
        'facebook' => 'https://www.facebook.com/sharer/sharer.php',
        'reddit' => 'https://www.reddit.com/submit'
    );
    
    if (!isset($base_urls[$platform])) {
        return '';
    }
    
    $text = sprintf(
        __('Check out this %s pick: %s - %s', 'hot-or-not-betting-picks'),
        $pick['sport'],
        $pick['matchup'],
        $pick['pick_text']
    );
    
    $url = get_permalink();
    
    switch ($platform) {
        case 'twitter':
            return $base_urls[$platform] . '?' . http_build_query(array(
                'text' => $text,
                'url' => $url
            ));
            
        case 'facebook':
            return $base_urls[$platform] . '?' . http_build_query(array(
                'u' => $url,
                'quote' => $text
            ));
            
        case 'reddit':
            return $base_urls[$platform] . '?' . http_build_query(array(
                'title' => $text,
                'url' => $url
            ));
    }
    
    return '';
}

// AJAX handler for refreshing picks
add_action('wp_ajax_hon_refresh_picks', 'hon_handle_refresh_picks');
add_action('wp_ajax_nopriv_hon_refresh_picks', 'hon_handle_refresh_picks');

function hon_handle_refresh_picks() {
    check_ajax_referer('hon_nonce', 'nonce');
    
    $pick_manager = new HON_Pick_Manager();
    $picks = $pick_manager->get_picks(6);
    
    // Generate HTML for picks
    ob_start();
    foreach ($picks as $pick) {
        // Include the pick template here
        // This would be the same as in picks-display.php
    }
    $html = ob_get_clean();
    
    wp_send_json_success(array(
        'html' => $html,
        'count' => count($picks)
    ));
}