<?php

if (!defined('ABSPATH')) {
    exit;
}

$pick_manager = new HON_Pick_Manager();
$settings = get_option('hon_settings', array());

// Set defaults
$limit = isset($atts['limit']) ? intval($atts['limit']) : 6;
$sport = isset($atts['sport']) ? sanitize_text_field($atts['sport']) : '';
$show_voting = isset($atts['show_voting']) ? $atts['show_voting'] === 'true' : ($settings['enable_voting'] ?? true);
$show_time = isset($atts['show_time']) ? $atts['show_time'] === 'true' : ($settings['show_game_times'] ?? true);
$show_insights = $settings['show_insights'] ?? true;

$picks = $pick_manager->get_picks($limit, $sport);

// Format game time function
function hon_format_game_time($game_time) {
    if (empty($game_time)) return '';
    
    $game_timestamp = strtotime($game_time);
    $now = current_time('timestamp');
    $today = strtotime('today', $now);
    $tomorrow = strtotime('tomorrow', $now);
    $game_day = strtotime('today', $game_timestamp);
    
    $time_string = date_i18n('g:i A', $game_timestamp);
    
    if ($game_day === $today) {
        return sprintf(__('Today %s ET', 'hot-or-not-betting-picks'), $time_string);
    } elseif ($game_day === $tomorrow) {
        return sprintf(__('Tomorrow %s ET', 'hot-or-not-betting-picks'), $time_string);
    } else {
        return date_i18n('D, M j, g:i A', $game_timestamp) . ' ET';
    }
}

?>

<div class="hon-picks-container" id="hon-picks-container">
    <?php if (empty($picks)): ?>
        <div class="hon-empty-state">
            <h3><?php _e('No picks available yet', 'hot-or-not-betting-picks'); ?></h3>
            <p><?php _e('Check back soon for today\'s hottest betting picks!', 'hot-or-not-betting-picks'); ?></p>
        </div>
    <?php else: ?>
        <div class="hon-picks-grid">
            <?php foreach ($picks as $pick): 
                $total_votes = $pick['hot_votes'] + $pick['not_votes'];
                $hot_percentage = $total_votes > 0 ? round(($pick['hot_votes'] / $total_votes) * 100, 1) : 0;
                $game_time_formatted = hon_format_game_time($pick['game_time']);
            ?>
                <div class="hon-pick-card" data-pick-id="<?php echo esc_attr($pick['pick_id']); ?>">
                    <div class="hon-pick-header">
                        <h3 class="hon-matchup"><?php echo esc_html($pick['matchup']); ?></h3>
                        <span class="hon-sport-badge"><?php echo esc_html($pick['sport']); ?></span>
                    </div>
                    
                    <?php if ($show_time && !empty($game_time_formatted)): ?>
                        <div class="hon-game-time">
                            <span class="hon-time-icon">🕒</span>
                            <?php echo esc_html($game_time_formatted); ?>
                        </div>
                    <?php endif; ?>
                    
                    <div class="hon-pick-details">
                        <div class="hon-pick-text"><?php echo esc_html($pick['pick_text']); ?></div>
                        <?php if (!empty($pick['odds'])): ?>
                            <div class="hon-odds"><?php echo esc_html($pick['odds']); ?></div>
                        <?php endif; ?>
                    </div>

                    <?php if ($show_insights && !empty($pick['insight'])): ?>
                        <div class="hon-insight">
                            <span class="hon-insight-icon">💡</span>
                            <?php echo esc_html($pick['insight']); ?>
                        </div>
                    <?php endif; ?>

                    <?php if ($show_voting): ?>
                        <div class="hon-voting-section">
                            <div class="hon-vote-buttons">
                                <button class="hon-hot-button" data-pick-id="<?php echo esc_attr($pick['pick_id']); ?>" data-vote-type="hot">
                                    🔥 <?php _e('Hot', 'hot-or-not-betting-picks'); ?> (<span class="hon-vote-count"><?php echo esc_html($pick['hot_votes']); ?></span>)
                                </button>
                                <button class="hon-not-button" data-pick-id="<?php echo esc_attr($pick['pick_id']); ?>" data-vote-type="not">
                                    ❌ <?php _e('Not', 'hot-or-not-betting-picks'); ?> (<span class="hon-vote-count"><?php echo esc_html($pick['not_votes']); ?></span>)
                                </button>
                            </div>
                            
                            <div class="hon-vote-stats">
                                <div class="hon-percentage"><span class="hon-hot-percentage"><?php echo esc_html($hot_percentage); ?></span>% <?php _e('Hot', 'hot-or-not-betting-picks'); ?></div>
                                <div class="hon-total-votes"><span class="hon-total-count"><?php echo esc_html($total_votes); ?></span> <?php _e('total votes', 'hot-or-not-betting-picks'); ?></div>
                            </div>
                        </div>
                    <?php endif; ?>

                    <div class="hon-cta-section">
                        <a href="#" class="hon-bet-button" target="_blank" rel="noopener">
                            <?php _e('Bet this on Caesars', 'hot-or-not-betting-picks'); ?> →
                        </a>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>
    <?php endif; ?>
</div>

<?php if ($show_voting): ?>
<script type="text/javascript">
document.addEventListener('DOMContentLoaded', function() {
    const voteButtons = document.querySelectorAll('.hon-hot-button, .hon-not-button');
    
    voteButtons.forEach(button => {
        button.addEventListener('click', function() {
            const pickId = this.getAttribute('data-pick-id');
            const voteType = this.getAttribute('data-vote-type');
            
            // Disable button during request
            this.disabled = true;
            
            // Send AJAX request
            const formData = new FormData();
            formData.append('action', 'hon_vote');
            formData.append('pick_id', pickId);
            formData.append('vote_type', voteType);
            formData.append('nonce', hon_ajax.nonce);
            
            fetch(hon_ajax.ajax_url, {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // Update vote counts
                    const card = document.querySelector(`[data-pick-id="${pickId}"]`);
                    const hotButton = card.querySelector('.hon-hot-button .hon-vote-count');
                    const notButton = card.querySelector('.hon-not-button .hon-vote-count');
                    const hotPercentage = card.querySelector('.hon-hot-percentage');
                    const totalCount = card.querySelector('.hon-total-count');
                    
                    if (data.data.pick) {
                        const pick = data.data.pick;
                        const totalVotes = parseInt(pick.hot_votes) + parseInt(pick.not_votes);
                        const hotPercent = totalVotes > 0 ? Math.round((pick.hot_votes / totalVotes) * 100 * 10) / 10 : 0;
                        
                        hotButton.textContent = pick.hot_votes;
                        notButton.textContent = pick.not_votes;
                        hotPercentage.textContent = hotPercent;
                        totalCount.textContent = totalVotes;
                        
                        // Disable all voting buttons for this pick
                        card.querySelectorAll('.hon-hot-button, .hon-not-button').forEach(btn => {
                            btn.disabled = true;
                            btn.style.opacity = '0.6';
                        });
                        
                        // Show success message
                        const message = document.createElement('div');
                        message.className = 'hon-vote-message';
                        message.textContent = '✅ Vote recorded!';
                        card.querySelector('.hon-voting-section').appendChild(message);
                        
                        setTimeout(() => {
                            if (message.parentNode) {
                                message.parentNode.removeChild(message);
                            }
                        }, 3000);
                    }
                } else {
                    alert(data.data || 'Error voting. Please try again.');
                    this.disabled = false;
                }
            })
            .catch(error => {
                console.error('Voting error:', error);
                alert('Error voting. Please try again.');
                this.disabled = false;
            });
        });
    });
});
</script>
<?php endif; ?>