<?php
/**
 * Admin Settings Page
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Add admin menu
 */
function lof_add_admin_menu() {
    add_menu_page(
        'Live Odds Feed Settings',
        'Live Odds Feed',
        'manage_options',
        'live-odds-feed-settings',
        'lof_settings_page',
        'dashicons-chart-line',
        30
    );
}
add_action('admin_menu', 'lof_add_admin_menu');

/**
 * Register settings
 */
function lof_register_settings() {
    register_setting('lof_settings_group', 'lof_api_key');
    register_setting('lof_settings_group', 'lof_cache_duration');
    register_setting('lof_settings_group', 'lof_sports');
}
add_action('admin_init', 'lof_register_settings');

/**
 * Enqueue admin styles
 */
function lof_admin_enqueue_scripts($hook) {
    if ($hook !== 'toplevel_page_live-odds-feed-settings') {
        return;
    }
    
    wp_enqueue_style('lof-admin-styles', LOF_PLUGIN_URL . 'assets/css/admin.css', array(), LOF_VERSION);
}
add_action('admin_enqueue_scripts', 'lof_admin_enqueue_scripts');

/**
 * Settings page content
 */
function lof_settings_page() {
    // Check user permissions
    if (!current_user_can('manage_options')) {
        return;
    }

    // Get current settings
    $api_key = get_option('lof_api_key', '');
    $cache_duration = get_option('lof_cache_duration', 300);
    $sports = get_option('lof_sports', array('americanfootball_nfl', 'basketball_nba', 'baseball_mlb', 'icehockey_nhl'));

    // Handle form submission
    if (isset($_POST['lof_settings_submit'])) {
        check_admin_referer('lof_settings_nonce');
        
        update_option('lof_api_key', sanitize_text_field($_POST['lof_api_key']));
        update_option('lof_cache_duration', intval($_POST['lof_cache_duration']));
        update_option('lof_sports', isset($_POST['lof_sports']) ? array_map('sanitize_text_field', $_POST['lof_sports']) : array());
        
        echo '<div class="notice notice-success is-dismissible"><p>Settings saved successfully!</p></div>';
        
        // Refresh values
        $api_key = get_option('lof_api_key');
        $cache_duration = get_option('lof_cache_duration');
        $sports = get_option('lof_sports');
    }

    // Test API connection
    $api_status = '';
    if (!empty($api_key) && isset($_POST['lof_test_api'])) {
        check_admin_referer('lof_settings_nonce');
        $test_result = lof_test_api_connection($api_key);
        if ($test_result['success']) {
            $api_status = '<div class="notice notice-success"><p>✓ API connection successful! Remaining requests: ' . esc_html($test_result['remaining']) . '</p></div>';
        } else {
            $api_status = '<div class="notice notice-error"><p>✗ API connection failed: ' . esc_html($test_result['message']) . '</p></div>';
        }
    }

    ?>
    <div class="wrap lof-admin-wrap">
        <h1>Live Odds Feed Settings</h1>
        
        <?php echo $api_status; ?>
        
        <form method="post" action="">
            <?php wp_nonce_field('lof_settings_nonce'); ?>
            
            <div class="lof-admin-section">
                <h2>The Odds API Configuration</h2>
                <p class="description">Get your free API key from <a href="https://the-odds-api.com" target="_blank">The Odds API</a></p>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="lof_api_key">API Key</label>
                        </th>
                        <td>
                            <input type="text" 
                                   id="lof_api_key" 
                                   name="lof_api_key" 
                                   value="<?php echo esc_attr($api_key); ?>" 
                                   class="regular-text" 
                                   placeholder="Enter your Odds API key"
                                   required />
                            <p class="description">Your unique API key from The Odds API</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="lof_cache_duration">Cache Duration (seconds)</label>
                        </th>
                        <td>
                            <input type="number" 
                                   id="lof_cache_duration" 
                                   name="lof_cache_duration" 
                                   value="<?php echo esc_attr($cache_duration); ?>" 
                                   min="30" 
                                   max="3600" 
                                   class="small-text" />
                            <p class="description">How long to cache odds data (default: 300 seconds / 5 minutes)</p>
                        </td>
                    </tr>
                </table>
            </div>
            
            <div class="lof-admin-section">
                <h2>Sports Selection</h2>
                <p class="description">Select which sports to display odds for</p>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">Sports</th>
                        <td>
                            <fieldset>
                                <label>
                                    <input type="checkbox" name="lof_sports[]" value="americanfootball_nfl" <?php checked(in_array('americanfootball_nfl', $sports)); ?> />
                                    NFL (American Football)
                                </label><br>
                                <label>
                                    <input type="checkbox" name="lof_sports[]" value="basketball_nba" <?php checked(in_array('basketball_nba', $sports)); ?> />
                                    NBA (Basketball)
                                </label><br>
                                <label>
                                    <input type="checkbox" name="lof_sports[]" value="baseball_mlb" <?php checked(in_array('baseball_mlb', $sports)); ?> />
                                    MLB (Baseball)
                                </label><br>
                                <label>
                                    <input type="checkbox" name="lof_sports[]" value="icehockey_nhl" <?php checked(in_array('icehockey_nhl', $sports)); ?> />
                                    NHL (Ice Hockey)
                                </label>
                            </fieldset>
                        </td>
                    </tr>
                </table>
            </div>
            
            <div class="lof-admin-section">
                <h2>Usage Instructions</h2>
                <div class="lof-shortcode-box">
                    <p><strong>Add the odds feed to any page or post using this shortcode:</strong></p>
                    <code class="lof-shortcode">[live_odds_feed]</code>
                    <p class="description">You can also specify a sport: <code>[live_odds_feed sport="basketball_nba"]</code></p>
                </div>
            </div>
            
            <p class="submit">
                <input type="submit" name="lof_settings_submit" class="button button-primary" value="Save Settings" />
                <?php if (!empty($api_key)): ?>
                    <input type="submit" name="lof_test_api" class="button button-secondary" value="Test API Connection" />
                <?php endif; ?>
            </p>
        </form>
        
        <div class="lof-admin-section">
            <h2>Current Status</h2>
            <table class="widefat">
                <tbody>
                    <tr>
                        <td><strong>API Key:</strong></td>
                        <td><?php echo !empty($api_key) ? '<span style="color: green;">✓ Configured</span>' : '<span style="color: red;">✗ Not configured</span>'; ?></td>
                    </tr>
                    <tr>
                        <td><strong>Plugin Version:</strong></td>
                        <td><?php echo LOF_VERSION; ?></td>
                    </tr>
                    <tr>
                        <td><strong>Cache Table:</strong></td>
                        <td><?php 
                            global $wpdb;
                            $table_name = $wpdb->prefix . 'live_odds_cache';
                            $count = $wpdb->get_var("SELECT COUNT(*) FROM $table_name");
                            echo $count . ' cached events';
                        ?></td>
                    </tr>
                </tbody>
            </table>
        </div>
    </div>
    <?php
}

/**
 * Test API connection
 */
function lof_test_api_connection($api_key) {
    $response = wp_remote_get('https://api.the-odds-api.com/v4/sports?apiKey=' . $api_key);
    
    if (is_wp_error($response)) {
        return array(
            'success' => false,
            'message' => $response->get_error_message()
        );
    }
    
    $body = wp_remote_retrieve_body($response);
    $data = json_decode($body, true);
    
    if (wp_remote_retrieve_response_code($response) === 200 && is_array($data)) {
        $remaining = wp_remote_retrieve_header($response, 'x-requests-remaining');
        return array(
            'success' => true,
            'remaining' => $remaining ? $remaining : 'Unknown'
        );
    }
    
    return array(
        'success' => false,
        'message' => 'Invalid API key or API error'
    );
}
