/**
 * Live Odds Feed JavaScript
 */

(function($) {
    'use strict';

    let currentSport = 'americanfootball_nfl';
    let isLoading = false;

    // Initialize on document ready
    $(document).ready(function() {
        const $container = $('#live-odds-feed');
        if ($container.length === 0) return;

        currentSport = $container.data('sport') || 'americanfootball_nfl';
        
        // Load initial odds
        loadOdds(currentSport);

        // Sport filter buttons
        $('.lof-sport-btn').on('click', function() {
            const sport = $(this).data('sport');
            $('.lof-sport-btn').removeClass('active');
            $(this).addClass('active');
            currentSport = sport;
            loadOdds(sport);
        });

        // Refresh button
        $('.lof-refresh-btn').on('click', function() {
            loadOdds(currentSport, true);
        });
    });

    /**
     * Load odds from API
     */
    function loadOdds(sport, forceRefresh = false) {
        if (isLoading) return;
        
        isLoading = true;
        const $grid = $('#lof-odds-grid');
        const $refreshBtn = $('.lof-refresh-btn');
        
        $refreshBtn.addClass('loading');
        
        // Show loading state
        $grid.html('<div class="lof-loading"><div class="lof-spinner"></div><p>Loading odds...</p></div>');

        const url = lofData.restUrl + 'odds/' + sport + (forceRefresh ? '?refresh=true' : '');

        $.ajax({
            url: url,
            method: 'GET',
            beforeSend: function(xhr) {
                xhr.setRequestHeader('X-WP-Nonce', lofData.nonce);
            },
            success: function(response) {
                if (response.success && response.odds) {
                    renderOdds(response.odds);
                    updateEventCount(response.count);
                } else {
                    showError('Failed to load odds');
                }
            },
            error: function(xhr) {
                let errorMsg = 'Failed to load odds';
                if (xhr.responseJSON && xhr.responseJSON.message) {
                    errorMsg = xhr.responseJSON.message;
                }
                showError(errorMsg);
            },
            complete: function() {
                isLoading = false;
                $refreshBtn.removeClass('loading');
            }
        });
    }

    /**
     * Render odds cards
     */
    function renderOdds(odds) {
        const $grid = $('#lof-odds-grid');
        
        if (odds.length === 0) {
            $grid.html('<div class="lof-no-odds"><p>No odds available for this sport</p></div>');
            return;
        }

        let html = '';
        odds.forEach(function(event) {
            html += renderOddsCard(event);
        });

        $grid.html(html);
    }

    /**
     * Render single odds card
     */
    function renderOddsCard(event) {
        let html = '<div class="lof-odds-card">';
        
        // Header
        html += '<div class="lof-card-header">';
        html += '<h3 class="lof-card-title">' + escapeHtml(event.away_team) + ' @ ' + escapeHtml(event.home_team) + '</h3>';
        html += '<div class="lof-card-meta">';
        html += '<span class="lof-card-time">' + formatDate(event.commence_time) + '</span>';
        html += '<span class="lof-card-badge">' + escapeHtml(event.sport_title) + '</span>';
        html += '</div>';
        html += '</div>';
        
        // Content
        html += '<div class="lof-card-content">';
        
        if (event.bookmakers && event.bookmakers.length > 0) {
            event.bookmakers.forEach(function(bookmaker) {
                html += '<div class="lof-bookmaker">';
                html += '<p class="lof-bookmaker-title">' + escapeHtml(bookmaker.title) + '</p>';
                html += '<div class="lof-markets">';
                
                if (bookmaker.markets) {
                    bookmaker.markets.forEach(function(market) {
                        html += '<div class="lof-market">';
                        html += '<p class="lof-market-label">' + getMarketLabel(market.key) + '</p>';
                        
                        if (market.outcomes) {
                            market.outcomes.forEach(function(outcome) {
                                html += '<div class="lof-outcome">';
                                html += '<span class="lof-outcome-name">' + truncate(escapeHtml(outcome.name), 12) + '</span>';
                                html += '<span class="lof-outcome-price">';
                                if (outcome.point) {
                                    html += (outcome.point > 0 ? '+' : '') + outcome.point + ' ';
                                }
                                html += formatOdds(outcome.price);
                                html += '</span>';
                                html += '</div>';
                            });
                        }
                        
                        html += '</div>';
                    });
                }
                
                html += '</div>';
                html += '</div>';
            });
        }
        
        html += '</div>';
        html += '</div>';
        
        return html;
    }

    /**
     * Format odds
     */
    function formatOdds(price) {
        return price > 0 ? '+' + price : price;
    }

    /**
     * Format date
     */
    function formatDate(dateString) {
        const date = new Date(dateString);
        return date.toLocaleString('en-US', {
            month: 'short',
            day: 'numeric',
            hour: 'numeric',
            minute: '2-digit',
            hour12: true
        });
    }

    /**
     * Get market label
     */
    function getMarketLabel(marketKey) {
        const labels = {
            'h2h': 'MONEYLINE',
            'spreads': 'SPREAD',
            'totals': 'TOTAL'
        };
        return labels[marketKey] || marketKey.toUpperCase();
    }

    /**
     * Update event count
     */
    function updateEventCount(count) {
        $('.lof-event-count').text(count + ' events');
    }

    /**
     * Show error
     */
    function showError(message) {
        const $grid = $('#lof-odds-grid');
        $grid.html('<div class="lof-no-odds"><p>' + escapeHtml(message) + '</p></div>');
    }

    /**
     * Escape HTML
     */
    function escapeHtml(text) {
        const map = {
            '&': '&amp;',
            '<': '&lt;',
            '>': '&gt;',
            '"': '&quot;',
            "'": '&#039;'
        };
        return String(text).replace(/[&<>"']/g, function(m) { return map[m]; });
    }

    /**
     * Truncate text
     */
    function truncate(text, maxLength) {
        if (text.length <= maxLength) return text;
        return text.substring(0, maxLength - 3) + '...';
    }

})(jQuery);
