<?php
/**
 * API Handler for The Odds API
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

class LOF_API_Handler {
    private $api_key;
    private $base_url = 'https://api.the-odds-api.com/v4';
    private $allowed_books = array('draftkings', 'fanduel', 'betmgm');
    
    public function __construct() {
        $this->api_key = get_option('lof_api_key', '');
    }
    
    /**
     * Get odds for a specific sport
     */
    public function get_odds($sport_key, $use_cache = true) {
        // Check cache first
        if ($use_cache) {
            $cached = $this->get_cached_odds($sport_key);
            if ($cached) {
                return $cached;
            }
        }
        
        if (empty($this->api_key)) {
            return array(
                'success' => false,
                'message' => 'API key not configured'
            );
        }
        
        $url = $this->base_url . '/sports/' . $sport_key . '/odds';
        $params = array(
            'apiKey' => $this->api_key,
            'regions' => 'us',
            'markets' => 'h2h,spreads,totals',
            'oddsFormat' => 'american'
        );
        
        $response = wp_remote_get(add_query_arg($params, $url), array(
            'timeout' => 30
        ));
        
        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'message' => $response->get_error_message()
            );
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if (wp_remote_retrieve_response_code($response) !== 200) {
            return array(
                'success' => false,
                'message' => 'API request failed',
                'data' => $data
            );
        }
        
        // Filter and format data
        $formatted_data = $this->format_odds_data($data);
        
        // Cache the results
        $this->cache_odds($sport_key, $formatted_data);
        
        return array(
            'success' => true,
            'odds' => $formatted_data,
            'count' => count($formatted_data),
            'remaining_requests' => wp_remote_retrieve_header($response, 'x-requests-remaining')
        );
    }
    
    /**
     * Format odds data
     */
    private function format_odds_data($data) {
        if (!is_array($data)) {
            return array();
        }
        
        $formatted = array();
        
        foreach ($data as $event) {
            // Filter bookmakers to only allowed ones
            $filtered_bookmakers = array();
            if (isset($event['bookmakers']) && is_array($event['bookmakers'])) {
                foreach ($event['bookmakers'] as $bookmaker) {
                    if (in_array($bookmaker['key'], $this->allowed_books)) {
                        $filtered_bookmakers[] = $bookmaker;
                    }
                }
            }
            
            // Only include event if it has bookmakers
            if (!empty($filtered_bookmakers)) {
                $formatted[] = array(
                    'id' => $event['id'],
                    'sport_key' => $event['sport_key'],
                    'sport_title' => $event['sport_title'],
                    'home_team' => $event['home_team'],
                    'away_team' => $event['away_team'],
                    'commence_time' => $event['commence_time'],
                    'bookmakers' => array_slice($filtered_bookmakers, 0, 3) // Limit to 3 bookmakers
                );
            }
        }
        
        // Sort by commence time (nearest first)
        usort($formatted, function($a, $b) {
            return strtotime($a['commence_time']) - strtotime($b['commence_time']);
        });
        
        return $formatted;
    }
    
    /**
     * Get cached odds
     */
    private function get_cached_odds($sport_key) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'live_odds_cache';
        $cache_duration = get_option('lof_cache_duration', 300);
        
        $query = $wpdb->prepare(
            "SELECT event_data FROM $table_name 
            WHERE sport_key = %s 
            AND cached_at > DATE_SUB(NOW(), INTERVAL %d SECOND)
            ORDER BY cached_at DESC
            LIMIT 1",
            $sport_key,
            $cache_duration
        );
        
        $cached = $wpdb->get_var($query);
        
        if ($cached) {
            return json_decode($cached, true);
        }
        
        return false;
    }
    
    /**
     * Cache odds data
     */
    private function cache_odds($sport_key, $data) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'live_odds_cache';
        
        // Clear old cache for this sport
        $wpdb->delete($table_name, array('sport_key' => $sport_key));
        
        // Insert new cache
        $wpdb->insert(
            $table_name,
            array(
                'sport_key' => $sport_key,
                'event_id' => 'bulk_' . time(),
                'event_data' => json_encode($data),
                'cached_at' => current_time('mysql')
            ),
            array('%s', '%s', '%s', '%s')
        );
    }
    
    /**
     * Clear all cache
     */
    public function clear_cache() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'live_odds_cache';
        $wpdb->query("TRUNCATE TABLE $table_name");
    }
    
    /**
     * Clean old cache entries
     */
    public function clean_old_cache() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'live_odds_cache';
        $wpdb->query(
            "DELETE FROM $table_name WHERE cached_at < DATE_SUB(NOW(), INTERVAL 1 HOUR)"
        );
    }
}
