<?php
/**
 * REST API Routes
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Register REST API routes
 */
function lof_register_rest_routes() {
    register_rest_route('live-odds-feed/v1', '/odds/(?P<sport>[a-z_]+)', array(
        'methods' => 'GET',
        'callback' => 'lof_get_odds_endpoint',
        'permission_callback' => '__return_true',
        'args' => array(
            'sport' => array(
                'required' => true,
                'validate_callback' => function($param) {
                    return in_array($param, array(
                        'americanfootball_nfl',
                        'basketball_nba',
                        'baseball_mlb',
                        'icehockey_nhl'
                    ));
                }
            )
        )
    ));
    
    register_rest_route('live-odds-feed/v1', '/clear-cache', array(
        'methods' => 'POST',
        'callback' => 'lof_clear_cache_endpoint',
        'permission_callback' => function() {
            return current_user_can('manage_options');
        }
    ));
}
add_action('rest_api_init', 'lof_register_rest_routes');

/**
 * Get odds endpoint
 */
function lof_get_odds_endpoint($request) {
    $sport = $request->get_param('sport');
    $force_refresh = $request->get_param('refresh') === 'true';
    
    $api_handler = new LOF_API_Handler();
    $result = $api_handler->get_odds($sport, !$force_refresh);
    
    if (!$result['success']) {
        return new WP_Error(
            'api_error',
            $result['message'],
            array('status' => 500)
        );
    }
    
    return new WP_REST_Response($result, 200);
}

/**
 * Clear cache endpoint
 */
function lof_clear_cache_endpoint() {
    $api_handler = new LOF_API_Handler();
    $api_handler->clear_cache();
    
    return new WP_REST_Response(array(
        'success' => true,
        'message' => 'Cache cleared successfully'
    ), 200);
}

/**
 * Schedule cache cleanup
 */
function lof_schedule_cache_cleanup() {
    if (!wp_next_scheduled('lof_clean_cache_hook')) {
        wp_schedule_event(time(), 'hourly', 'lof_clean_cache_hook');
    }
}
add_action('wp', 'lof_schedule_cache_cleanup');

/**
 * Clean cache hook
 */
function lof_clean_cache() {
    $api_handler = new LOF_API_Handler();
    $api_handler->clean_old_cache();
}
add_action('lof_clean_cache_hook', 'lof_clean_cache');
