<?php
/**
 * Plugin Name: Live Odds Feed
 * Plugin URI: https://gamblingSite.com
 * Description: Display live betting odds from The Odds API for NFL, NBA, MLB, and NHL games. Supports moneyline, spreads, and totals from DraftKings, FanDuel, and BetMGM.
 * Version: 1.0.0
 * Author: GamblingSite.com
 * Author URI: https://gamblingSite.com
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: live-odds-feed
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('LOF_VERSION', '1.0.0');
define('LOF_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('LOF_PLUGIN_URL', plugin_dir_url(__FILE__));
define('LOF_PLUGIN_BASENAME', plugin_basename(__FILE__));

// Include required files
require_once LOF_PLUGIN_DIR . 'includes/api-handler.php';
require_once LOF_PLUGIN_DIR . 'includes/rest-routes.php';
require_once LOF_PLUGIN_DIR . 'admin/settings-page.php';

/**
 * Activate plugin
 */
function lof_activate() {
    // Create custom database table for odds cache
    global $wpdb;
    $table_name = $wpdb->prefix . 'live_odds_cache';
    $charset_collate = $wpdb->get_charset_collate();

    $sql = "CREATE TABLE IF NOT EXISTS $table_name (
        id bigint(20) NOT NULL AUTO_INCREMENT,
        sport_key varchar(50) NOT NULL,
        event_id varchar(100) NOT NULL,
        event_data longtext NOT NULL,
        cached_at datetime DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY  (id),
        KEY sport_key (sport_key),
        KEY event_id (event_id)
    ) $charset_collate;";

    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql);

    // Set default options
    add_option('lof_api_key', '');
    add_option('lof_cache_duration', 300); // 5 minutes default
    add_option('lof_sports', array('americanfootball_nfl', 'basketball_nba', 'baseball_mlb', 'icehockey_nhl'));
}
register_activation_hook(__FILE__, 'lof_activate');

/**
 * Deactivate plugin
 */
function lof_deactivate() {
    // Clear scheduled cron if any
    wp_clear_scheduled_hook('lof_fetch_odds_cron');
}
register_deactivation_hook(__FILE__, 'lof_deactivate');

/**
 * Enqueue frontend assets
 */
function lof_enqueue_scripts() {
    wp_enqueue_style('lof-styles', LOF_PLUGIN_URL . 'assets/css/odds-feed.css', array(), LOF_VERSION);
    
    // Enqueue Google Fonts
    wp_enqueue_style('lof-fonts', 'https://fonts.googleapis.com/css2?family=Chivo:wght@300;400;700;900&family=Manrope:wght@400;500;600&family=JetBrains+Mono:wght@400;500;700&display=swap', array(), null);
    
    wp_enqueue_script('lof-script', LOF_PLUGIN_URL . 'assets/js/odds-feed.js', array('jquery'), LOF_VERSION, true);
    
    // Pass REST API URL to JavaScript
    wp_localize_script('lof-script', 'lofData', array(
        'restUrl' => rest_url('live-odds-feed/v1/'),
        'nonce' => wp_create_nonce('wp_rest')
    ));
}
add_action('wp_enqueue_scripts', 'lof_enqueue_scripts');

/**
 * Register shortcode
 */
function lof_shortcode($atts) {
    $atts = shortcode_atts(array(
        'sport' => 'americanfootball_nfl',
        'limit' => '20'
    ), $atts);

    ob_start();
    ?>
    <div id="live-odds-feed" data-sport="<?php echo esc_attr($atts['sport']); ?>" data-limit="<?php echo esc_attr($atts['limit']); ?>">
        <div class="lof-header">
            <div class="lof-title-section">
                <h1 class="lof-title">LIVE ODDS FEED</h1>
                <p class="lof-subtitle">Real-time betting odds from multiple bookmakers</p>
            </div>
        </div>
        
        <div class="lof-sport-filters">
            <button class="lof-sport-btn active" data-sport="americanfootball_nfl">NFL</button>
            <button class="lof-sport-btn" data-sport="basketball_nba">NBA</button>
            <button class="lof-sport-btn" data-sport="baseball_mlb">MLB</button>
            <button class="lof-sport-btn" data-sport="icehockey_nhl">NHL</button>
        </div>
        
        <div class="lof-controls">
            <div class="lof-live-badge">
                <span class="lof-pulse"></span>
                LIVE
                <span class="lof-event-count">0 events</span>
            </div>
            <button class="lof-refresh-btn">
                <span class="lof-refresh-icon">↻</span> Refresh
            </button>
        </div>
        
        <div id="lof-odds-grid" class="lof-odds-grid">
            <div class="lof-loading">
                <div class="lof-spinner"></div>
                <p>Loading odds...</p>
            </div>
        </div>
    </div>
    <?php
    return ob_get_clean();
}
add_shortcode('live_odds_feed', 'lof_shortcode');

/**
 * Add settings link on plugin page
 */
function lof_add_settings_link($links) {
    $settings_link = '<a href="admin.php?page=live-odds-feed-settings">Settings</a>';
    array_unshift($links, $settings_link);
    return $links;
}
add_filter('plugin_action_links_' . LOF_PLUGIN_BASENAME, 'lof_add_settings_link');
