(function(){
  function isHalfPoint(x){
    // True if it ends in .5 (within float tolerance)
    return Math.abs((x * 2) - Math.round(x * 2)) < 1e-9 && (Math.round(x * 2) % 2 === 1);
  }

  function clamp(n, min, max){ return Math.max(min, Math.min(max, n)); }

  function setState(box, state){
    box.classList.remove("gs-ou-win","gs-ou-lose","gs-ou-push");
    box.classList.add(state);
  }

  function init(root){
    const mode = (root.dataset.mode === "score") ? "score" : "line";

    const min   = parseFloat(root.dataset.min);
    const max   = parseFloat(root.dataset.max);
    const start = parseFloat(root.dataset.start);
    const step  = parseFloat(root.dataset.step);
    const home0 = parseFloat(root.dataset.home);
    const away0 = parseFloat(root.dataset.away);

    const scoreMin = parseFloat(root.dataset.scoreMin || min);
    const scoreMax = parseFloat(root.dataset.scoreMax || max);

    const label = root.dataset.label || "Mock Game";

    const range = root.querySelector(".gs-ou-range");
    const badgeLabel = root.querySelector(".gs-ou-badge-label");
    const badgeValue = root.querySelector(".gs-ou-badge-value");

    const elLabel = root.querySelector(".gs-ou-label");
    const elHome = root.querySelector(".gs-ou-home");
    const elAway = root.querySelector(".gs-ou-away");
    const elTotal = root.querySelector(".gs-ou-total");

    const elMin = root.querySelector(".gs-ou-min");
    const elMax = root.querySelector(".gs-ou-max");

    const sliderLabel = root.querySelector(".gs-ou-slider-label");

    const overBox = root.querySelector(".gs-ou-over");
    const underBox = root.querySelector(".gs-ou-under");

    const overStatus = root.querySelector(".gs-ou-over-status");
    const underStatus = root.querySelector(".gs-ou-under-status");

    const overExplain = root.querySelector(".gs-ou-over-explain");
    const underExplain = root.querySelector(".gs-ou-under-explain");

    elLabel.textContent = label;

    // Keep an initial score differential for mode="score"
    const diff = home0 - away0;

    function renderScore(home, away){
      const totalPoints = home + away;
      elHome.textContent = String(Math.round(home));
      elAway.textContent = String(Math.round(away));
      elTotal.textContent = String(Math.round(totalPoints));
      return totalPoints;
    }

    function evaluate(line, totalPoints){
      // Push only possible on whole-number lines
      const half = isHalfPoint(line);
      const diffLine = totalPoints - line;

      if (Math.abs(diffLine) < 1e-9 && !half){
        setState(overBox, "gs-ou-push");
        setState(underBox, "gs-ou-push");
        overStatus.textContent = "⏸ Push";
        underStatus.textContent = "⏸ Push";
        overExplain.textContent = `Total (${Math.round(totalPoints)}) equals the line (${line.toFixed(1)}). Over is a push.`;
        underExplain.textContent = `Total (${Math.round(totalPoints)}) equals the line (${line.toFixed(1)}). Under is a push.`;
        return;
      }

      if (totalPoints > line){
        setState(overBox, "gs-ou-win");
        setState(underBox, "gs-ou-lose");
        overStatus.textContent = "✅ Win";
        underStatus.textContent = "❌ Loss";
        overExplain.textContent = `Combined points (${Math.round(totalPoints)}) are above ${line.toFixed(1)}.`;
        underExplain.textContent = `Combined points (${Math.round(totalPoints)}) are not below ${line.toFixed(1)}.`;
      } else {
        setState(overBox, "gs-ou-lose");
        setState(underBox, "gs-ou-win");
        overStatus.textContent = "❌ Loss";
        underStatus.textContent = "✅ Win";
        overExplain.textContent = `Combined points (${Math.round(totalPoints)}) are not above ${line.toFixed(1)}.`;
        underExplain.textContent = `Combined points (${Math.round(totalPoints)}) are below ${line.toFixed(1)}.`;
      }
    }

    function updateLineMode(){
      const line = parseFloat(range.value);
      badgeLabel.textContent = "Total Line:";
      badgeValue.textContent = " " + line.toFixed(1);

      const totalPoints = renderScore(home0, away0);
      evaluate(line, totalPoints);
    }

    function updateScoreMode(){
      // line fixed at "start"
      const line = start;

      // Slider controls combined total. Preserve diff.
      const total = parseFloat(range.value);

      // Convert total + diff into home/away. Keep integers.
      // home = (total + diff)/2; away = total - home
      let home = (total + diff) / 2;
      let away = total - home;

      // Round to integers and re-balance to keep exact total.
      home = Math.round(home);
      away = Math.round(away);

      // Fix rounding drift
      const drift = (home + away) - Math.round(total);
      if (drift !== 0){
        // adjust away to preserve combined total
        away -= drift;
      }

      badgeLabel.textContent = "Total Line:";
      badgeValue.textContent = " " + line.toFixed(1);

      const totalPoints = renderScore(home, away);
      evaluate(line, totalPoints);
    }

    // Configure slider based on mode
    if (mode === "line"){
      sliderLabel.textContent = "Adjust the total line";
      range.min = min;
      range.max = max;
      range.step = step;
      range.value = start;

      elMin.textContent = min;
      elMax.textContent = max;

      range.addEventListener("input", updateLineMode);
      updateLineMode();
    } else {
      sliderLabel.textContent = "Adjust the final score total";
      range.min = scoreMin;
      range.max = scoreMax;
      range.step = 1;
      range.value = (home0 + away0);

      elMin.textContent = scoreMin;
      elMax.textContent = scoreMax;

      // Badge still shows line; line fixed at start
      range.addEventListener("input", updateScoreMode);
      updateScoreMode();
    }
  }

  function boot(){
    document.querySelectorAll(".gs-ou-slider-wrap").forEach(init);
  }

  if (document.readyState === "loading"){
    document.addEventListener("DOMContentLoaded", boot);
  } else {
    boot();
  }
})();