<?php
/**
 * Plugin Name: GamblingSite Over/Under Slider
 * Description: Beginner-friendly interactive Over/Under explainer sliders (adjust line OR final score) via shortcodes.
 * Version: 1.0.0
 * Author: GamblingSite.com
 * License: GPLv2 or later
 */

if (!defined('ABSPATH')) { exit; }

define('GS_OU_SLIDER_VERSION', '1.0.0');
define('GS_OU_SLIDER_URL', plugin_dir_url(__FILE__));
define('GS_OU_SLIDER_PATH', plugin_dir_path(__FILE__));

/**
 * Enqueue assets when shortcode is present.
 */
function gs_ou_slider_register_assets() {
  wp_register_style(
    'gs-ou-slider',
    GS_OU_SLIDER_URL . 'assets/ou-slider.css',
    array(),
    GS_OU_SLIDER_VERSION
  );

  wp_register_script(
    'gs-ou-slider',
    GS_OU_SLIDER_URL . 'assets/ou-slider.js',
    array(),
    GS_OU_SLIDER_VERSION,
    true
  );
}
add_action('wp_enqueue_scripts', 'gs_ou_slider_register_assets');

/**
 * Shortcode:
 * [ou_slider mode="line" min="210" max="230" start="220.5" step="0.5" home="112" away="109" label="Mock Final Score"]
 *
 * mode="line"  => slider adjusts the total line
 * mode="score" => slider adjusts the combined score while the line stays fixed at "start"
 */
function gs_ou_slider_shortcode($atts) {
  $a = shortcode_atts(array(
    'mode'  => 'line',
    'min'   => '210',
    'max'   => '230',
    'start' => '220.5',
    'step'  => '0.5',
    'home'  => '112',
    'away'  => '109',
    'label' => 'Mock Game',

    // Optional: limits for score slider (mode="score")
    'score_min' => '',
    'score_max' => '',
  ), $atts);

  // Sanitize
  $mode = ($a['mode'] === 'score') ? 'score' : 'line';

  $min = is_numeric($a['min']) ? $a['min'] : '210';
  $max = is_numeric($a['max']) ? $a['max'] : '230';
  $start = is_numeric($a['start']) ? $a['start'] : '220.5';
  $step = is_numeric($a['step']) ? $a['step'] : '0.5';
  $home = is_numeric($a['home']) ? $a['home'] : '112';
  $away = is_numeric($a['away']) ? $a['away'] : '109';

  $score_min = ($a['score_min'] !== '' && is_numeric($a['score_min'])) ? $a['score_min'] : $min;
  $score_max = ($a['score_max'] !== '' && is_numeric($a['score_max'])) ? $a['score_max'] : $max;

  $id = 'gs_ou_slider_' . wp_generate_uuid4();

  // Enqueue assets only when shortcode used
  wp_enqueue_style('gs-ou-slider');
  wp_enqueue_script('gs-ou-slider');

  ob_start(); ?>
  <div class="gs-ou-slider-wrap"
       id="<?php echo esc_attr($id); ?>"
       data-mode="<?php echo esc_attr($mode); ?>"
       data-min="<?php echo esc_attr($min); ?>"
       data-max="<?php echo esc_attr($max); ?>"
       data-start="<?php echo esc_attr($start); ?>"
       data-step="<?php echo esc_attr($step); ?>"
       data-home="<?php echo esc_attr($home); ?>"
       data-away="<?php echo esc_attr($away); ?>"
       data-score-min="<?php echo esc_attr($score_min); ?>"
       data-score-max="<?php echo esc_attr($score_max); ?>"
       data-label="<?php echo esc_attr($a['label']); ?>">
    <div class="gs-ou-card" role="group" aria-label="Interactive over under explainer">
      <div class="gs-ou-head">
        <div>
          <div class="gs-ou-title">Interactive Over/Under Explainer</div>
          <div class="gs-ou-sub">
            <?php if ($mode === 'line'): ?>
              Drag the total line and see Over/Under win or lose instantly.
            <?php else: ?>
              Drag the final score total and see Over/Under win or lose instantly.
            <?php endif; ?>
          </div>
        </div>

        <div class="gs-ou-badge" aria-live="polite">
          <span class="gs-ou-badge-label"></span>
          <span class="gs-ou-badge-value"></span>
        </div>
      </div>

      <div class="gs-ou-score">
        <div class="gs-ou-score-left">
          <div class="gs-ou-score-label"><span class="gs-ou-label"></span> Final</div>
          <div class="gs-ou-score-box">
            <span class="gs-ou-team">Home</span>
            <span class="gs-ou-points gs-ou-home"></span>
            <span class="gs-ou-dash">–</span>
            <span class="gs-ou-points gs-ou-away"></span>
            <span class="gs-ou-team">Away</span>
          </div>
          <div class="gs-ou-score-total">
            Combined points: <strong><span class="gs-ou-total"></span></strong>
          </div>
        </div>

        <div class="gs-ou-results">
          <div class="gs-ou-result gs-ou-over" aria-live="polite">
            <div class="gs-ou-result-top">
              <span class="gs-ou-pill">OVER</span>
              <span class="gs-ou-status gs-ou-over-status"></span>
            </div>
            <div class="gs-ou-explain gs-ou-over-explain"></div>
          </div>

          <div class="gs-ou-result gs-ou-under" aria-live="polite">
            <div class="gs-ou-result-top">
              <span class="gs-ou-pill">UNDER</span>
              <span class="gs-ou-status gs-ou-under-status"></span>
            </div>
            <div class="gs-ou-explain gs-ou-under-explain"></div>
          </div>
        </div>
      </div>

      <div class="gs-ou-slider">
        <label class="gs-ou-slider-label"></label>
        <input class="gs-ou-range" type="range" />
        <div class="gs-ou-scale">
          <span class="gs-ou-min"></span>
          <span class="gs-ou-max"></span>
        </div>
      </div>

      <div class="gs-ou-footnote">
        Tip: Half-points (like 220.5) remove “push” results — it’s always win or loss.
      </div>
    </div>
  </div>
  <?php
  return ob_get_clean();
}
add_shortcode('ou_slider', 'gs_ou_slider_shortcode');
