<?php
/**
 * Plugin Name: AI Parlay Builder
 * Plugin URI: https://your-gambling-site.com
 * Description: AI-powered parlay builder with real-time odds and intelligent betting analysis
 * Version: 1.0.0
 * Author: Your Name
 * License: GPL v2 or later
 * Text Domain: ai-parlay-builder
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('AI_PARLAY_BUILDER_VERSION', '1.0.0');
define('AI_PARLAY_BUILDER_PLUGIN_URL', plugin_dir_url(__FILE__));
define('AI_PARLAY_BUILDER_PLUGIN_PATH', plugin_dir_path(__FILE__));

// Main plugin class
class AIParlay_Builder {
    
    public function __construct() {
        add_action('init', array($this, 'init'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('admin_menu', array($this, 'admin_menu'));
        add_action('wp_ajax_generate_parlay', array($this, 'ajax_generate_parlay'));
        add_action('wp_ajax_nopriv_generate_parlay', array($this, 'ajax_generate_parlay'));
        add_action('wp_ajax_get_sports', array($this, 'ajax_get_sports'));
        add_action('wp_ajax_nopriv_get_sports', array($this, 'ajax_get_sports'));
        add_action('wp_ajax_get_odds', array($this, 'ajax_get_odds'));
        add_action('wp_ajax_nopriv_get_odds', array($this, 'ajax_get_odds'));
        add_action('admin_init', array($this, 'admin_init'));
        
        // Register activation hook
        register_activation_hook(__FILE__, array($this, 'activate'));
        
        // Register shortcode
        add_shortcode('ai_parlay_builder', array($this, 'shortcode'));
    }
    
    public function init() {
        // Plugin initialization
    }
    
    public function activate() {
        // Create database table for parlay history if needed
        $this->create_tables();
    }
    
    private function create_tables() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'ai_parlay_history';
        
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE $table_name (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            parlay_id varchar(255) NOT NULL,
            parlay_data longtext NOT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY parlay_id (parlay_id)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
    
    public function enqueue_scripts() {
        wp_enqueue_script('ai-parlay-builder-js', AI_PARLAY_BUILDER_PLUGIN_URL . 'assets/parlay-builder.js', array('jquery'), AI_PARLAY_BUILDER_VERSION, true);
        wp_enqueue_style('ai-parlay-builder-css', AI_PARLAY_BUILDER_PLUGIN_URL . 'assets/parlay-builder.css', array(), AI_PARLAY_BUILDER_VERSION);
        
        // Localize script for AJAX
        wp_localize_script('ai-parlay-builder-js', 'aiParlay', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('ai_parlay_nonce'),
            'plugin_url' => AI_PARLAY_BUILDER_PLUGIN_URL
        ));
    }
    
    public function admin_menu() {
        add_options_page(
            'AI Parlay Builder Settings',
            'AI Parlay Builder',
            'manage_options',
            'ai-parlay-builder',
            array($this, 'admin_page')
        );
    }
    
    public function admin_init() {
        register_setting('ai_parlay_builder_settings', 'ai_parlay_odds_api_key');
        register_setting('ai_parlay_builder_settings', 'ai_parlay_openai_api_key');
        register_setting('ai_parlay_builder_settings', 'ai_parlay_affiliate_links');
        
        add_settings_section(
            'ai_parlay_api_section',
            'API Configuration',
            array($this, 'api_section_callback'),
            'ai_parlay_builder_settings'
        );
        
        add_settings_section(
            'ai_parlay_affiliate_section',
            'Affiliate Links',
            array($this, 'affiliate_section_callback'),
            'ai_parlay_builder_settings'
        );
        
        add_settings_field(
            'odds_api_key',
            'TheOddsAPI Key',
            array($this, 'odds_api_key_callback'),
            'ai_parlay_builder_settings',
            'ai_parlay_api_section'
        );
        
        add_settings_field(
            'openai_api_key',
            'OpenAI API Key',
            array($this, 'openai_api_key_callback'),
            'ai_parlay_builder_settings',
            'ai_parlay_api_section'
        );
        
        add_settings_field(
            'affiliate_links',
            'Affiliate Links',
            array($this, 'affiliate_links_callback'),
            'ai_parlay_builder_settings',
            'ai_parlay_affiliate_section'
        );
    }
    
    public function api_section_callback() {
        echo '<p>Configure your API keys for odds data and AI analysis.</p>';
    }
    
    public function affiliate_section_callback() {
        echo '<p>Configure your affiliate links for sportsbooks.</p>';
    }
    
    public function odds_api_key_callback() {
        $value = get_option('ai_parlay_odds_api_key', '');
        echo '<input type="text" name="ai_parlay_odds_api_key" value="' . esc_attr($value) . '" class="regular-text" />';
        echo '<p class="description">Get your API key from <a href="https://the-odds-api.com" target="_blank">TheOddsAPI</a></p>';
    }
    
    public function openai_api_key_callback() {
        $value = get_option('ai_parlay_openai_api_key', '');
        echo '<input type="text" name="ai_parlay_openai_api_key" value="' . esc_attr($value) . '" class="regular-text" />';
        echo '<p class="description">Get your API key from <a href="https://openai.com" target="_blank">OpenAI</a></p>';
    }
    
    public function affiliate_links_callback() {
        $value = get_option('ai_parlay_affiliate_links', '');
        echo '<textarea name="ai_parlay_affiliate_links" rows="10" cols="50" class="large-text">' . esc_textarea($value) . '</textarea>';
        echo '<p class="description">Enter affiliate links in JSON format:<br>';
        echo '<code>{"betmgm": "https://your-betmgm-affiliate-link", "fanduel": "https://your-fanduel-affiliate-link"}</code></p>';
    }
    
    public function admin_page() {
        ?>
        <div class="wrap">
            <h1>AI Parlay Builder Settings</h1>
            <form method="post" action="options.php">
                <?php
                settings_fields('ai_parlay_builder_settings');
                do_settings_sections('ai_parlay_builder_settings');
                submit_button();
                ?>
            </form>
            
            <h2>Usage</h2>
            <p>Use the shortcode <code>[ai_parlay_builder]</code> to display the parlay builder on any page or post.</p>
            
            <h2>Test API Connections</h2>
            <button type="button" id="test-odds-api" class="button">Test TheOddsAPI</button>
            <button type="button" id="test-openai-api" class="button">Test OpenAI API</button>
            <div id="api-test-results" style="margin-top: 20px;"></div>
            
            <script>
            jQuery(document).ready(function($) {
                $('#test-odds-api').click(function() {
                    $('#api-test-results').html('Testing TheOddsAPI...');
                    $.post(ajaxurl, {
                        action: 'test_odds_api',
                        nonce: '<?php echo wp_create_nonce('ai_parlay_admin_nonce'); ?>'
                    }, function(response) {
                        $('#api-test-results').html('<strong>TheOddsAPI:</strong> ' + response.data);
                    });
                });
                
                $('#test-openai-api').click(function() {
                    $('#api-test-results').html('Testing OpenAI API...');
                    $.post(ajaxurl, {
                        action: 'test_openai_api',
                        nonce: '<?php echo wp_create_nonce('ai_parlay_admin_nonce'); ?>'
                    }, function(response) {
                        $('#api-test-results').html('<strong>OpenAI API:</strong> ' + response.data);
                    });
                });
            });
            </script>
        </div>
        <?php
    }
    
    public function shortcode($atts) {
        $atts = shortcode_atts(array(
            'style' => 'default',
            'width' => '100%',
            'height' => 'auto'
        ), $atts);
        
        ob_start();
        ?>
        <div id="ai-parlay-builder" class="ai-parlay-container" style="width: <?php echo esc_attr($atts['width']); ?>; height: <?php echo esc_attr($atts['height']); ?>;">
            <div class="parlay-header">
                <h2>🎯 AI Parlay Builder</h2>
                <p>Powered by AI • Real-time Odds • Smart Analysis</p>
            </div>
            
            <div class="parlay-content">
                <div class="parlay-inputs">
                    <div class="input-section">
                        <h3>⚙️ Build Your Parlay</h3>
                        
                        <div class="field-group">
                            <label>Select Sports</label>
                            <div id="sports-selection" class="sports-grid">
                                <!-- Sports will be loaded via AJAX -->
                            </div>
                        </div>
                        
                        <div class="field-group">
                            <label>Date Range</label>
                            <select id="date-range">
                                <option value="today">Today</option>
                                <option value="next_3_days">Next 3 Days</option>
                                <option value="this_week">This Week</option>
                            </select>
                        </div>
                        
                        <div class="field-group">
                            <label>Risk Level</label>
                            <div class="risk-buttons">
                                <button type="button" class="risk-btn" data-risk="conservative">Conservative</button>
                                <button type="button" class="risk-btn active" data-risk="balanced">Balanced</button>
                                <button type="button" class="risk-btn" data-risk="risky">Risky</button>
                            </div>
                        </div>
                        
                        <div class="field-group">
                            <label>Number of Legs: <span id="legs-display">3</span></label>
                            <input type="range" id="num-legs" min="2" max="10" value="3" />
                        </div>
                        
                        <div class="field-group">
                            <label>Bet Types</label>
                            <div class="bet-types">
                                <label><input type="checkbox" value="moneyline" checked> Moneyline</label>
                                <label><input type="checkbox" value="spread" checked> Spread</label>
                                <label><input type="checkbox" value="total"> Total (O/U)</label>
                            </div>
                        </div>
                        
                        <button id="generate-parlay" class="generate-btn">🚀 Generate AI Parlays</button>
                    </div>
                </div>
                
                <div class="parlay-results">
                    <div id="loading-state" class="loading" style="display: none;">
                        <div class="spinner"></div>
                        <p>Generating AI Parlays...</p>
                    </div>
                    
                    <div id="parlay-suggestions" class="suggestions-container">
                        <!-- Parlay suggestions will be loaded here -->
                    </div>
                    
                    <div id="empty-state" class="empty-state">
                        <div class="empty-icon">🎯</div>
                        <h3>Ready to Build Smart Parlays?</h3>
                        <p>Select your preferences and let our AI analyze the best betting opportunities for you.</p>
                    </div>
                </div>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }
    
    // AJAX Handlers
    public function ajax_get_sports() {
        check_ajax_referer('ai_parlay_nonce', 'nonce');
        
        $odds_api_key = get_option('ai_parlay_odds_api_key');
        if (empty($odds_api_key)) {
            wp_send_json_error('TheOddsAPI key not configured');
        }
        
        $response = wp_remote_get("https://api.the-odds-api.com/v4/sports?apiKey=" . $odds_api_key);
        
        if (is_wp_error($response)) {
            wp_send_json_error('Failed to fetch sports data');
        }
        
        $body = wp_remote_retrieve_body($response);
        $sports_data = json_decode($body, true);
        
        // Filter to popular sports
        $popular_sports = array(
            array('key' => 'americanfootball_nfl', 'title' => 'NFL'),
            array('key' => 'basketball_nba', 'title' => 'NBA'),
            array('key' => 'baseball_mlb', 'title' => 'MLB'),
            array('key' => 'icehockey_nhl', 'title' => 'NHL'),
            array('key' => 'soccer_epl', 'title' => 'English Premier League'),
            array('key' => 'americanfootball_ncaaf', 'title' => 'College Football'),
            array('key' => 'basketball_ncaab', 'title' => 'College Basketball')
        );
        
        wp_send_json_success($popular_sports);
    }
    
    public function ajax_get_odds() {
        check_ajax_referer('ai_parlay_nonce', 'nonce');
        
        $sport = sanitize_text_field($_POST['sport']);
        $odds_api_key = get_option('ai_parlay_odds_api_key');
        
        if (empty($odds_api_key)) {
            wp_send_json_error('TheOddsAPI key not configured');
        }
        
        $url = "https://api.the-odds-api.com/v4/sports/{$sport}/odds";
        $args = array(
            'timeout' => 30,
            'headers' => array('Content-Type' => 'application/json'),
        );
        
        $query_params = array(
            'apiKey' => $odds_api_key,
            'regions' => 'us',
            'markets' => 'h2h,spreads,totals',
            'oddsFormat' => 'american',
            'dateFormat' => 'iso'
        );
        
        $response = wp_remote_get($url . '?' . http_build_query($query_params), $args);
        
        if (is_wp_error($response)) {
            wp_send_json_error('Failed to fetch odds data');
        }
        
        $body = wp_remote_retrieve_body($response);
        $odds_data = json_decode($body, true);
        
        wp_send_json_success($odds_data);
    }
    
    public function ajax_generate_parlay() {
        check_ajax_referer('ai_parlay_nonce', 'nonce');
        
        $sports = sanitize_text_field($_POST['sports']);
        $risk_level = sanitize_text_field($_POST['risk_level']);
        $num_legs = intval($_POST['num_legs']);
        $bet_types = sanitize_text_field($_POST['bet_types']);
        $date_range = sanitize_text_field($_POST['date_range']);
        
        $sports_array = explode(',', $sports);
        $bet_types_array = explode(',', $bet_types);
        
        // Get odds data for selected sports
        $all_odds = array();
        foreach ($sports_array as $sport) {
            $odds_data = $this->fetch_odds_data($sport);
            if ($odds_data) {
                $all_odds = array_merge($all_odds, $odds_data);
            }
        }
        
        if (empty($all_odds)) {
            // Use fallback sample data
            $all_odds = $this->get_sample_data($sports_array[0]);
        }
        
        // Generate AI parlay
        $parlay_data = $this->generate_ai_parlay($all_odds, $sports_array, $risk_level, $num_legs, $bet_types_array);
        
        if ($parlay_data) {
            // Save to database (optional)
            $this->save_parlay_history($parlay_data);
            
            wp_send_json_success($parlay_data);
        } else {
            wp_send_json_error('Failed to generate parlay');
        }
    }
    
    private function fetch_odds_data($sport) {
        $odds_api_key = get_option('ai_parlay_odds_api_key');
        if (empty($odds_api_key)) {
            return false;
        }
        
        $url = "https://api.the-odds-api.com/v4/sports/{$sport}/odds";
        $query_params = array(
            'apiKey' => $odds_api_key,
            'regions' => 'us',
            'markets' => 'h2h,spreads,totals',
            'oddsFormat' => 'american',
            'dateFormat' => 'iso'
        );
        
        $response = wp_remote_get($url . '?' . http_build_query($query_params));
        
        if (is_wp_error($response)) {
            return false;
        }
        
        $body = wp_remote_retrieve_body($response);
        return json_decode($body, true);
    }
    
    private function generate_ai_parlay($odds_data, $sports, $risk_level, $num_legs, $bet_types) {
        $openai_api_key = get_option('ai_parlay_openai_api_key');
        if (empty($openai_api_key)) {
            return $this->generate_fallback_parlay($odds_data, $num_legs);
        }
        
        // Prepare data for AI
        $prepared_data = $this->prepare_odds_for_ai($odds_data, $bet_types, 10);
        
        $prompt = $this->build_ai_prompt($prepared_data, $sports, $risk_level, $num_legs, $bet_types);
        
        // Call OpenAI API
        $ai_response = $this->call_openai_api($openai_api_key, $prompt);
        
        if ($ai_response) {
            return $this->process_ai_response($ai_response);
        } else {
            return $this->generate_fallback_parlay($odds_data, $num_legs);
        }
    }
    
    private function prepare_odds_for_ai($odds_data, $bet_types, $limit = 10) {
        $prepared = array();
        $count = 0;
        
        foreach ($odds_data as $event) {
            if ($count >= $limit || empty($event['bookmakers'])) {
                continue;
            }
            
            $bookmaker = $event['bookmakers'][0];
            $event_summary = array(
                'event_id' => $event['id'],
                'sport' => $event['sport_key'],
                'home_team' => $event['home_team'],
                'away_team' => $event['away_team'],
                'commence_time' => $event['commence_time'],
                'markets' => array()
            );
            
            foreach ($bookmaker['markets'] as $market) {
                $include_market = false;
                
                if (in_array('moneyline', $bet_types) && $market['key'] == 'h2h') {
                    $include_market = true;
                } elseif (in_array('spread', $bet_types) && $market['key'] == 'spreads') {
                    $include_market = true;
                } elseif (in_array('total', $bet_types) && $market['key'] == 'totals') {
                    $include_market = true;
                }
                
                if ($include_market) {
                    $simplified_outcomes = array();
                    foreach ($market['outcomes'] as $outcome) {
                        $simplified_outcome = array(
                            'name' => $outcome['name'],
                            'price' => $outcome['price']
                        );
                        if (isset($outcome['point'])) {
                            $simplified_outcome['point'] = $outcome['point'];
                        }
                        $simplified_outcomes[] = $simplified_outcome;
                    }
                    $event_summary['markets'][$market['key']] = $simplified_outcomes;
                }
            }
            
            $prepared[] = $event_summary;
            $count++;
        }
        
        return $prepared;
    }
    
    private function build_ai_prompt($prepared_data, $sports, $risk_level, $num_legs, $bet_types) {
        return "Create 1 optimized parlay suggestion for " . implode(', ', $sports) . " with {$risk_level} risk.

Request: {$num_legs} legs, bet types: " . implode(', ', $bet_types) . "

Available Games:
" . json_encode($prepared_data, JSON_PRETTY_PRINT) . "

Return JSON object with format:
{
    \"selections\": [{
        \"event_id\": \"string\",
        \"sport\": \"string\", 
        \"home_team\": \"string\",
        \"away_team\": \"string\",
        \"selected_team\": \"string\",
        \"bet_type\": \"string\",
        \"line\": number or null,
        \"odds\": number,
        \"commence_time\": \"string\"
    }],
    \"confidence_score\": number (1-10),
    \"ai_rationale\": \"brief strategy explanation\",
    \"risk_assessment\": \"brief risk analysis\"
}

Focus on value bets and match {$risk_level} risk level.";
    }
    
    private function call_openai_api($api_key, $prompt) {
        $url = 'https://api.openai.com/v1/chat/completions';
        
        $data = array(
            'model' => 'gpt-4',
            'messages' => array(
                array(
                    'role' => 'system',
                    'content' => 'You are an expert sports betting analyst. Create optimized parlay combinations based on risk levels and betting value. Always respond with valid JSON only.'
                ),
                array(
                    'role' => 'user',
                    'content' => $prompt
                )
            ),
            'max_tokens' => 2048,
            'temperature' => 0.7
        );
        
        $args = array(
            'body' => json_encode($data),
            'headers' => array(
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $api_key
            ),
            'timeout' => 30
        );
        
        $response = wp_remote_post($url, $args);
        
        if (is_wp_error($response)) {
            return false;
        }
        
        $body = wp_remote_retrieve_body($response);
        $response_data = json_decode($body, true);
        
        if (isset($response_data['choices'][0]['message']['content'])) {
            return $response_data['choices'][0]['message']['content'];
        }
        
        return false;
    }
    
    private function process_ai_response($ai_response) {
        $parlay_data = json_decode($ai_response, true);
        
        if (!$parlay_data) {
            return false;
        }
        
        // Calculate total odds and probability
        $total_odds = $this->calculate_parlay_odds($parlay_data['selections']);
        
        return array(
            'parlay_id' => uniqid('parlay_'),
            'selections' => $parlay_data['selections'],
            'total_odds' => $total_odds['american_odds'],
            'implied_probability' => $total_odds['implied_probability'],
            'confidence_score' => $parlay_data['confidence_score'],
            'ai_rationale' => $parlay_data['ai_rationale'],
            'risk_assessment' => $parlay_data['risk_assessment']
        );
    }
    
    private function calculate_parlay_odds($selections) {
        $total_decimal_odds = 1.0;
        
        foreach ($selections as $selection) {
            $odds = $selection['odds'];
            if ($odds > 0) {
                $decimal_odds = ($odds / 100) + 1;
            } else {
                $decimal_odds = (100 / abs($odds)) + 1;
            }
            $total_decimal_odds *= $decimal_odds;
        }
        
        if ($total_decimal_odds >= 2) {
            $american_odds = ($total_decimal_odds - 1) * 100;
        } else {
            $american_odds = -100 / ($total_decimal_odds - 1);
        }
        
        $implied_probability = (1 / $total_decimal_odds) * 100;
        
        return array(
            'american_odds' => round($american_odds),
            'implied_probability' => round($implied_probability, 2)
        );
    }
    
    private function generate_fallback_parlay($odds_data, $num_legs) {
        $selections = array();
        $count = 0;
        
        foreach ($odds_data as $event) {
            if ($count >= $num_legs || empty($event['bookmakers'])) {
                continue;
            }
            
            $bookmaker = $event['bookmakers'][0];
            foreach ($bookmaker['markets'] as $market) {
                if ($market['key'] == 'h2h' && !empty($market['outcomes'])) {
                    $outcome = $market['outcomes'][0];
                    $selections[] = array(
                        'event_id' => $event['id'],
                        'sport' => $event['sport_key'],
                        'home_team' => $event['home_team'],
                        'away_team' => $event['away_team'],
                        'selected_team' => $outcome['name'],
                        'bet_type' => 'moneyline',
                        'odds' => $outcome['price'],
                        'commence_time' => $event['commence_time']
                    );
                    $count++;
                    break;
                }
            }
        }
        
        if (empty($selections)) {
            return false;
        }
        
        $total_odds = $this->calculate_parlay_odds($selections);
        
        return array(
            'parlay_id' => uniqid('parlay_'),
            'selections' => $selections,
            'total_odds' => $total_odds['american_odds'],
            'implied_probability' => $total_odds['implied_probability'],
            'confidence_score' => 5.0,
            'ai_rationale' => 'Basic parlay suggestion based on available odds data.',
            'risk_assessment' => 'Medium risk based on standard betting metrics.'
        );
    }
    
    private function get_sample_data($sport) {
        // Return sample data based on sport
        $sample_data = array();
        
        if ($sport == 'baseball_mlb') {
            $sample_data = array(
                array(
                    'id' => 'sample_mlb_1',
                    'sport_key' => 'baseball_mlb',
                    'home_team' => 'New York Yankees',
                    'away_team' => 'Boston Red Sox',
                    'commence_time' => date('c'),
                    'bookmakers' => array(
                        array(
                            'markets' => array(
                                array(
                                    'key' => 'h2h',
                                    'outcomes' => array(
                                        array('name' => 'New York Yankees', 'price' => -150),
                                        array('name' => 'Boston Red Sox', 'price' => 130)
                                    )
                                )
                            )
                        )
                    )
                ),
                array(
                    'id' => 'sample_mlb_2',
                    'sport_key' => 'baseball_mlb',
                    'home_team' => 'Los Angeles Dodgers',
                    'away_team' => 'San Francisco Giants',
                    'commence_time' => date('c'),
                    'bookmakers' => array(
                        array(
                            'markets' => array(
                                array(
                                    'key' => 'h2h',
                                    'outcomes' => array(
                                        array('name' => 'Los Angeles Dodgers', 'price' => -180),
                                        array('name' => 'San Francisco Giants', 'price' => 155)
                                    )
                                )
                            )
                        )
                    )
                )
            );
        }
        
        return $sample_data;
    }
    
    private function save_parlay_history($parlay_data) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'ai_parlay_history';
        
        $wpdb->insert(
            $table_name,
            array(
                'parlay_id' => $parlay_data['parlay_id'],
                'parlay_data' => json_encode($parlay_data),
                'created_at' => current_time('mysql')
            ),
            array('%s', '%s', '%s')
        );
    }
}

// Initialize the plugin
new AIParlay_Builder();